/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2009(c).
 * 
 * file: TerralibQueryData.java
 * created: 14/01/2009
 */
package org.geotools.data.terralib.query;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import org.apache.log4j.Logger;
import org.geotools.data.AttributeReader;
import org.geotools.data.terralib.FeatureTypeInfo;
import org.geotools.data.terralib.TerralibService;
import org.geotools.data.terralib.exception.NullArgumentException;
import org.geotools.data.terralib.exception.TerralibProviderRuntimeException;
import org.geotools.data.terralib.swig.DatabasePortal;
import org.geotools.data.terralib.swig.PersistenceTransferenceObject;
import org.geotools.data.terralib.swig.TerralibServiceNative;
import org.geotools.data.terralib.util.TypeAttributeMap;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.feature.type.GeometryDescriptor;

import com.vividsolutions.jts.geom.MultiLineString;
import com.vividsolutions.jts.geom.MultiPoint;
import com.vividsolutions.jts.geom.MultiPolygon;

/**
 * This Class is the Terralib data access {@link AttributeReader} implementation.
 * @author fmoura, alexmc
 * @since TDK 3.0.0
 */
public class TerralibQueryData implements QueryData
{
	private enum WriteMode {UPDATE, INSERT};
	private static final Logger _logger = Logger.getLogger(TerralibQueryData.class);
	private static final int NO_GEOMETRY_FIELD_FOUND = -1;
	private WriteMode _currentWriteMode = WriteMode.UPDATE;
	private TerralibService _terralibService;
    private DatabasePortal _databasePortal;
    private List<AttributeDescriptor> _attributeDescriptors;
    private FeatureTypeInfo _featureTypeInfo;
    private int _numFields = 0;
    
    private Object[] _currentFieldsValues = null;
    private Object[] _writeBuffer = null;
	private int _geometryAttributeIndex = NO_GEOMETRY_FIELD_FOUND;
	private boolean _closed = false;
	private String _lockUUID;

    /**
     * Constructor
     * @param terralibService The terralib service instance to be used.
     * @param portal The DatabasePortal that holds the query result
     * @param featureTypeInfo The FeatureTypeInfo structure data has auxiliary information that TerralibQueryData needs to correctly retrieve the query data.
     * @throws IOException if a it could not fetch the first line
     * @since TDK 3.0.0
     */
	protected TerralibQueryData(TerralibService terralibService,DatabasePortal portal, FeatureTypeInfo featureTypeInfo) throws IOException
	{
		this(terralibService,portal,featureTypeInfo,null);
	}
	
	
	/**
     * Constructor
     * @param terralibService The terralib service instance to be used.
     * @param portal The DatabasePortal that holds the query result
     * @param featureTypeInfo The FeatureTypeInfo structure data has auxiliary information that TerralibQueryData needs to correctly retrieve the query data.
     * @param lockUUID
     * @throws IOException if a it could not fetch the first line
     * @since TDK 3.0.0
     */
    protected TerralibQueryData(TerralibService terralibService,DatabasePortal portal, FeatureTypeInfo featureTypeInfo, String lockUUID) throws IOException
    {
    	if (terralibService == null)
    	{
    		throw new NullArgumentException("terralibService");
    	}
    	
        if(portal == null)
        {
            throw new NullArgumentException("portal");
        }
        
        if(featureTypeInfo == null)
        {
            throw new NullArgumentException("featureTypeInfo");
        }
        _lockUUID = lockUUID;
        _terralibService = terralibService;
        _databasePortal = portal;
        _attributeDescriptors = featureTypeInfo.getTypeAttributeList();
        _numFields = featureTypeInfo.getTypeAttributeList().size();
        for(int i = 0 ; i < _numFields ; i++)
        {
        	if(_attributeDescriptors.get(i) instanceof GeometryDescriptor)
        	{
        		_geometryAttributeIndex = i;
        		break;
        	}
        }
        _featureTypeInfo = featureTypeInfo;
    }

    /**
     * It also advances to the next record due to terralib database portal implementation
     * @throws IOException
     * @since TDK 3.0.0
     */
    private void readRowValues() throws IOException
    {
    	
		_currentFieldsValues = new Object[_numFields+1];
	    _currentFieldsValues[_numFields] = _databasePortal.getString(TerralibServiceNative.getFEATURE_ID_COMPLETE_ATTRIBUTE_NAME());
	   
	    for(int i = 0; i < _numFields ; i++)
	    {
	    	if(i == _geometryAttributeIndex )
	    	{
	    		continue;
	    	}
	        Object value = getAttributeValue(i);
	        _currentFieldsValues[i] = value;
	    } 
	    if(_geometryAttributeIndex != NO_GEOMETRY_FIELD_FOUND)
	    {
	    	_currentFieldsValues[_geometryAttributeIndex] = getGeometryValue();
	    }
 
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#finalize()
     */
    @Override
    protected void finalize() throws Throwable
    {
		synchronized (_databasePortal) {
			_databasePortal.setToFreeNativeTarget(true);
			_databasePortal.delete();
			if(!_closed)
			{
				_terralibService.releaseLock(_lockUUID);
			}
		}
		super.finalize();
    }

    /* (non-Javadoc)
     * @see org.geotools.data.AttributeReader#close()
     */
    @Override
    public void close() throws IOException
    {
    	synchronized(_databasePortal)
    	{
    		if(_closed) return;
            
        	_databasePortal.freeResult();
            _terralibService.releaseLock(_lockUUID);
            _closed = true;
    	}
    	
    }

    /* (non-Javadoc)
     * @see org.geotools.data.AttributeReader#getAttributeCount()
     */
    @Override
    public int getAttributeCount()
    {
        return _numFields;
    }

    /* (non-Javadoc)
     * @see org.geotools.data.AttributeReader#getAttributeType(int)
     */
    @Override
    public AttributeDescriptor getAttributeType(int index)
            throws ArrayIndexOutOfBoundsException
    {
        if((index >= _numFields) || (index < 0))
        {
            throw new ArrayIndexOutOfBoundsException("Field index should be between 0 and "+_attributeDescriptors.size()+". Index = "+index);
        }

        return _attributeDescriptors.get(index);
    }

    /* (non-Javadoc)
     * @see org.geotools.data.AttributeReader#hasNext()
     */
    @Override
    public boolean hasNext() throws IOException
    {
    	
    	return _databasePortal.hasNext();

    }

    /* (non-Javadoc)
     * @see org.geotools.data.AttributeReader#next()
     */
    @Override
    public void next() throws IOException
    {
    	_writeBuffer = null;
    	if (hasNext())
    	{
    		readRowValues();
    	}
    	else
    	{
    		_currentFieldsValues = new Object[_numFields+1];
    		_currentWriteMode = WriteMode.INSERT;
    	}
    }

    /* (non-Javadoc)
     * @see org.geotools.data.AttributeReader#read(int)
     */
    @Override
    public Object read(int index) throws IOException,
            ArrayIndexOutOfBoundsException
    {
        if((index >= _numFields) || (index < 0))
        {
            throw new ArrayIndexOutOfBoundsException("Field index should be between 0 and "+_attributeDescriptors.size()+". Index = "+index);
        }
        
        return _currentFieldsValues[index];
    }
    
    /**
     * Retrieves the geometry attribute for the current row. It also advances to the next row due to terralib database portal implementation
     * @return The geometry class ( Point, LineSgment or Polygon ) that represents the default geometry for the current row.
     * @throws IOException if it encounters a problem while reading the geometry value.
     * @since TDK 3.0.0
     */
    private Object getGeometryValue() throws IOException
    {

        TypeAttributeMap typeMap = TypeAttributeMap.fromGeometryType(_featureTypeInfo.getDefaultGeometryType());
        switch (typeMap) 
        {
			case TA_LINES:
				return _databasePortal.fetchLine();	
			case TA_MULTIPOINT:
				return _databasePortal.fetchPoint();	
			case TA_MULTIPOLYGON:
				return _databasePortal.fetchPolygon();	
			default:
				throw new IOException("Binding type not recognized : " + _featureTypeInfo.getDefaultGeometryType().getBinding().getCanonicalName());
		}

//        Class<?> bindingClass  = _featureTypeInfo.getDefaultGeometryType().getBinding();
        
//        TeAttrDataType teType = TypeAttributeMap.translateAttribute(_featureTypeInfo.getDefaultGeometryType());
//        switch (teType) 
//        {
//			case TeLINE2DTYPE:
//				return _databasePortal.fetchLine();	
//			case TePOINTTYPE:
//				return _databasePortal.fetchPoint();	
//			case TePOLYGONTYPE:
//				return _databasePortal.fetchPolygon();	
//			default:
//				throw new IOException("Binding type not recognized : " + bindingClass.getCanonicalName());
//		}
        
//        if(MultiPoint.class.isAssignableFrom(bindingClass))
//        {
//            return _databasePortal.fetchPoint();
//        }
//        else if (MultiLineString.class.isAssignableFrom(bindingClass))
//        {
//            return _databasePortal.fetchLine();
//        }
//        else if (MultiPolygon.class.isAssignableFrom(bindingClass))
//        {
//            return _databasePortal.fetchPolygon();
//        }
//        else
//        {
//            throw new IOException("Binding type not recognized : " + bindingClass.getCanonicalName());
//        }
    }

    
    /**
     * Retrieves the attributes values for the current row
     * @param index 
     * @return
     * @throws IOException
     * @since
     */
    private Object getAttributeValue(int index) throws IOException 
    {
        AttributeDescriptor attribute = _attributeDescriptors.get(index);
//        Class<?> bindingClass = attribute.getType().getBinding();

        final String attributeName = attribute.getLocalName();
        final String attributeFullName = _featureTypeInfo.getAttributeTableName(attributeName)+"."+attributeName;
        
        TypeAttributeMap typeMap = TypeAttributeMap.fromAttributeType(attribute.getType());
        switch (typeMap) 
        {
			case TA_INTEGER:
				return Integer.valueOf(_databasePortal.getInt(attributeFullName));	
			case TA_STRING:
				return _databasePortal.getString(attributeFullName);
			case TA_REAL:
				return _databasePortal.getDouble(attributeFullName);
			case TA_DATE:
			{
			    String data = _databasePortal.getDate(attributeFullName);
                Date date;
			    try
                {
                    date = buildDateFormat().parse(data);
                }
                catch (ParseException e)
                {
                    throw new IOException("Error parsing date field (" + data + ") to Java Date.",e);
                }
			    return date;
			}  
			default:
				throw new TerralibProviderRuntimeException("Error reading attribute value. Unsupported data type: " + typeMap);
//			default:
//				return getGeometryValue();
		}
    }

    /*
     * (non-Javadoc)
     * @see org.geotools.data.AttributeWriter#write(int, java.lang.Object)
     */
    @Override
    public void write(int position, Object attribute) throws IOException
    {
    	if (position < 0 || position >= _numFields)
    	{
    		throw new IllegalArgumentException(
    				"The position should be in the current row's range [0," + _numFields + ").");
    	}

    	if (_writeBuffer == null)
    	{
    		assert _currentFieldsValues != null && _currentFieldsValues.length == _numFields;
    	
    		// create and initialize the new write buffer.
    		_writeBuffer = new Object[_numFields+1];
    	}
    	
    	_writeBuffer[position] = attribute;
    }
    
    /* (non-Javadoc)
     * @see org.tecgraf.tdk.data.terralib.query.QueryData#flush()
     */
    @Override
    public void flush(boolean writeOnlyGeometries ) throws IOException
    {
    	
    		// first, we check if there's data to persist.
        	if (_writeBuffer == null)
        	{
        		return;
        	}

        	// next, we translate the attributes to a persistence transference object
        	// while searching for an object id attribute (this is what differentiates
        	// the insert from the update calls)

        	Object attributeToWrite = null;
        	int attributeIndex = 0;

        	PersistenceTransferenceObject transferenceObject = new PersistenceTransferenceObject();

        	for ( AttributeDescriptor attrDesc : _attributeDescriptors)
        	{
        		attributeToWrite = _writeBuffer[attributeIndex];
        		
        		if (attributeToWrite != null)
        		{
    	    		setTransferenceValue(attrDesc, attributeToWrite, transferenceObject);
        		}
        		
        		++attributeIndex;
        	}

        	final String typeName = _featureTypeInfo.getTypeName(); 

        	try
        	{
    	    	if (_currentWriteMode == WriteMode.UPDATE)
    	    	{
    	    		addCurrentFeatureId(transferenceObject);
    	    		_terralibService.update(typeName, transferenceObject, writeOnlyGeometries);
    	    	}
    	    	else
    	    	{
    	    		if (getObjectId() != null)
    	    			addCurrentFeatureId(transferenceObject);	    			
    	    		String newFeatureId = _terralibService.insert(typeName, transferenceObject,writeOnlyGeometries);
    	    		updateWriteBufferFeatureId(newFeatureId);
    	    	}
        	}
        	catch (Throwable t)
        	{
        		_logger.error("Attribute " + _currentWriteMode + " failed.", t);
        		throw new IOException("Attribute " + _currentWriteMode + " failed.", t);
        	}
        	
        	// update the current read buffer's data with the attribute values written/updated.
        	updateReadBufferFromWriteBuffer();

        	// reset the write buffer.
        	_writeBuffer = null;

    }

    /**
     * Adds the feature id (object_id) value to the transference object. 
     * @param transferenceObject To add the feature id
     */
	private void addCurrentFeatureId(PersistenceTransferenceObject transferenceObject) 
	{
//		String featureIdAttributeName = TerralibServiceNative.getFEATURE_ID_COLUMN_NAME();
//		int featureIdAttributeIndex = findAttributeIndexByName(featureIdAttributeName);
//		
		String featureIdCompleteAttributeName = 
			TerralibServiceNative.getFEATURE_ID_COMPLETE_ATTRIBUTE_NAME();
//		transferenceObject.setStringAttribute(featureIdCompleteAttributeName, 
//				(String) _currentFieldsValues[featureIdAttributeIndex]);
		transferenceObject.setStringAttribute(featureIdCompleteAttributeName, 
				getObjectId());

	}

    //TODO: Substituir por um mapa.
	private int findAttributeIndexByName(String attributeName)
	{
		for (int attributeDescriptorIndex = 0; attributeDescriptorIndex < _attributeDescriptors.size(); 
		     ++attributeDescriptorIndex)
		{
			if (_attributeDescriptors.get(attributeDescriptorIndex).getLocalName().equals(attributeName)) 
			{
				return attributeDescriptorIndex;
			}
		}

		return -1;
	}

	private void updateWriteBufferFeatureId(String newFeatureId)
	{
//		final String featureIdAttrName = 
//			TerralibServiceNative.getFEATURE_ID_COLUMN_NAME();
//		
//		int index = 0;
//		for (AttributeDescriptor attributeDescriptor : _attributeDescriptors)
//		{
//			if (featureIdAttrName.equals(attributeDescriptor.getLocalName()))
//			{
//				_writeBuffer[index] = newFeatureId;
//			}
//			++index;
//		}
		setObjectId(newFeatureId);
	}

	private void updateReadBufferFromWriteBuffer() {
		Object updatedAttribute = null;
    	for (int attributeIndex = 0; attributeIndex < _writeBuffer.length; ++attributeIndex)
    	{
    		updatedAttribute = _writeBuffer[attributeIndex];
    		if (updatedAttribute != null)
    		{
    			_currentFieldsValues[attributeIndex] = updatedAttribute; 
    		}
    	}
	}

	private SimpleDateFormat buildDateFormat()
	{
	    return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
	}
	
	private void setTransferenceValue(AttributeDescriptor attributeDescription,
			Object attributeValue, PersistenceTransferenceObject transferenceObject)
	{
		final Class<?> bindingClass = attributeDescription.getType().getBinding();
		final String attributeName = attributeDescription.getLocalName();
		final String completeAttributeName = _featureTypeInfo.getAttributeTableName(attributeName) + "." + attributeName; 
		
		TypeAttributeMap typeMap = TypeAttributeMap.fromAttributeType(attributeDescription.getType());

		switch (typeMap)
		{
			case TA_INTEGER:
				Integer integerAttribute;
				if (attributeValue instanceof Long)
					integerAttribute = ((Long)attributeValue).intValue();
				else
					integerAttribute = (Integer) attributeValue;
				transferenceObject.setIntegerAttribute(completeAttributeName, integerAttribute);
				break;
			case TA_REAL:
				Double doubleAttribute = (Double) attributeValue;
				transferenceObject.setDoubleAttribute(completeAttributeName, doubleAttribute);
				break;
			case TA_STRING:
				String stringAttribute = (String) attributeValue;
				transferenceObject.setStringAttribute(completeAttributeName, stringAttribute);
				break;
			case TA_DATE:
			    Date dateAttribute = (Date) attributeValue;
			    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
			    System.out.println(format.format(dateAttribute));
			    transferenceObject.setDateAttribute(completeAttributeName, format.format(dateAttribute));
			    break;
			case TA_MULTIPOINT:
				MultiPoint pointGeometries = (MultiPoint) attributeValue;
				transferenceObject.setPointGeometries(pointGeometries);
				break;
			case TA_LINES:
				MultiLineString lineGeometries = (MultiLineString) attributeValue;
				transferenceObject.setLineGeometries(lineGeometries);
				break;
			case TA_MULTIPOLYGON:
				MultiPolygon polygonGeometries = (MultiPolygon) attributeValue;
				transferenceObject.setPolygonGeometries(polygonGeometries);
				break;
			default:
				String message = "Unrecognized attribute class: \"" + bindingClass.getName() + "\".";
				_logger.error("Failed to set the transference attribute. Reason: " + message);
				throw new IllegalArgumentException(message);				
		}
		
//		try
//		{
//			if (Integer.class.isAssignableFrom(bindingClass))
//			{
//				Integer integerAttribute = (Integer) attributeValue;
//				transferenceObject.setIntegerAttribute(completeAttributeName, integerAttribute);
//			}
//			else if (String.class.isAssignableFrom(bindingClass))
//			{
//				String stringAttribute = (String) attributeValue;
//				transferenceObject.setStringAttribute(completeAttributeName, stringAttribute);
//			}
//			else if (Double.class.isAssignableFrom(bindingClass))
//			{
//				Double doubleAttribute = (Double) attributeValue;
//				transferenceObject.setDoubleAttribute(completeAttributeName, doubleAttribute);
//			}
//			else if (MultiPoint.class.isAssignableFrom(bindingClass))
//			{
//				MultiPoint pointGeometries = (MultiPoint) attributeValue;
//				transferenceObject.setPointGeometries(pointGeometries);
//			}
//			else if (MultiLineString.class.isAssignableFrom(bindingClass))
//			{
//				MultiLineString lineGeometries = (MultiLineString) attributeValue;
//				transferenceObject.setLineGeometries(lineGeometries);
//			}
//			else if (MultiPolygon.class.isAssignableFrom(bindingClass))
//			{
//				MultiPolygon polygonGeometries = (MultiPolygon) attributeValue;
//				transferenceObject.setPolygonGeometries(polygonGeometries);
//			}
//			else
//			{
//				String message = "Unrecognized attribute class: \"" + bindingClass.getName() + "\".";
//				_logger.error("Failed to set the transference attribute. Reason: " + message);
//				throw new IllegalArgumentException(message);
//			}
//		}
//		catch (ClassCastException e)
//		{
//			String message = "The attribute "+attributeName+"'s value class is incompatible with the given value type.";
//			_logger.error("Failed to set the transference attribute. Reason: " + message);
//			throw new IllegalArgumentException(message,e);
//		}
	}

    /* (non-Javadoc)
     * @see org.tecgraf.tdk.data.terralib.query.QueryData#isClosed()
     */
    @Override
    public boolean isClosed()
    {
        return _databasePortal.isClosed();
    }

    /*
     * (non-Javadoc)
     * @see org.geotools.data.terralib.query.QueryData#getObjectId()
     */
	@Override
	public String getObjectId() 
	{
		return (String)_currentFieldsValues[_numFields];	//+1?
	}

	@Override
	public void setObjectId(String objectId) 
	{
		_currentFieldsValues[_numFields] = objectId;
		
	}
}
