/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2008(c).
 * 
 * file: TerralibDataStoreFactory.java
 * created: 05/03/2009
 */
package org.geotools.data.terralib;

import static org.geotools.data.terralib.connection.DBConnectionParamUtils.DBDRIVER;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.DBNAME;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.DBPATH;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.DBTYPE;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.HOST;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.PASSWD;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.PORT;
import static org.geotools.data.terralib.connection.DBConnectionParamUtils.USER;

import java.io.IOException;
import java.io.Serializable;
import java.util.Map;

import org.apache.log4j.Logger;
import org.geotools.data.AbstractDataStoreFactory;
import org.geotools.data.DataStore;
import org.geotools.data.terralib.NativeLibraryLoader;
import org.geotools.data.terralib.connection.DBConnectionFactory;
import org.geotools.data.terralib.connection.DBConnectionParamUtils;


/**
 * Creates a TerralibDataStore based on the correct params.
 * <p>
 * This factory should be registered in the META-INF/ folder, under services/
 * in the DataStoreFactorySpi file.
 * </p>
 * 
 * @author fabiomano
 * @since TDK 3.0.0
 */
public class TerralibDataStoreFactory extends AbstractDataStoreFactory
{

    private static final Logger _logger = Logger.getLogger(TerralibDataStoreFactory.class);

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.DataStoreFactorySpi#createDataStore(Map<String, Serializable>)
     */
    @Override
    public DataStore createDataStore(Map<String, Serializable> params) throws IOException
    {
        if (params == null)
            throw new IllegalArgumentException("The TerraLib connection parameters map can't be null.");
            
        if (!canProcess(params))
            throw new IOException("The parameters map isn't correct!!");
        
        return new TerralibDataStore(new DBConnectionFactory(),params);
        
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.AbstractDataStoreFactory#canProcess(Map<String, Serializable>)
     */    
    @SuppressWarnings("unchecked")
    @Override
    public boolean canProcess(Map params)
    {
        if (params == null)
            throw new IllegalArgumentException("The TerraLib connection parameters map can't be null.");

        if ((!super.canProcess(params)))
            return false;
        
        if(!loadDlls())
        {
            return false;
        }
     
        try
        {
            DBConnectionParamUtils.checkParams(params);
            return true;
        }
        catch (IllegalArgumentException e)
        {
            return false;
        }
    }
    

    /**
     * @return
     * @since
     */
    private boolean loadDlls()
    {
        boolean result = NativeLibraryLoader.loadLibraries();
        if(!result)
        {
            _logger .error("Error trying to load terralib provider dlls. Please check your java.library.path property.");
        }
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.DataStoreFactorySpi#createNewDataStore(Map<String, Serializable>)
     */    
    @Override
    public DataStore createNewDataStore(Map<String, Serializable> params) throws IOException
    {
        throw new UnsupportedOperationException();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.DataStoreFactorySpi#getDescription()
     */     
    @Override
    public String getDescription()
    {
        return "TerraLib database";
    }
    
    public String getDisplayName() 
    {
        return "TerraLib";
    }    

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.DataStoreFactorySpi#getParametersInfo()
     */      
    @Override
    public Param[] getParametersInfo() 
    {
        return new Param[] { DBTYPE, DBDRIVER ,DBPATH, DBNAME,HOST,PORT,USER,PASSWD };
    }
    
    public boolean isAvailable() 
    {
        return loadDlls();
    }    
    

}
