/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2009(c).
 * 
 * file: TerralibAttributePersister.java
 * created: 03/04/2009
 */
package org.geotools.data.terralib.feature.attribute;

import java.io.IOException;
import java.util.Map;

import org.geotools.data.terralib.exception.TypeNotFoundException;
import org.opengis.feature.Feature;
import org.opengis.feature.type.FeatureType;

/**
 * This class is responsible for the attribute persistence strategy for Terralib features 
 * @author fmoura
 * @since TDK 3.0.0
 */
public interface TerralibAttributePersistenceHandler
{
    /**
     * Asks if this class is responsible for the non geometry attribute persistence for the given Feature Type. 
     * @param featureTypeName The feature type name
     * @param typeOperation The operation that will be executed
     * @return true if this feature type's attributes persistence is handled by this class.
     * @throws TypeNotFoundException if no type is found with the given feature type name.
     * @since TDK 3.0.0
     */
    boolean canProcess(String featureTypeName, TerralibAttributePersistenceHandlerTypeOperation typeOperation);
    
    /**
     * Writes only the given feature non geometry attributes to the database.
     * @param feature The feature whose attributes will be written to the database.
     * @throws IOException if an error occurs during this method execution.
     * @since TDK 3.0.0
     */
    void insertFeatureAttributes(Feature feature) throws IOException;
    
    
    /**
     * Updates only the given feature non geometry attributes at the database.
     * @param feature The feature whose attributes will be updated at the database.
     * @throws IOException if an error occurs during this method execution.
     * @since TDK 3.0.0
     */
    void updateFeatureAttributes(Feature feature) throws IOException;
    
    /**
     * Removes only the given feature non geometry attributes from the database.
     * @param featureId The featureId that identifies the feature whose attributes will be removed from the database.
     * @throws IOException if an error occurs during this method execution.
     * @since TDK 3.0.0
     */
    void removeFeatureAttributes(String featureId) throws IOException;
    
    /**
     * Creates the feature type's attribute table(s).
     * @param feature The feature whose attributes will be created at the database
     * @return A map containing the Table,LinkColumn association, to link the
     * newly created attribute table(s) to the feature table.
     * @throws IOException if an error occurs during this method execution.
     */
    Map<String,String> createFeatureTypeAttributes(FeatureType feature) throws IOException;

    /**
     * Undo the just-executed createFeatureTypeAttributes method. This method is not
     * called at the drop table, it is an undo call for the createFeatureTypeAttributes
     * in case something goes wrong after the attribute tables are created.
     * This method respondes to TerralibAttributePersistenceHandlerTypeOperation.CREATE_FEATURE_TYPE
     * operation at the canProcess().
     * @param feature The feature whose attributes creating will be undone
     * @param linkColumns Map containing the Table,LinkColumn association
     * @throws IOException if an error occurs during this method execution.
     */
    void undoCreateFeatureTypeAttributes(FeatureType feature, Map<String,String> linkColumns) throws IOException;;
    
    /**
     * Drops the feature type's attribute table(s).
     * @param feature The feature whose attributes will be dropped from the database
     * @throws IOException if an error occurs during this method execution.
     */
    void dropFeatureTypeAttributes(FeatureType feature) throws IOException;

    /**
     * Removes all the given featureType non geometry attributes from the database.
     * @param typeName The feature type name.
     * @throws IOException if an error occurs during this method execution.
     */
	void removeAll(String typeName) throws IOException;
    
}
