/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2009(c).
 * 
 * file: TypeAttributeMap.java
 * created: 05/05/2009
 */
package org.geotools.data.terralib.util;

import java.util.Date;

import org.geotools.data.terralib.geometry.MultiTextGeometry;
import org.geotools.data.terralib.geometry.TextGeometry;
import org.geotools.data.terralib.swig.TeAttrDataType;
import org.geotools.data.terralib.swig.TeGeomRep;
import org.opengis.feature.type.AttributeType;
import org.opengis.feature.type.GeometryType;

import com.vividsolutions.jts.geom.LineString;
import com.vividsolutions.jts.geom.MultiLineString;
import com.vividsolutions.jts.geom.MultiPoint;
import com.vividsolutions.jts.geom.MultiPolygon;
import com.vividsolutions.jts.geom.Point;
import com.vividsolutions.jts.geom.Polygon;

/**
 * Controls the mapping and conversion between TerraLib Attribute 
 * data type, Terralib Geometry rep and GeoTools AttributeType. 
 * @author fabiomano
 * @since TDK3.0.0
 */
public enum TypeAttributeMap 
{
    // mapping entries
    // ATTENTION: ORDER IS IMPORTANT
    // - if there is more than one possible mapping (e.g., Terralib
    //   TeINT type can be possibly mapped to Integer or Long), then
    //   the FIRST entry will be preferred (i.e., all Terralib TeINT
    //   values are translated to Integers, even though both Integer
    //   and Long values are translated to TeINT)
    
    // attribute mappings
	TA_INTEGER(TeAttrDataType.TeINT,Integer.class,null)
	,TA_LONG(TeAttrDataType.TeINT,Long.class,null)	//we will translate long to int, terralib does not support long
	,TA_REAL(TeAttrDataType.TeREAL,Double.class,null)
	,TA_STRING(TeAttrDataType.TeSTRING,String.class,null)
	,TA_DATE(TeAttrDataType.TeDATETIME,Date.class,null)

	// geometry mappings: all geometries are mapped to their multi geometry
	//   equivalents, since non-multi geometry types are not supported by
	//   Terralib
	,TA_MULITEXT(TeAttrDataType.TeTEXTTYPE,MultiTextGeometry.class, TeGeomRep.TeTEXT)
	,TA_MULTIPOINT(TeAttrDataType.TePOINTTYPE,MultiPoint.class, TeGeomRep.TePOINTS)
	,TA_MULTILINE(TeAttrDataType.TeLINE2DTYPE,MultiLineString.class, TeGeomRep.TeLINES)
    ,TA_MULTIPOLYGON(TeAttrDataType.TePOLYGONTYPE,MultiPolygon.class, TeGeomRep.TePOLYGONS)
    ,TA_TEXT(TeAttrDataType.TeTEXTTYPE,TextGeometry.class, TeGeomRep.TeTEXT)
    ,TA_POINT(TeAttrDataType.TePOINTTYPE,Point.class, TeGeomRep.TePOINTS)
    ,TA_LINE(TeAttrDataType.TeLINE2DTYPE,LineString.class, TeGeomRep.TeLINES)
    ,TA_POLYGON(TeAttrDataType.TePOLYGONTYPE,Polygon.class, TeGeomRep.TePOLYGONS)
	;
	
	
	private TeAttrDataType _terralibAttr;
	private Class<?> _bindingClass;
	private TeGeomRep _geoRep;
	
	private TypeAttributeMap(TeAttrDataType terralibAttr, Class<?> bindingClass, TeGeomRep geoRep)
	{
		_terralibAttr = terralibAttr;
		_bindingClass = bindingClass;
		_geoRep = geoRep;
	}
	
	public Class<?> getBindingClass()
	{
		return _bindingClass;
	}
	
	public TeAttrDataType getTeAttributeType()
	{
		return _terralibAttr;
	}
	
	public TeGeomRep getGeomRep()
	{
		return _geoRep;
	}
	
	public boolean isGeometry()
	{
		return (_geoRep != null);
	}
	
	/**
	 * Returns the preferred type mapping given a TeGeomRep.
	 * ATTENTION: there may be more than one possible mapping (e.g.,
	 * TePOINTS > TA_MULTIPOINT or TA_POINT), but only the preferred
	 * one is returned (e.g., TA_MULTIPOINT)
	 *   
	 * @param geom
	 * @return
	 */
	public static TypeAttributeMap fromTeGeomRep(TeGeomRep geom)
	{
		for (int i=0; i< TypeAttributeMap.values().length; i++)
		{
			if (TypeAttributeMap.values()[i]._geoRep == geom)
				return TypeAttributeMap.values()[i];
		}
		return null;
	}
    
    /**
     * Returns the preferred type mapping given a TeAttrDataType.
     * ATTENTION: there may be more than one possible mapping (e.g.,
     * TeINT > TA_INTEGER or TA_LONG), but only the preferred
     * one is returned (e.g., TA_INTEGER)
     *   
     * @param attrDataType
     * @return
     */
    public static TypeAttributeMap fromTeAttrDataType(TeAttrDataType attrDataType)
    {
        for (int i=0; i< TypeAttributeMap.values().length; i++)
        {
            if (TypeAttributeMap.values()[i]._terralibAttr == attrDataType)
                return TypeAttributeMap.values()[i];
        }
        return null;
    }   
	
    /**
     * Returns the preferred type mapping given a Java binding class.
     * ATTENTION: if there is more than one possible mapping, only
     * the preferred is returned.
     *   
	 * @param binding
	 * @return
	 */
	public static TypeAttributeMap fromBindingClass(Class<?> binding)
	{
		for (int i=0; i< TypeAttributeMap.values().length; i++)
		{
			if (TypeAttributeMap.values()[i]._bindingClass.isAssignableFrom(binding))
				return TypeAttributeMap.values()[i];
		}
		return null;
	}	
	
    /**
     * Returns the preferred type mapping given an OpenGis GeometryType.
     * ATTENTION: if there is more than one possible mapping, only
     * the preferred is returned.
     *   
	 * @param geomType
	 * @return
	 */
	public static TypeAttributeMap fromGeometryType(GeometryType geomType)
	{
		return fromBindingClass(geomType.getBinding());
	}	
	
    /**
     * Returns the preferred type mapping given an OpenGis AttributeType.
     * ATTENTION: if there is more than one possible mapping, only
     * the preferred is returned.
     *   
	 * @param attrType
	 * @return
	 */
	public static TypeAttributeMap fromAttributeType(AttributeType attrType)
	{
		return fromBindingClass(attrType.getBinding());
	}	
}
