package tecgraf.openbus.browser;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ContainerAdapter;
import java.awt.event.ContainerEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.util.prefs.Preferences;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.PlainDocument;

import net.miginfocom.swing.MigLayout;

import org.eclipse.wb.swing.FocusTraversalOnArray;

import tecgraf.openbus.browser.AuthPanelListItem.SelectedConnectionChangeListener;
import tecgraf.openbus.browser.ManagedConnection.ConnectionState;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;
import tecgraf.openbus.exception.AlreadyLoggedIn;

/**
 * Painel de autenticao.  a parte da tela onde o usurio entra com servidor,
 * porta, login, senha, se conecta, v a lista das conexes feitas e d logoff
 * quando deseja.
 * 
 * <p>
 * Todas as operaes feitas a respeito de criar, conectar e desconectar so
 * feitas na instncia de {@link ManagedConnectionPool} informada no construtor.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
@SuppressWarnings("serial")
class AuthPanel extends JPanel {

	private ManagedConnectionPool managedConnectionPool;

	private final JTextField txtHost;
	private final JTextField txtPort;
	private final JTextField txtEntity;
	private final JPasswordField txtPassword;
	private JComboBox cmbLoginType;
	private final ButtonGroup connectionChooser;

	private final CertificatePanel certificatePanel;
	private LRUComboBox cmbLRU;
	private JScrollPane scrollPane;
	private JPanel pnlLogins;
	private JButton btnDoLogin;

	private final Preferences persistedPrefs = Preferences.userNodeForPackage(getClass());

	public enum LoginType {
		PASSWORD("Senha"),
		PRIVATE_KEY("Chave");
		//SHARED_AUTH ("Compartilhado"); 
		private final String label;

		LoginType(String label) {
			this.label = label;
		}

		@Override
		public String toString() {
			return label;
		}
	}

	private final KeyListener focusTransferKeyListener = new KeyAdapter() {
		@Override
		public void keyPressed(KeyEvent e) {
			if (e.getKeyCode() == KeyEvent.VK_ENTER) {
				e.consume();
				((Component) e.getSource()).transferFocus();
			}
			super.keyTyped(e);
		}
	};

	private final KeyListener doLoginKeyListener = new KeyAdapter() {
		@Override
		public void keyPressed(KeyEvent e) {
			if (e.getKeyCode() == KeyEvent.VK_ENTER) {
				e.consume();
				btnDoLogin.doClick();
			}
			super.keyTyped(e);
		}
	};

	private final DocumentListener lruClearDocumentListener = new DocumentListener() {

		private final void doUpdate() {
			String lruCandidate = getLRUText();
			int idx = ((DefaultComboBoxModel) cmbLRU.getModel()).getIndexOf(lruCandidate);
			int idxSelected = cmbLRU.getSelectedIndex();
			if (idx != idxSelected) {
				cmbLRU.setSelectedIndex(idx);
			}
		}

		@Override
		public void removeUpdate(DocumentEvent e) {
			doUpdate();
		}

		@Override
		public void insertUpdate(DocumentEvent e) {
			doUpdate();
		}

		@Override
		public void changedUpdate(DocumentEvent e) {
			doUpdate();
		}
	};

	private final class ValidLRUPartDocument extends PlainDocument {
		@Override
		public void insertString(int offs, String str, AttributeSet a) throws BadLocationException {
			if (str.indexOf('@') == -1 && str.indexOf(':') == -1)
				super.insertString(offs, str, a);
		}
	}

	private final class CertificatePanel extends JPanel {

		final JTextField txtKeyFile;
		final JButton btnFindCertificate;

		public CertificatePanel() {
			setLayout(new MigLayout("", "0[grow,fill]0", "0[grow, fill]0"));

			txtKeyFile = new JTextField();
			txtKeyFile.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
			txtKeyFile.addKeyListener(doLoginKeyListener);
			add(txtKeyFile, "cell 0 0, grow");

			btnFindCertificate = new JButton("...");
			btnFindCertificate.setMaximumSize(new Dimension(18, 18));
			add(btnFindCertificate, "cell 0 0");
			btnFindCertificate.addActionListener(new ActionListener() {
				@Override
				public void actionPerformed(ActionEvent e) {
					File f = lookForCertificateFile();
					if (f != null && f.exists() && f.isFile())
						txtKeyFile.setText(f.getAbsolutePath());
				}
			});
		}
	}

	public AuthPanel(ManagedConnectionPool argManagedConnectionPool) {
		this.managedConnectionPool = argManagedConnectionPool;

		setLayout(new MigLayout("", "[grow,left]", "[][][][grow]"));

		connectionChooser = new ButtonGroup();

		JLabel lblHost = new JLabel("Servidor:");
		add(lblHost, "flowx,cell 0 0");

		txtHost = new JTextField();
		txtHost.setDocument(new ValidLRUPartDocument());
		txtHost.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
		txtHost.addKeyListener(focusTransferKeyListener);
		txtHost.getDocument().addDocumentListener(lruClearDocumentListener);
		lblHost.setLabelFor(txtHost);
		add(txtHost, "cell 0 0, grow");

		JLabel lblPort = new JLabel("Porta:");
		add(lblPort, "cell 0 0");

		txtPort = new JTextField();
		txtPort.setColumns(5);
		txtPort.setMaximumSize(new Dimension(50, 20));
		txtPort.setDocument(new IntegerDocument());
		txtPort.addKeyListener(focusTransferKeyListener);
		txtPort.getDocument().addDocumentListener(lruClearDocumentListener);
		lblPort.setLabelFor(txtPort);
		add(txtPort, "cell 0 0");

		JLabel lblEntity = new JLabel("Entidade:");
		add(lblEntity, "flowx,cell 0 1");

		txtEntity = new JTextField();
		txtEntity.setDocument(new ValidLRUPartDocument());
		txtEntity.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
		txtEntity.addKeyListener(focusTransferKeyListener);
		txtEntity.getDocument().addDocumentListener(lruClearDocumentListener);
		lblEntity.setLabelFor(txtEntity);
		add(txtEntity, "cell 0 1,grow");

		cmbLoginType = new JComboBox();
		cmbLoginType.setModel(new DefaultComboBoxModel(LoginType.values()));
		cmbLoginType.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
		cmbLoginType.addKeyListener(focusTransferKeyListener);
		cmbLoginType.addItemListener(new ItemListener() {
			@Override
			public void itemStateChanged(ItemEvent e) {
				if (cmbLoginType.getModel().getSelectedItem() == null) {
					txtPassword.setVisible(false);
					certificatePanel.setVisible(false);
					return;
				}

				lruClearDocumentListener.changedUpdate(null);

				txtPassword.setEnabled(true);
				switch ((LoginType) cmbLoginType.getModel().getSelectedItem()) {
					case PASSWORD:
						txtPassword.setVisible(true);
						certificatePanel.setVisible(false);
						break;
					case PRIVATE_KEY:
						txtPassword.setVisible(false);
						certificatePanel.setVisible(true);
						break;
					default:
						throw new IllegalStateException();
				}
			}
		});
		add(cmbLoginType, "cell 0 1");

		txtPassword = new JPasswordField();
		txtPassword.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
		txtPassword.addKeyListener(doLoginKeyListener);
		add(txtPassword, "hidemode 3,cell 0 1,grow");

		certificatePanel = new CertificatePanel();
		certificatePanel.txtKeyFile.getDocument().addDocumentListener(lruClearDocumentListener);
		add(certificatePanel, "cell 0 1, hidemode 3, grow");

		JLabel lblLRU = new JLabel("LRU:");
		lblLRU.setToolTipText("Lista das configuraes recentemente utilizadas");
		add(lblLRU, "flowx,cell 0 2");

		txtPassword.setVisible(true);
		certificatePanel.setVisible(false);

		cmbLRU = new LRUComboBox();
		cmbLRU.setToolTipText("Ultimas configuraes de login utilizadas");
		cmbLRU.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
		cmbLRU.addKeyListener(focusTransferKeyListener);
		cmbLRU.addItemListener(new ItemListener() {
			@Override
			public void itemStateChanged(ItemEvent e) {
				String lruVal = (String) cmbLRU.getModel().getSelectedItem();
				if (lruVal == null)
					return;
				configureByLRUText(lruVal);
			}
		});
		lblLRU.setLabelFor(cmbLRU);
		add(cmbLRU, "cell 0 2, grow");

		btnDoLogin = new JButton("Login");
		btnDoLogin.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
		btnDoLogin.addActionListener(getDoLoginActionListener());
		add(btnDoLogin, "cell 0 2");
		btnDoLogin.setToolTipText("Realizar login. Clique com o boto direito para opes");
		btnDoLogin.addMouseListener(new MouseAdapter() {
			@Override
			public void mousePressed(MouseEvent e) {
				if (e.isPopupTrigger()) {
					e.consume();
					JPopupMenu opt = new JPopupMenu();
					final JCheckBoxMenuItem mnuAutoLogin = new JCheckBoxMenuItem(
						"Auto-login ao iniciar, se possvel");
					
					mnuAutoLogin.setSelected(persistedPrefs.getBoolean("autoLogin", true));
					
					mnuAutoLogin.addActionListener(new ActionListener() {
						@Override
						public void actionPerformed(ActionEvent e) {
							persistedPrefs.putBoolean("autoLogin", mnuAutoLogin.isSelected());
						}
					});
					opt.add(mnuAutoLogin);
					opt.show(btnDoLogin, 0, btnDoLogin.getHeight()+2);
				}
			}
		});

		scrollPane = new JScrollPane();
		add(scrollPane, "cell 0 3,grow");

		pnlLogins = new JPanel();
		pnlLogins.setLayout(new BoxLayout(pnlLogins, BoxLayout.Y_AXIS));
		pnlLogins.addContainerListener(new ContainerAdapter() {
			@Override
			public void componentRemoved(ContainerEvent e) {
				if (e.getChild() instanceof AuthPanelListItem) {
					AuthPanelListItem item = (AuthPanelListItem) e.getChild();
					managedConnectionPool.removeConnection(item.getConnection());
					if (managedConnectionPool.getPreferredConnection() == null) {
						for (Component candidate : e.getContainer().getComponents()) {
							if (candidate instanceof AuthPanelListItem) {
								AuthPanelListItem superCandidate = (AuthPanelListItem) candidate;
								if (superCandidate.getConnection().getState() == ConnectionState.AUTENTICATED) {
									superCandidate.setSelected();
									break;
								}
							}
						}
					}
				}
			}
		});
		scrollPane.setViewportView(pnlLogins);

		setFocusTraversalPolicy(new FocusTraversalOnArray(
		  new Component[] {
		      txtHost,
		      txtPort,
		      txtEntity,
		      cmbLoginType,
		      txtPassword,
		      certificatePanel,
		      btnDoLogin,
		      cmbLRU
		  }));

		cmbLRU.loadLRUJoinedText(persistedPrefs.get("lru", ""));
		cmbLRU.getModel().addListDataListener(new ListDataListener() {
			private void save() {
				persistedPrefs.put("lru", cmbLRU.getLRUJoinedText());
			}

			@Override
			public void intervalRemoved(ListDataEvent e) {
				save();
			}

			@Override
			public void intervalAdded(ListDataEvent e) {
				save();
			}

			@Override
			public void contentsChanged(ListDataEvent e) {
				save();
			}
		});

		//TODO:Opo de auto-login
		if (persistedPrefs.getBoolean("autoLogin", true)) {
			SwingUtilities.invokeLater(new Runnable() {
				@Override
				public void run() {
					if (isUserInputValidForLogin(false))
						btnDoLogin.doClick();
				}
			});
		}

	}

	private String getLRUText() {
		StringBuilder res = new StringBuilder();
		res.append(txtEntity.getText());
		res.append("@");
		res.append(txtHost.getText());
		res.append(":");
		res.append(txtPort.getText());
		if (cmbLoginType.getModel().getSelectedItem() == LoginType.PRIVATE_KEY) {
			res.append(":");
			res.append(certificatePanel.txtKeyFile.getText());
		}
		return res.toString();
	}

	private File lookForCertificateFile() {
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setDialogTitle("Indique a chave privada");
		fileChooser.setDialogType(JFileChooser.OPEN_DIALOG);
		fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		fileChooser.setApproveButtonText("OK");
		String currentDirFile = persistedPrefs.get("pkLookUpDir", System.getenv("user.home"));
		fileChooser.setSelectedFile(null);
		if (currentDirFile != null) {
			File d = new File(currentDirFile);
			if (d.exists() && d.isDirectory())
				fileChooser.setCurrentDirectory(d);
		}
		fileChooser.showDialog(this, "Ok");
		persistedPrefs.put("pkLookUpDir", fileChooser.getCurrentDirectory().getAbsolutePath());
		return fileChooser.getSelectedFile();
	}

	private void configureByLRUText(String lruText) {
		Pattern pat = Pattern.compile("([^\\s@]+)@([^\\s@]+)\\:([0-9]+)(?:\\:(.+))?");
		Matcher m = pat.matcher(lruText);
		if (!m.matches())
			throw new IllegalStateException("Texto LRU no est no padro correto: \"" + lruText + "\"");

		if (!txtEntity.getText().equals(m.group(1)))
			txtEntity.setText(m.group(1));

		if (!txtHost.getText().equals(m.group(2)))
			txtHost.setText(m.group(2));

		if (!txtPort.getText().equals(m.group(3)))
			txtPort.setText(m.group(3));

		if (m.group(4) != null) {
			if (cmbLoginType.getSelectedItem() != LoginType.PRIVATE_KEY)
				cmbLoginType.setSelectedItem(LoginType.PRIVATE_KEY);
			if (!certificatePanel.txtKeyFile.getText().equals(m.group(4)))
				certificatePanel.txtKeyFile.setText(m.group(4));
		}
		else {
			if (cmbLoginType.getSelectedItem() != LoginType.PASSWORD)
				cmbLoginType.setSelectedItem(LoginType.PASSWORD);
		}
	}

	private ActionListener getDoLoginActionListener() {
		return new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {

				if (!isUserInputValidForLogin(true))
					return;

				// Limpar conexes com erro da lista de logins
				Component actualLogins[] = pnlLogins.getComponents();
				for (Component c : actualLogins) {
					if (c instanceof AuthPanelListItem) {
						AuthPanelListItem a = (AuthPanelListItem) c;
						if (a.getConnection().getState() == ConnectionState.ERROR
						  || a.getConnection().getState() == ConnectionState.DISCONNECTED) {
							pnlLogins.remove(c);
						}
					}
				}

				final ManagedConnection newCnn;
				LoginType loginType = (LoginType) cmbLoginType.getModel().getSelectedItem();
				switch (loginType) {
					case PASSWORD:
						newCnn =
						  managedConnectionPool.createAndAddConnection(txtHost.getText(), Integer.parseInt(txtPort.getText()),
						    txtEntity.getText(), new String(txtPassword.getPassword()));
						break;
					case PRIVATE_KEY:
						try {
							newCnn =
							  managedConnectionPool.createAndAddConnection(txtHost.getText(), Integer.parseInt(txtPort.getText()),
							    txtEntity.getText(), new File(certificatePanel.txtKeyFile.getText()));
						}
						catch (Throwable t) {
							JOptionPane.showMessageDialog(AuthPanel.this, "Chave privada invlida: '" + t.getMessage() + "'", t
							  .getClass().getSimpleName() + " na chave privada",
							  JOptionPane.ERROR_MESSAGE);
							return;
						}
						break;
					default:
						throw new IllegalStateException();
				}

				final AuthPanelListItem newItem =
				  new AuthPanelListItem(connectionChooser, newCnn, new SelectedConnectionChangeListener() {
					  @Override
					  public void selectedConnection(ManagedConnection cnn) {
						  managedConnectionPool.setPreferredConnection(cnn);
					  }
				  });

				pnlLogins.add(newItem, 0);

				new Thread() {
					@Override
					public void run() {
						boolean ok = false;
						try {
							newCnn.autenticate();
							ok = true;
							SwingUtilities.invokeLater(new Runnable() {
								@Override
								public void run() {
									newItem.setSelected();
									String lru = getLRUText();
									cmbLRU.addLRUItem(lru);
									cmbLRU.setSelectedItem(lru);
								}
							});
							managedConnectionPool.notifyObservers();
						}
						catch (AlreadyLoggedIn e) {
						}
						catch (ServiceFailure e) {
							JOptionPane.showMessageDialog(AuthPanel.this, e.message, e.getClass().getSimpleName()
							  + " durante a autenticao!",
							  JOptionPane.ERROR_MESSAGE);
						}
						catch (Throwable e) {
							JOptionPane.showMessageDialog(AuthPanel.this, e.getMessage(), e.getClass().getSimpleName()
							  + " durante a autenticao!",
							  JOptionPane.ERROR_MESSAGE);
						}
						if (!ok) {
							SwingUtilities.invokeLater(new Runnable() {
								@Override
								public void run() {
									pnlLogins.remove(newItem);
								}
							});
						}
					}

				}.start();

			}

		};
	}

	private boolean validateTextFieldNotEmpty(boolean messageBox, JTextField txt, String fieldName) {
		if (txt.getText() == null || txt.getText().trim().isEmpty()) {
			if (messageBox) {
				JOptionPane.showMessageDialog(AuthPanel.this, fieldName + " est em branco.", "Ops!",
				  JOptionPane.ERROR_MESSAGE);
				txt.requestFocus();
			}
			return false;
		}
		return true;
	}

	private boolean isUserInputValidForLogin(boolean messageBox) {
		if (!validateTextFieldNotEmpty(messageBox, txtHost, "Host"))
			return false;
		if (!validateTextFieldNotEmpty(messageBox, txtPort, "Porta"))
			return false;
		if (!validateTextFieldNotEmpty(messageBox, txtEntity, "Entidade"))
			return false;
		LoginType loginType = (LoginType) cmbLoginType.getModel().getSelectedItem();
		switch (loginType) {
			case PASSWORD:
				if (!validateTextFieldNotEmpty(messageBox, txtPassword, "Senha"))
					return false;
				break;
			case PRIVATE_KEY:
				if (!validateTextFieldNotEmpty(messageBox, certificatePanel.txtKeyFile, "Chave privada"))
					return false;
				break;
		}
		return true;
	}

}
