package tecgraf.openbus.browser;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.BufferedImage;
import java.awt.image.RescaleOp;

import javax.swing.ButtonGroup;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.Timer;
import javax.swing.event.AncestorEvent;
import javax.swing.event.AncestorListener;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import net.miginfocom.swing.MigLayout;

import org.omg.CORBA.CompletionStatus;
import org.omg.CORBA.SystemException;

import tecgraf.openbus.browser.ManagedConnection.ConnectionState;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;

/**
 * Item do painel de autenticao da interface. Cada instncia desta classe
 * representa uma conexo com um servidor Openbus.
 * 
 * <p>
 * Esta instncia ir observar a conexo para exibir na interface todas as
 * mudanas de estado, assim como as vezes em que a conexo for usada como
 * receptora de requisies remotas de fora.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
@SuppressWarnings("serial")
class AuthPanelListItem extends JPanel {

	public interface SelectedConnectionChangeListener {
		public void selectedConnection(ManagedConnection cnn);
	}

	private final SelectedConnectionChangeListener changeListener;
	private final ManagedConnection cnn;
	private final Timer animationTimer = new Timer(100, new ActionListener() {
		@Override
		public void actionPerformed(ActionEvent e) {

			if (!AuthPanelListItem.this.isVisible() || AuthPanelListItem.this.getParent() == null) {
				animationTimer.stop();
				return;
			}

			int cnnCounter = cnn.getDispatchesToMeCounter();

			if (myDispatchCounter != cnnCounter)
				signalingDispatch = 10;
			else if (signalingDispatch > 0)
				signalingDispatch -= 1;

			myDispatchCounter = cnnCounter;

			if (signalingDispatch == 0)
				animationTimer.stop();
			else if (!animationTimer.isRunning())
				animationTimer.start();

			lblIcon.repaint();
		}
	});

	private static final ImageIcon ICON_DISCONNECTED = new ImageIcon(
	  AuthPanelListItem.class.getResource("login_disconnected_32.png"));
	private static final ImageIcon ICON_CONNECTED = new ImageIcon(
	  AuthPanelListItem.class.getResource("login_connected_32.png"));
	private static final ImageIcon ICON_CONNECTING = new ImageIcon(
	  AuthPanelListItem.class.getResource("login_connecting_32.gif"));
	private static final ImageIcon ICON_ERROR = new ImageIcon(
	  AuthPanelListItem.class.getResource("login_error_32.png"));
	private static final ImageIcon ICON_RECONNECTING = new ImageIcon(
	  AuthPanelListItem.class.getResource("login_reconnecting_32.gif"));
	private static final ImageIcon ICON_TRASH = new ImageIcon(
	  AuthPanelListItem.class.getResource("trashicon-12.png"));
	private static final BufferedImage ICON_LISTEN;
	static {
		ICON_LISTEN = new BufferedImage(16, 16, BufferedImage.TYPE_4BYTE_ABGR);
		Graphics2D g2 = ICON_LISTEN.createGraphics();
		g2.drawImage(new ImageIcon(OpenbusBrowser.class.getResource("ear-listen-16.png")).getImage(), 0, 0, null);
		g2.dispose();
	}

	private final AsyncSwingObserver myObserver = new AsyncSwingObserver() {
		@Override
		public void eventInEDT(AsyncObservable src) {
			updateController();
		}
	};

	private final JTextField txtBusID;
	private final JTextField txtLoginID;
	private final JTextField txtHostAndCredential;
	private final JLabel lblLoginType;

	private final JLabel lblIcon;

	private final JRadioButton rdbCnnSelector;

	private final JButton btnLogoff;

	private volatile int myDispatchCounter = 0;
	private volatile int signalingDispatch = 0;

	private final Icon connectedIcon = new Icon() {

		BufferedImage canvas = new BufferedImage(32, 32, BufferedImage.TYPE_4BYTE_ABGR);

		@Override
		public void paintIcon(Component c, Graphics g, int x, int y) {
			Graphics2D g2 = canvas.createGraphics();
			g2.setBackground(new Color(0, 0, 0, 0));
			g2.clearRect(0, 0, 32, 32);
			ICON_CONNECTED.paintIcon(c, g2, x, y);
			if (signalingDispatch > 0) {
				float[] scales = { 1f, 1f, 1f, (signalingDispatch / 10f) };
				float[] offsets = new float[4];
				RescaleOp rop = new RescaleOp(scales, offsets, null);
				g2.drawImage(ICON_LISTEN, rop, 16, 16);
			}
			g2.dispose();
			g.drawImage(canvas, 0, 0, null);
		}

		@Override
		public int getIconWidth() {
			return 32;
		}

		@Override
		public int getIconHeight() {
			return 32;
		}
	};

	public AuthPanelListItem(final ButtonGroup buttonGroup, final ManagedConnection argCnn,
	  final SelectedConnectionChangeListener argChangeListener) {
		this.cnn = argCnn;
		this.changeListener = argChangeListener;
		setLayout(new MigLayout("", "[20px:20px:20px,center][20px:20px:20px,center][32px:32px:32px,fill][grow]",
		  "1[15px:15px:15px,center]2[15px:15px:15px,center]1"));
		setMinimumSize(new Dimension(590, 36));
		setMaximumSize(new Dimension(Integer.MAX_VALUE, 36));

		Font lblFont = new Font(Font.DIALOG, Font.BOLD, 10);
		Font txtFont = new Font(Font.MONOSPACED, Font.PLAIN, 10);

		btnLogoff = new JButton(ICON_TRASH);
		btnLogoff.setMinimumSize(new Dimension(20, 20));
		btnLogoff.setMaximumSize(new Dimension(20, 20));
		btnLogoff.setFont(txtFont);

		btnLogoff.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				if (cnn.getState() == ConnectionState.ERROR) {
					killMeFromList();
					return;
				}

				try {
					if (JOptionPane.showConfirmDialog(AuthPanelListItem.this, "Confirma logoff?!", "Logoff",
					  JOptionPane.YES_NO_OPTION) == JOptionPane.NO_OPTION)
						return;

					if (cnn.logout()) {
						killMeFromList();
					}
				}
				catch (ServiceFailure e1) {
					JOptionPane.showMessageDialog(AuthPanelListItem.this, "Chave privada invlida: '" + e1.message + "'", e1
					  .getClass().getSimpleName() + " no logout",
					  JOptionPane.ERROR_MESSAGE);
				}
			}

			private void killMeFromList() {
				Container parent = getParent();
				parent.remove(AuthPanelListItem.this);
				parent.invalidate();
				parent.doLayout();
				parent.repaint();
			}
		});
		add(btnLogoff, "cell 0 0 1 2");

		rdbCnnSelector = new JRadioButton();
		rdbCnnSelector.addChangeListener(new ChangeListener() {
			@Override
			public void stateChanged(ChangeEvent e) {
				if (rdbCnnSelector.isSelected())
					changeListener.selectedConnection(cnn);
			}
		});
		add(rdbCnnSelector, "cell 1 0 1 2");

		lblIcon = new JLabel();
		lblIcon.setMinimumSize(new Dimension(32, 32));
		lblIcon.setMaximumSize(new Dimension(32, 32));
		lblIcon.setPreferredSize(new Dimension(32, 32));
		//lblIcon.setIcon(icon);
		add(lblIcon, "cell 2 0 1 2");

		JLabel lblBusid = new JLabel("BusID:");
		lblBusid.setFont(lblFont);
		add(lblBusid, "flowx,cell 3 0");

		txtBusID = new JTextField();
		txtBusID.setOpaque(false);
		txtBusID.setEditable(false);
		txtBusID.setBorder(null);
		txtBusID.setFont(txtFont);
		txtBusID.setMinimumSize(new Dimension(0, 0));
		add(txtBusID, "cell 3 0,grow");

		JLabel lblLoginID = new JLabel("LoginID:");
		lblLoginID.setFont(lblFont);
		add(lblLoginID, "flowx,cell 3 0");

		txtLoginID = new JTextField();
		txtLoginID.setOpaque(false);
		txtLoginID.setEditable(false);
		txtLoginID.setBorder(null);
		txtLoginID.setFont(txtFont);
		txtLoginID.setMinimumSize(new Dimension(0, 0));
		add(txtLoginID, "cell 3 0,grow");

		lblLoginType = new JLabel("Login por senha:");
		lblLoginType.setFont(lblFont);
		add(lblLoginType, "flowx,cell 3 1");

		txtHostAndCredential = new JTextField();
		txtHostAndCredential.setOpaque(false);
		txtHostAndCredential.setEditable(false);
		txtHostAndCredential.setBorder(null);
		txtHostAndCredential.setFont(txtFont);
		txtHostAndCredential.setMinimumSize(new Dimension(0, 0));
		add(txtHostAndCredential, "cell 3 1,grow");

		addAncestorListener(new AncestorListener() {
			@Override
			public void ancestorRemoved(AncestorEvent event) {
				buttonGroup.remove(rdbCnnSelector);
				cnn.removeObserver(myObserver);
			}

			@Override
			public void ancestorAdded(AncestorEvent event) {
				cnn.addObserver(myObserver);
				buttonGroup.add(rdbCnnSelector);
				myObserver.event(cnn);
			}

			@Override
			public void ancestorMoved(AncestorEvent event) {
			}
		});

	}

	private final void updateController() {
		Icon icon = null;
		boolean selectable = false;
		boolean logoffable = false;
		String iconTooltip;
		switch (cnn.getState()) {
			case CONNECTED:
				icon = ICON_CONNECTED;
				selectable = false;
				logoffable = false;
				iconTooltip = "Conectado, sem autenticao.";
				break;
			case CONNECTING:
				icon = ICON_CONNECTING;
				selectable = false;
				logoffable = false;
				iconTooltip = "Aguarde conexo...";
				break;
			case DISCONNECTED:
				icon = ICON_DISCONNECTED;
				selectable = false;
				logoffable = false;
				iconTooltip = "Desconectado.";
				break;
			case ERROR:
				icon = ICON_ERROR;
				selectable = false;
				logoffable = true;
				iconTooltip = getErrorTooltip();
				break;
			case AUTENTICATING:
				icon = ICON_CONNECTING;
				selectable = false;
				logoffable = false;
				iconTooltip = "Autenticando no Openbus...";
				break;
			case AUTENTICATED:
				icon = connectedIcon;
				selectable = true;
				logoffable = true;
				iconTooltip = "Conectado e autenticado";
				break;
			case REAUTENTICATING:
				icon = ICON_RECONNECTING;
				selectable = false;
				logoffable = false;
				iconTooltip = "Renovando conexo e autenticao...";
				break;
			default:
				throw new IllegalStateException();
		}

		if (selectable != rdbCnnSelector.isEnabled()) {
			if (!selectable)
				rdbCnnSelector.setSelected(false);
			rdbCnnSelector.setEnabled(selectable);
		}

		if (logoffable != btnLogoff.isEnabled())
			btnLogoff.setEnabled(logoffable);

		if (lblIcon.getIcon() != icon) {
			lblIcon.setIcon(icon);
		}

		if (!iconTooltip.equals(lblIcon.getToolTipText()))
			lblIcon.setToolTipText(iconTooltip);

		String busid = cnn.busid();
		if (!txtBusID.getText().equals(busid))
			txtBusID.setText(busid);

		String loginId = "N/A";
		if (cnn.login() != null && cnn.login().id != null && !cnn.login().id.isEmpty())
			loginId = cnn.login().id;
		if (!txtLoginID.getText().equals(loginId))
			txtLoginID.setText(loginId);

		String tooltip;
		String loginType;
		switch (cnn.getLoginType()) {
			case PASSWORD:
				loginType = "Login por senha:";
				tooltip = "";
				break;
			case PRIVATE_KEY:
				loginType = "Login por chave:";
				tooltip = "Chave privada: " + cnn.getKeyFile().getAbsolutePath();
				break;
			default:
				throw new IllegalStateException();
		}
		if (!lblLoginType.getText().equals(loginType))
			lblLoginType.setText(loginType);

		if (lblLoginType.getToolTipText() == null)
			lblLoginType.setToolTipText("");

		if (!lblLoginType.getToolTipText().equals(tooltip))
			lblLoginType.setToolTipText(tooltip);

		StringBuilder hostCred = new StringBuilder();
		hostCred.append(cnn.getEntity());
		hostCred.append("@");
		hostCred.append(cnn.getHost());
		hostCred.append(":");
		hostCred.append(cnn.getPort());
		String hostCredStr = hostCred.toString();

		if (txtHostAndCredential.getToolTipText() == null)
			txtHostAndCredential.setToolTipText("");

		if (!txtHostAndCredential.equals(hostCredStr))
			txtHostAndCredential.setText(hostCredStr);

		if (!txtHostAndCredential.getToolTipText().equals(tooltip))
			txtHostAndCredential.setToolTipText(tooltip);

		if (signalingDispatch > 0 || cnn.getDispatchesToMeCounter() != myDispatchCounter) {
			animationTimer.getActionListeners()[0].actionPerformed(null);
		}

	}

	private String getErrorTooltip() {
		if (cnn.getLastError() == null)
			return "No sei o que houve. :-(";

		StringBuilder res = new StringBuilder();
		res.append("<html>");
		Throwable e = cnn.getLastError();
		int count = 0;
		while (e != null) {
			count += 1;
			if (count > 1) {
				res.append("<br><br>causada por ");
			}
			res.append("<b>" + e.getClass().getName() + "</b><br>");
			String msg;
			if (e instanceof ServiceFailure) {
				msg = ((ServiceFailure) e).message;
				if (msg == null || msg.isEmpty())
					msg = e.getMessage();
			}
			else
				msg = e.getMessage();

			if (e instanceof SystemException) {
				SystemException se = (SystemException) e;
				msg = msg + " (completed=" + getCompletionStatusString(se.completed) + "; minor=" + se.minor + ")";
			}

			if (msg != null && !msg.isEmpty()) {
				res.append("<small><i><pre>" + msg + "</pre></i></small>");
			}

			e = e.getCause();
		}
		res.append("</html>");
		return res.toString();
	}

	private String getCompletionStatusString(CompletionStatus completed) {
		switch (completed.value()) {
			case CompletionStatus._COMPLETED_MAYBE:
				return "maybe";
			case CompletionStatus._COMPLETED_NO:
				return "no";
			case CompletionStatus._COMPLETED_YES:
				return "yes";
			default:
				return "Status " + completed.value();
		}
	}

	public void setSelected() {
		rdbCnnSelector.setSelected(true);
	}

	public boolean isSelected() {
		return rdbCnnSelector.isSelected();
	}

	public ManagedConnection getConnection() {
		return cnn;
	}

}