package tecgraf.openbus.browser;

import java.io.File;
import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;

import org.omg.CORBA.ORB;
import org.omg.CORBA.ORBPackage.InvalidName;

import tecgraf.openbus.Connection;
import tecgraf.openbus.InvalidLoginCallback;
import tecgraf.openbus.OpenBusContext;
import tecgraf.openbus.PrivateKey;
import tecgraf.openbus.browser.AuthPanel.LoginType;
import tecgraf.openbus.core.OpenBusPrivateKey;
import tecgraf.openbus.core.v2_0.OctetSeqHolder;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;
import tecgraf.openbus.core.v2_0.services.access_control.AccessDenied;
import tecgraf.openbus.core.v2_0.services.access_control.LoginInfo;
import tecgraf.openbus.core.v2_0.services.access_control.LoginProcess;
import tecgraf.openbus.core.v2_0.services.access_control.MissingCertificate;
import tecgraf.openbus.exception.AlreadyLoggedIn;
import tecgraf.openbus.exception.InvalidLoginProcess;

/**
 * Decorador para um bean de conexo do Openbus, que adiciona as seguintes
 * funcionalidades convenientes para a interface:
 * <ul>
 * <li>O faz exportar a fase do ciclo de vida em que est (ConnectionState);
 * <li>Persiste no prprio bean todas as informaes sobre host, porta, e
 * autenticao, de forma imutvel;
 * <li>Torna o estado da conexo observvel por outras partes do sistema, para
 * viabilizar refletir na interface grfica mudanas no estado da conexo;
 * <li>Acumula triggers pr-logoff para serem executados automaticamente no
 * logoff.
 * </ul>
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
public class ManagedConnection extends AsyncObservable implements Connection {

	public enum ConnectionState {
		DISCONNECTING,
		DISCONNECTED,
		CONNECTING,
		CONNECTED,
		AUTENTICATING,
		AUTENTICATED,
		REAUTENTICATING,
		ERROR
	}

	public interface LogoffSyncListener {
		public void preLogoff(ManagedConnection cnn);
	}

	private Connection myConnection;
	private Throwable lastError;
	private long lastErrorTimestamp;

	private final OpenBusContext openbusContext;
	private final LoginType loginType;
	private final String host;
	private final int port;
	private final String entity;
	private final String password;
	private final OpenBusPrivateKey keyFile;
	private final File keyFilePath;

	private final AtomicInteger dispatchesToMeCounter = new AtomicInteger(0);

	private final CopyOnWriteArrayList<LogoffSyncListener> logoffListeners =
	  new CopyOnWriteArrayList<ManagedConnection.LogoffSyncListener>();

	private volatile ConnectionState state;

	private final InvalidLoginCallback myInvalidLoginCallback = new InvalidLoginCallback() {
		@Override
		public void invalidLogin(Connection arg0, LoginInfo arg1) {
			if (state == ConnectionState.ERROR) {
				System.out.println("Ignorando onInvalidLoginCallback por estar em erro.");
				return;
			}

			setStateWithNotify(ConnectionState.REAUTENTICATING);

			try {
				autenticate();
			}
			catch (tecgraf.openbus.exception.AlreadyLoggedIn a) {

			}
			catch (Throwable e) {
				e.printStackTrace(System.err);
				lastError = e;
				lastErrorTimestamp = System.currentTimeMillis();
				setStateWithNotify(ConnectionState.ERROR);
			}
		}
	};

	protected ManagedConnection(OpenBusContext openbusContext, String host, int port, String entity, String password) {
		super();
		this.myConnection = null;
		this.state = ConnectionState.DISCONNECTED;
		this.openbusContext = openbusContext;
		this.loginType = LoginType.PASSWORD;
		this.host = host;
		this.port = port;
		this.entity = entity;
		this.password = password;
		this.keyFile = null;
		this.keyFilePath = null;
	}

	protected ManagedConnection(OpenBusContext openbusContext, String host, int port, String entity, File keyFile)
	  throws NoSuchAlgorithmException, InvalidKeySpecException, IOException {
		super();
		this.myConnection = null;
		this.state = ConnectionState.DISCONNECTED;
		this.openbusContext = openbusContext;
		this.loginType = LoginType.PRIVATE_KEY;
		this.host = host;
		this.port = port;
		this.entity = entity;
		this.password = null;
		this.keyFile = OpenBusPrivateKey
		  .createPrivateKeyFromFile(keyFile.getAbsolutePath());
		this.keyFilePath = keyFile;
	}

	public final void connect() {
		if (this.myConnection == null) {
			setStateWithNotify(ConnectionState.CONNECTING);
			myConnection = openbusContext.createConnection(host, port);
			myConnection.onInvalidLoginCallback(myInvalidLoginCallback);
			setStateWithNotify(ConnectionState.CONNECTED);
		}
	}

	private void setStateWithNotify(ConnectionState state) {
		this.state = state;
		setChanged();
		notifyObservers();
	}

	public final void autenticate() throws AccessDenied, AlreadyLoggedIn, ServiceFailure, MissingCertificate {

		connect();

		switch (loginType) {
			case PASSWORD:
				this.loginByPassword(entity, password.getBytes());
				break;
			case PRIVATE_KEY:
				this.loginByCertificate(entity, keyFile);
				break;
		}

	}

	public Throwable getLastError() {
		return lastError;
	}

	public void setLastError(Throwable lastError) {
		this.lastError = lastError;
	}

	public long getLastErrorTimestamp() {
		return lastErrorTimestamp;
	}

	public void setLastErrorTimestamp(long lastErrorTimestamp) {
		this.lastErrorTimestamp = lastErrorTimestamp;
	}

	public ConnectionState getState() {
		return state;
	}

	@Override
	public String busid() {
		if (myConnection == null || myConnection.busid() == null)
			return "N/A";
		return myConnection.busid();
	}

	@Override
	public LoginInfo login() {
		if (myConnection == null)
			return null;
		return myConnection.login();
	}

	@Override
	public void loginByCertificate(String arg0, PrivateKey arg1) throws AlreadyLoggedIn, AccessDenied,
	  MissingCertificate, ServiceFailure {

		if (arg0 == null)
			throw new IllegalArgumentException("loginByCertificate com entidade nula!!");

		if (arg1 == null)
			throw new IllegalArgumentException("loginByCertificate com PrivateKey nula!!");

		if (state != ConnectionState.REAUTENTICATING)
			setStateWithNotify(ConnectionState.AUTENTICATING);

		try {
			myConnection.loginByCertificate(arg0, arg1);
			setStateWithNotify(ConnectionState.AUTENTICATED);
		}
		catch (AlreadyLoggedIn e) {
			setStateWithNotify(ConnectionState.AUTENTICATED);
			throw e;
		}
		catch (Throwable e) {
			lastError = e;
			lastErrorTimestamp = System.currentTimeMillis();
			setStateWithNotify(ConnectionState.ERROR);
		}

	}

	@Override
	public void loginByPassword(String arg0, byte[] arg1) throws AccessDenied, AlreadyLoggedIn, ServiceFailure {

		if (state != ConnectionState.REAUTENTICATING)
			setStateWithNotify(ConnectionState.AUTENTICATING);

		try {
			myConnection.loginByPassword(arg0, arg1);
			setStateWithNotify(ConnectionState.AUTENTICATED);
		}
		catch (AlreadyLoggedIn e) {
			setStateWithNotify(ConnectionState.AUTENTICATED);
			throw e;
		}
		catch (Throwable e) {
			lastError = e;
			lastErrorTimestamp = System.currentTimeMillis();
			setStateWithNotify(ConnectionState.ERROR);
		}

	}

	@Override
	public void loginBySharedAuth(LoginProcess arg0, byte[] arg1) throws AlreadyLoggedIn, InvalidLoginProcess,
	  AccessDenied, ServiceFailure {
		if (state != ConnectionState.REAUTENTICATING)
			setStateWithNotify(ConnectionState.AUTENTICATING);

		try {
			myConnection.loginBySharedAuth(arg0, arg1);
			setStateWithNotify(ConnectionState.AUTENTICATED);
		}
		catch (AlreadyLoggedIn e) {
			setStateWithNotify(ConnectionState.AUTENTICATED);
			throw e;
		}
		catch (Throwable e) {
			lastError = e;
			lastErrorTimestamp = System.currentTimeMillis();
			setStateWithNotify(ConnectionState.ERROR);
		}

	}

	@Override
	public boolean logout() throws ServiceFailure {
		setStateWithNotify(ConnectionState.DISCONNECTING);
		for (LogoffSyncListener listeners : logoffListeners)
			listeners.preLogoff(this);
		boolean res = myConnection.logout();
		setStateWithNotify(ConnectionState.DISCONNECTED);
		return res;
	}

	@Override
	public InvalidLoginCallback onInvalidLoginCallback() {
		return myConnection.onInvalidLoginCallback();
	}

	@Override
	public void onInvalidLoginCallback(InvalidLoginCallback arg0) {
		throw new IllegalAccessError("Tentativa indevida de trocar o onInvalidLoginCallback()!");
	}

	@Override
	public ORB orb() {
		return myConnection.orb();
	}

	@Override
	public LoginProcess startSharedAuth(OctetSeqHolder arg0) throws ServiceFailure {
		return myConnection.startSharedAuth(arg0);
	}

	/**
	 * @return the loginType
	 */
	public LoginType getLoginType() {
		return loginType;
	}

	/**
	 * @return the host
	 */
	public String getHost() {
		return host;
	}

	/**
	 * @return the port
	 */
	public int getPort() {
		return port;
	}

	/**
	 * @return the entity
	 */
	public String getEntity() {
		return entity;
	}

	/**
	 * @return the keyFilePath
	 */
	public File getKeyFile() {
		return keyFilePath;
	}

	public void setContextCurrentConnection() {
		openbusContext.setCurrentConnection(myConnection);
	}

	public static void setContextCurrentConnection(Connection cnn) {
		if (cnn instanceof ManagedConnection)
			getContext(cnn).setCurrentConnection(((ManagedConnection)cnn).getMyConnection());
		else
			getContext(cnn).setCurrentConnection(cnn);
	}

	public void setDefaultContextConnection() {
		openbusContext.setDefaultConnection(myConnection);
	}

	public static void setDefaultContextConnection(Connection cnn) {
		if (cnn instanceof ManagedConnection)
			getContext(cnn).setDefaultConnection(((ManagedConnection)cnn).getMyConnection());
		else
			getContext(cnn).setDefaultConnection(cnn);
	}

	public static OpenBusContext getContext(Connection cnn) {
		try {
			return (OpenBusContext) cnn.orb().resolve_initial_references("OpenBusContext");
		}
		catch (InvalidName e) {
			throw new IllegalStateException("Cad o OpenBusContext?!?!", e);
		}
	}

	public OpenBusContext getContext() {
		return openbusContext;
	}

	public Connection getMyConnection() {
		return myConnection;
	}

	public void addLogoffSyncListener(LogoffSyncListener listener) {
		logoffListeners.add(listener);
	}

	public void clearLogoffSyncListener() {
		logoffListeners.clear();
	}

	public boolean removeLogoffSyncListener(LogoffSyncListener listener) {
		return logoffListeners.remove(listener);
	}

	public int getDispatchesToMeCounter() {
		return dispatchesToMeCounter.get();
	}

	public void incrementDispatchesToMeCounter() {
		dispatchesToMeCounter.incrementAndGet();
		setChanged();
		notifyObservers();
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((entity == null) ? 0 : entity.hashCode());
		result = prime * result + ((host == null) ? 0 : host.hashCode());
		result = prime * result + ((loginType == null) ? 0 : loginType.hashCode());
		result = prime * result + ((myConnection == null) ? 0 : myConnection.hashCode());
		result = prime * result + port;
		return result;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (!(obj instanceof ManagedConnection)) {
			return false;
		}
		ManagedConnection other = (ManagedConnection) obj;
		if (entity == null) {
			if (other.entity != null) {
				return false;
			}
		}
		else if (!entity.equals(other.entity)) {
			return false;
		}
		if (host == null) {
			if (other.host != null) {
				return false;
			}
		}
		else if (!host.equals(other.host)) {
			return false;
		}
		if (loginType != other.loginType) {
			return false;
		}
		if (myConnection == null) {
			if (other.myConnection != null) {
				return false;
			}
		}
		else if (!myConnection.equals(other.myConnection)) {
			return false;
		}
		if (port != other.port) {
			return false;
		}
		return true;
	}

}