package tecgraf.openbus.browser;

import java.io.File;
import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.List;
import java.util.Observable;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Logger;

import tecgraf.openbus.CallDispatchCallback;
import tecgraf.openbus.Connection;
import tecgraf.openbus.OpenBusContext;
import tecgraf.openbus.browser.AsyncObservable.AsyncObserver;
import tecgraf.openbus.browser.ManagedConnection.ConnectionState;

/**
 * Colecionador de conexes com o barramento, que prov funcionalidades de
 * recuperao de conexes para atendimento de requisies externas ou para uso
 * ativo.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
class ManagedConnectionPool extends Observable implements ConnectionProvider {

	private final List<ManagedConnection> conns = new CopyOnWriteArrayList<ManagedConnection>();
	private volatile ManagedConnection preferedConn;
	private final Logger logger = Logger.getLogger(ManagedConnectionPool.class.getName());

	private final OpenBusContext openBusContext;

	private final CallDispatchCallback callDispatchCallback = new CallDispatchCallback() {
		@Override
		public Connection dispatch(OpenBusContext context, String busid, String loginId,
		  byte[] object_id, String operation) {
			ManagedConnection res = getActiveConnectionsOfBusID(busid);
			if (res != null) {
				res.incrementDispatchesToMeCounter();
				return res.getMyConnection();
			}
			logger.warning("Requisio de entrada para ORB recebida sem nenhuma conexo aberta!" +
			  "Detalhes: busid=" + busid + ", loginid=" + loginId + ", operation=" + operation);
			return null;
		}
	};

	private final AsyncObserver connectionObserver  = new AsyncObserver() {
		@Override
		public void event(AsyncObservable src) {
			setChanged();
			notifyObservers();
		}
	};
	
	public ManagedConnectionPool(OpenBusContext openBusContext) {
		this.openBusContext = openBusContext;
		this.openBusContext.onCallDispatch(callDispatchCallback);
	}

	/**
	 * Incluir nova conexo no pool.
	 * 
	 * @param cnn Nova conexo a ser includa no pool.
	 */
	private void addConnection(ManagedConnection cnn) {
		conns.add(cnn);
		cnn.addObserver(connectionObserver);
		setChanged();
	}

	/**
	 * Criar nova conexo, j se conectando e incluindo no pool.
	 * 
	 * <p>
	 * O processo de autenticao no  feito por este mtodo. Aps criar a
	 * conexo, o mtodo {@link ManagedConnection#autenticate()} deve ser chamado
	 * para que a conexo e autenticao sejam feitos.
	 * 
	 * @param host Servidor do Openbus a se conectar.
	 * @param port Porta do servidor Openbus
	 * @param entity Entidade a ser usada para autenticao.
	 * @param password Senha para autenticao login-senha.
	 * @return Bean com a conexo.
	 */
	public ManagedConnection createAndAddConnection(String host, int port, String entity, String password) {
		ManagedConnection res = new ManagedConnection(openBusContext, host, port, entity, password);
		addConnection(res);
		return res;
	}

	/**
	 * Criar nova conexo, j se conectando e incluindo no pool.
	 * 
	 * <p>
	 * O processo de autenticao no  feito por este mtodo. Aps criar a
	 * conexo, o mtodo {@link ManagedConnection#autenticate()} deve ser chamado
	 * para que a conexo e autenticao sejam feitos.
	 * 
	 * @param host Servidor do Openbus a se conectar.
	 * @param port Porta do servidor Openbus
	 * @param entity Entidade a ser usada para autenticao.
	 * @param keyFile Arquivo com a chave privada, para autenticao por
	 *          certificado.
	 * @return Bean com a conexo.
	 */
	public ManagedConnection createAndAddConnection(String host, int port, String entity, File keyFile)
	  throws NoSuchAlgorithmException, InvalidKeySpecException, IOException {
		ManagedConnection res = new ManagedConnection(openBusContext, host, port, entity, keyFile);
		addConnection(res);
		return res;
	}

	/**
	 * Remover conexo do pool.
	 * 
	 * <p>
	 * A remoo do pool envolve implicitamente o logout().
	 * 
	 * @param cnn Bean da conexo que ser desfeita e removida.
	 * @return true se a conexo estava de fato no pool.
	 */
	public boolean removeConnection(ManagedConnection cnn) {
		if (preferedConn == cnn)
			preferedConn = null;
		if (cnn.getState() == ConnectionState.AUTENTICATED) {
			try {
				cnn.logout();
			}
			catch (Throwable e) {
				e.printStackTrace(System.err);
			}
		}
		cnn.removeObserver(connectionObserver);
		return conns.remove(cnn);
	}

	/**
	 * Configura a conexo de uso preferencial tanto para recebimento de chamadas
	 * externas como para fazer chamadas remotas.
	 * 
	 * <p>
	 * Esta conexo preferencial  usada para desfazer ambiguidades no caso de
	 * conexes de entrada ou para refletir a opo do usurio, que a faz atravs
	 * dos RadioButtons da lista de conexes.
	 * 
	 * @param cnn A conexo que ser a preferencial.
	 * @throws IllegalStateException Caso a conexo informada no esteja no pool
	 */
	public final void setPreferredConnection(ManagedConnection cnn) {
		if (conns.remove(cnn)) {
			preferedConn = cnn;
			conns.add(0, cnn);
			setChanged();
		}
		else
			throw new IllegalStateException("Conexo no est no pool!");
	}

	/**
	 * Recuperar qual das conexes do pool  a de uso preferencial.
	 * 
	 * <p>
	 * Esta conexo preferencial  usada para desfazer ambiguidades no caso de
	 * conexes de entrada ou para refletir a opo do usurio, que a faz atravs
	 * dos RadioButtons da lista de conexes.
	 * 
	 * @return Conexo preferencial.
	 */
	public final ManagedConnection getPreferredConnection() {
		ManagedConnection pref = preferedConn;
		if (pref != null)
			if (pref.getState() == ConnectionState.AUTENTICATED)
				return pref;
		return null;
	}

	@Override
	public ManagedConnection getConnectionToUse() {
		ManagedConnection res = getPreferredConnection();
		if (res != null)
			return res;

		for (ManagedConnection cnn : conns) {
			if (cnn.getState() == ConnectionState.AUTENTICATED)
				return cnn;
		}

		return null;
	}

	@Override
	public ManagedConnection getManagedConnectionOf(Connection primitiveConnection) {
		for (ManagedConnection cnn : conns) {
			if (cnn.getMyConnection() == primitiveConnection)
				return cnn;
		}
		return null;
	}

	@Override
	public int getNumActiveConnectionsOfBusID(String busID) {
		int res = 0;
		for (ManagedConnection cnn : conns) {
			if (cnn.busid().equals(busID)) {
				if (cnn.getState() == ConnectionState.AUTENTICATED
				  || cnn.getState() == ConnectionState.DISCONNECTING) {
					res += 1;
				}
			}
		}
		return res;
	}

	@Override
	public ManagedConnection getActiveConnectionsOfBusID(String busID) {
		ManagedConnection res = getPreferredConnection();
		if (res != null && res.busid().equals(busID))
			return res;

		for (ManagedConnection cnn : conns) {
			if (cnn.busid().equals(busID)) {
				if (cnn.getState() == ConnectionState.AUTENTICATED
				  || cnn.getState() == ConnectionState.DISCONNECTING) {
					return cnn;
				}
			}
		}

		return null;
	}

	/**
	 * Limpar o pool inteiro, desfazendo todas as conexes com logout.
	 */
	public synchronized void shutdown() {
		preferedConn = null;
		for (ManagedConnection cnn : conns) {
			try {
				if (cnn.getState() != ConnectionState.DISCONNECTED)
					cnn.logout();
			}
			catch (Throwable e) {
				e.printStackTrace(System.err);
			}
			cnn.removeObserver(connectionObserver);
		}
		conns.clear();
	}

	/**
	 * Recuperar a instncia de {@link OpenBusContext} usada por este pool para as
	 * operaes pertinentes.
	 * 
	 * @return A instncia usada.
	 */
	public OpenBusContext getOpenBusContext() {
		return openBusContext;
	}

}
