package tecgraf.openbus.browser;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowFocusListener;
import java.io.IOException;
import java.lang.Thread.UncaughtExceptionHandler;
import java.util.concurrent.atomic.AtomicInteger;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;

import net.miginfocom.swing.MigLayout;

/**
 * Dilogo singleton e modal para exibio de excees para o usurio, da forma
 * mais detalhada possvel.
 * 
 * A instncia deste dilogo, quando usada em
 * {@link Thread#setUncaughtExceptionHandler(UncaughtExceptionHandler)} ou em
 * {@link Thread#setDefaultUncaughtExceptionHandler(UncaughtExceptionHandler)}
 * faz com que as excees no tratadas sejam exibidas para o usurio em um
 * dilogo modal.
 * 
 * O dilogo, porm, s ir exibir no mximo cinco excees para no
 * sobrecarregar a tela. Caso hajam mais, elas estaro no log (System.out).
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
@SuppressWarnings("serial")
class SwingUncaughtExceptionHandler extends JDialog implements UncaughtExceptionHandler {

	private JPanel exceptions;
	private final AtomicInteger errorCounter = new AtomicInteger(0);
	private JButton btnOk;

	public SwingUncaughtExceptionHandler() {
		super();
		setModalityType(ModalityType.APPLICATION_MODAL);
		init();
	}

	public SwingUncaughtExceptionHandler(Dialog owner) {
		super(owner, ModalityType.APPLICATION_MODAL);
		init();
	}

	public SwingUncaughtExceptionHandler(Frame owner) {
		super(owner, ModalityType.APPLICATION_MODAL);
		init();
	}

	public SwingUncaughtExceptionHandler(Window owner) {
		super(owner, ModalityType.APPLICATION_MODAL);
		init();
	}

	private final void init() {
		setLayout(new MigLayout("", "[grow]", "[grow,fill][grow][center]"));
		setMinimumSize(new Dimension(600, 200));
		setPreferredSize(new Dimension(600, 200));

		JLabel title = new JLabel("Ocorreu uma exceo!");
		title.setIcon(new ImageIcon(getClass().getResource("login_error_32.png")));
		title.setFont(new Font(Font.DIALOG, Font.BOLD, 16));
		add(title, "cell 0 0");

		exceptions = new JPanel();
		exceptions.setLayout(new MigLayout("", "0[grow,fill]0"));
		final JScrollPane scroll = new JScrollPane(exceptions);
		scroll.setBorder(null);

		btnOk = new JButton("Ok");
		btnOk.requestFocus();
		btnOk.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				setVisible(false);
				exceptions.removeAll();
				errorCounter.set(0);
			}
		});

		setDefaultCloseOperation(JDialog.HIDE_ON_CLOSE);
		add(scroll, "cell 0 1, grow");
		add(btnOk, "cell 0 2, center");

		setAlwaysOnTop(true);

		setKeyListener(this, new KeyAdapter() {
			@Override
			public void keyPressed(KeyEvent e) {
				if (e.getKeyCode() == KeyEvent.VK_ENTER || e.getKeyCode() == KeyEvent.VK_BACK_SPACE) {
					e.consume();
					btnOk.doClick();
				}
			}
		});

		addWindowFocusListener(new WindowFocusListener() {
			@Override
			public void windowLostFocus(WindowEvent arg0) {
			}

			@Override
			public void windowGainedFocus(WindowEvent arg0) {
				btnOk.requestFocus();
			}
		});

		addWindowListener(new WindowAdapter() {
			@Override
			public void windowActivated(WindowEvent arg0) {
				btnOk.requestFocus();
			}

			@Override
			public void windowGainedFocus(WindowEvent arg0) {
				btnOk.requestFocus();
			}
		});
	}

	private static void setKeyListener(Component c, KeyListener l) {
		c.addKeyListener(l);

		if (c instanceof Container) {
			for (Component f : ((Container) c).getComponents()) {
				setKeyListener(f, l);
			}
		}
	}

	@Override
	public void uncaughtException(final Thread t, final Throwable e) {

		e.printStackTrace(System.err);

		try {
			Runnable shower = new Runnable() {
				@Override
				public void run() {
					SwingUncaughtExceptionHandler dialog = SwingUncaughtExceptionHandler.this;
					synchronized (dialog) {

						if (dialog.isVisible()) {
							int counter = dialog.errorCounter.incrementAndGet();
							if (counter <= 1)
								dialog.setTitle("ERRO");
							else if (counter > 1)
								dialog.setTitle(counter + " ERROS");

							if (counter > 5)
								return;

							JPanel div = new JPanel();
							div.setMinimumSize(new Dimension(0, 5));
							div.setMaximumSize(new Dimension(Integer.MAX_VALUE, 5));
							div.setOpaque(true);
							div.setBackground(Color.black);
							exceptions.add(div, "cell 0 0, flowy");

							if (counter == 5) {
								JLabel lblMore = new JLabel("As demais excees esto no log.");
								exceptions.add(lblMore, "cell 0 0, flowy");
							}
							else if (counter < 5) {
								exceptions.add(new SwingUncaughtExceptionHandlerItem(e, t), "cell 0 0, flowy");
							}
						}
						else {
							dialog.setTitle("ERRO");
							dialog.errorCounter.set(1);
							dialog.exceptions.removeAll();
							dialog.exceptions.add(new SwingUncaughtExceptionHandlerItem(e, t), "cell 0 0, flowy");
							dialog.exceptions.invalidate();
							dialog.exceptions.doLayout();
							dialog.setLocationRelativeTo(dialog.getParent());
							dialog.setVisible(true);
						}
					}

				}
			};

			if (SwingUtilities.isEventDispatchThread())
				shower.run();
			else
				SwingUtilities.invokeAndWait(shower);

		}
		catch (Throwable e2) {
			e2.printStackTrace(System.err);
		}

	}

	public static void main(String[] args) throws IOException {
		final UncaughtExceptionHandler handler = new SwingUncaughtExceptionHandler();

		for (int i = 0; i < 10; i += 1) {
			final int k = i;
			new Thread() {
				@Override
				public void run() {
					try {
						Thread.sleep(k * 1000);
					}
					catch (InterruptedException e) {
						e.printStackTrace();
					}
					handler.uncaughtException(Thread.currentThread(),
					  new RuntimeException("Deu ruim " + k + "!!", new RuntimeException("Deu ruim " + k + "!!",
					    new RuntimeException("Deu ruim " + k + "!!", new NullPointerException()))));
				}
			}.start();
		}
		JOptionPane.showMessageDialog(null, "Oi!");
	}

}
