package tecgraf.openbus.browser;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.reflect.Field;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextPane;

import net.miginfocom.swing.MigLayout;

/**
 * Item do painel de exibio detalhada de excees para o usurio (
 * {@link SwingUncaughtExceptionHandler}).
 * 
 * Cada exceo da cadeia, ou acumuladas por execues concorrentes, sero
 * representadas por uma instncia deste item.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
@SuppressWarnings("serial")
public class SwingUncaughtExceptionHandlerItem extends JPanel {

	private final JScrollPane scrStackTrace;
	private final JTextPane txtExceptionClass;
	private final JTextPane txtMessage;
	private final JButton btnShowStack;
	private final JTextArea txtStackTrace;
	private final JLabel lblCausedBy;

	/**
	 * Construir um componente que exibe os detalhes da exceo.
	 * 
	 * @param e Qual exceo ser exibida. Deste objeto sero extrados os
	 *          parmetros.
	 * @param t Em qual thread a exceo ocorreu. O nome da thread aparecer na
	 *          interface.
	 */
	public SwingUncaughtExceptionHandlerItem(Throwable e, Thread t) {
		setLayout(new MigLayout("", "0[grow,fill]0", "0[]0[]0[16px::200px,grow,fill]5[top]0"));
		setFocusable(false);

		txtExceptionClass = new JTextPane();
		txtExceptionClass.setFont(new Font("Dialog", Font.BOLD, 15));
		txtExceptionClass.setMinimumSize(new Dimension(0, 20));
		txtExceptionClass.setEditable(false);
		txtExceptionClass.setOpaque(false);
		add(txtExceptionClass, "cell 0 0,grow");

		txtMessage = new JTextPane();
		txtMessage.setFont(new Font("Dialog", Font.PLAIN, 11));
		txtMessage.setMinimumSize(new Dimension(0, 20));
		txtMessage.setEditable(false);
		txtMessage.setOpaque(false);
		add(txtMessage, "cell 0 1,grow, hidemode 3");

		btnShowStack = new JButton("...");
		btnShowStack.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				scrStackTrace.setVisible(!scrStackTrace.isVisible());
				Component x = scrStackTrace.getParent();
				while (x != null) {
					x.validate();
					x = x.getParent();
				}
			}
		});
		btnShowStack.setMinimumSize(new Dimension(16, 16));
		btnShowStack.setMaximumSize(new Dimension(16, 16));
		btnShowStack.setPreferredSize(new Dimension(16, 16));
		btnShowStack.setFont(new Font("Dialog", Font.PLAIN, 10));
		add(btnShowStack, "flowx,cell 0 2,top");

		scrStackTrace = new JScrollPane();
		scrStackTrace.setBorder(null);
		scrStackTrace.setOpaque(false);
		scrStackTrace.setMinimumSize(new Dimension(0, 0));
		scrStackTrace.setVisible(false);
		add(scrStackTrace, "cell 0 2, hidemode 3, shrink");

		txtStackTrace = new JTextArea();
		txtStackTrace.setFont(new Font(Font.MONOSPACED, Font.PLAIN, 10));
		txtStackTrace.setEditable(false);
		txtStackTrace.setMinimumSize(new Dimension(0, 0));
		txtStackTrace.setOpaque(false);
		scrStackTrace.setViewportView(txtStackTrace);

		lblCausedBy = new JLabel(">>");
		lblCausedBy.setMaximumSize(new Dimension(20, 20));
		lblCausedBy.setMinimumSize(new Dimension(20, 20));
		lblCausedBy.setFont(new Font("Dialog", Font.PLAIN, 10));
		add(lblCausedBy, "cell 0 3");

		if (e == null)
			e = new NullPointerException("Testando 1,2,3...");

		txtExceptionClass.setText(e.getClass().getName());

		if (e.getMessage() == null || e.getMessage().trim().isEmpty())
			txtMessage.setVisible(false);
		else
			txtMessage.setText(e.getMessage());

		StringBuilder st = new StringBuilder();
		if (t != null)
			st.append("Thread " + t.getName() + " (" + t.getId() + ")");
		for (Field f : e.getClass().getFields()) {
			try {
				if (st.length() > 1)
					st.append("\n");
				st.append("* ");
				st.append(f.getName());
				st.append("=");
				Object val = f.get(e);
				if (val != null)
					st.append(val);
				else
					st.append("null");
			}
			catch (Throwable x) {
			}
		}

		for (StackTraceElement ste : e.getStackTrace()) {
			if (st.length() > 1)
				st.append("\n");
			st.append("em ");
			st.append(ste.getClassName());
			st.append(".");
			st.append(ste.getMethodName());
			if (ste.getFileName() != null && !ste.getFileName().isEmpty()) {
				st.append("(" + ste.getFileName());
				if (ste.getLineNumber() >= 0)
					st.append(":" + ste.getLineNumber());
				st.append(")");
			}
		}
		txtStackTrace.setText(st.toString());

		if (e.getCause() != null) {
			add(new SwingUncaughtExceptionHandlerItem(e.getCause(), null), "cell 0 3");
		}
		else {
			lblCausedBy.setVisible(false);
		}

	}

}
