package tecgraf.openbus.browser;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Rectangle;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Arrays;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextPane;

import net.miginfocom.swing.MigLayout;

/**
 * Implementao do painel de "dica do dia" que aparece tipicamente no rodap da
 * aplicao, podendo ser ocultado pelo usurio.
 * 
 * Se o usurio clicar na caixa de texto com a dica e digitar "troll", ele
 * habilita as dicas "troll", um "easter egg" desta aplicao sem utilidade
 * prtica.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
@SuppressWarnings("serial")
public class TipPanel extends JPanel implements TipPanelInterface {

	private static final ImageIcon IMAGE_TIP = new ImageIcon(TipPanel.class.getResource("tip_32.png"));
	private static final ImageIcon IMAGE_TROLL = new ImageIcon(TipPanel.class.getResource("troll.png"));
	private static final ImageIcon IMAGE_LEFT = new ImageIcon(TipPanel.class.getResource("tip-left-16.png"));
	private static final ImageIcon IMAGE_RIGHT = new ImageIcon(TipPanel.class.getResource("tip-right-16.png"));

	private String[] tips;
	private String[] trolls;
	private boolean trollsEnabled = false;
	private int currentTip = 0;

	private final JTextPane tipArea;
	private final JLabel lblIcon;
	private final JLabel lblLeft;
	private final JLabel lblRight;

	public TipPanel() {

		setMaximumSize(new Dimension(Integer.MAX_VALUE, 55));
		setMinimumSize(new Dimension(0, 55));
		//setPreferredSize(new Dimension(Integer.MAX_VALUE, 55));

		setLayout(new MigLayout("", "0[32px:32px:32px,center][grow]0", "0[32px:32px:32px,center][grow,center]0"));

		lblIcon = new JLabel(IMAGE_TIP);

		lblIcon.setMinimumSize(new Dimension(32, 32));
		lblIcon.setMaximumSize(new Dimension(32, 32));
		add(lblIcon, "flowy,cell 0 0");

		final JScrollPane scrollPane = new JScrollPane();
		add(scrollPane, "cell 1 0 1 2,grow");

		tipArea = new JTextPane();
		tipArea.setFont(new Font(Font.MONOSPACED, Font.PLAIN, 10));
		tipArea.setMinimumSize(new Dimension(0, 0));
		tipArea.setEditable(false);
		tipArea.addKeyListener(new KeyAdapter() {
			ArrayList<Character> keyQueue = new ArrayList<Character>(6);

			@Override
			public void keyPressed(KeyEvent e) {
				if (e.getKeyChar() > 'a' && e.getKeyChar() < 'z') {
					keyQueue.add(e.getKeyChar());
					if (keyQueue.size() > 5)
						keyQueue.remove(0);
					if (Arrays.equals(keyQueue.toArray(new Character[] {}), new Character[] { 't', 'r', 'o', 'l', 'l' })) {
						getToolkit().beep();
						trollsEnabled = true;
						if (getNumTips() > 1) {
							int newTip;
							do {
								newTip = (int) (System.currentTimeMillis() % getNumTips());
							} while (newTip == currentTip);
						}
						showCurrentTip();
					}
				}
			}
		});
		scrollPane.setViewportView(tipArea);

		lblLeft = getArrowLabel(IMAGE_LEFT);
		lblLeft.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseClicked(MouseEvent e) {
				if (!lblLeft.isEnabled())
					return;
				currentTip = (currentTip - 1);
				if (currentTip == -1)
					currentTip = getNumTips() - 1;
				showCurrentTip();
				e.consume();
			}
		});
		add(lblLeft, "flowx,cell 0 1, center, gap 0");

		lblRight = getArrowLabel(IMAGE_RIGHT);
		lblRight.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseClicked(MouseEvent e) {
				if (!lblRight.isEnabled())
					return;
				currentTip = (currentTip + 1) % getNumTips();
				showCurrentTip();
				e.consume();
			}
		});
		add(lblRight, "cell 0 1, center, gap 0");

		setTips(null, null);
	}

	@Override
	public void setTips(String tips[], String trolls[]) {
		if (tips == null)
			tips = new String[] {};
		if (trolls == null)
			trolls = new String[] {};
		if (Arrays.equals(this.tips, tips) && Arrays.equals(this.trolls, trolls))
			return;

		this.tips = tips;
		this.trolls = trolls;
		trollsEnabled = tips == null || tips.length == 0 || (System.currentTimeMillis() % 50) < 3;
		if (getNumTips() > 1)
			this.currentTip = (int) (System.currentTimeMillis() % getNumTips());
		else
			this.currentTip = 0;
		lblLeft.setEnabled(getNumTips() > 1);
		lblRight.setEnabled(getNumTips() > 1);
		showCurrentTip();
	}

	/**
	 * @return Contagem de dicas, de todos os tipos.
	 */
	private final int getNumTips() {
		int nTips = (tips != null ? tips.length : 0);
		int nTrolls = (trolls != null ? trolls.length : 0);
		if (trollsEnabled)
			return nTips + nTrolls;
		else
			return nTips;
	}

	/**
	 * Recuperar dica de ndica i, considerando um array virtual onde as dicas de
	 * {@link #tips} so concatenadas com as de {@link #trolls}.
	 * 
	 * @param i ndice da dica.
	 * @return Dica propriamente dita.
	 */
	private final String getTip(int i) {
		if (tips != null && i < tips.length)
			return tips[i];
		else if (trolls != null && trolls.length > 0)
			return trolls[i - (tips != null ? tips.length : 0)];

		return "";
	}

	/**
	 * Perguntar se o ndice i, recupervel por {@link #getTip(int)}, recupera de
	 * {@link #tips} ou de {@link #trolls}.
	 * 
	 * @param i ndice sobre o qual se deseja saber.
	 * @return true se for do {@link #trolls}.
	 */
	private final boolean isTroll(int i) {
		return tips == null || i >= tips.length;
	}

	/**
	 * Configura os componentes visuais do componente para que a dica atual seja
	 * exibida. A dica "atual"  a cujo ndice  retornado por {@link #currentTip}
	 * .
	 */
	private void showCurrentTip() {
		String tip = getTip(currentTip);
		if (tip == null)
			tip = "";
		tipArea.setText(tip);
		if (!tip.equals("") && isTroll(currentTip) && lblIcon.getIcon() != IMAGE_TROLL) {
			lblIcon.setIcon(IMAGE_TROLL);
		}
		else if (lblIcon.getIcon() != IMAGE_TIP) {
			lblIcon.setIcon(IMAGE_TIP);
		}
		tipArea.scrollRectToVisible(new Rectangle(0, 0, 1, 1));
	}

	/**
	 * Convenincia para montar o label com uma seta (dir ou esq), que servir
	 * para o usurio navegar pelas dicas manualmente.  inserido o comportamento
	 * do "mouseOver" para realar o componente quando o mouse passar sobre ele,
	 * indicando ao usurio que se pode clicar ali.
	 * 
	 * @param icon Qual cone do boto, que deve ter 16x16 pixels.
	 * @return JLabel configurado para se adequar ao layout.
	 */
	private static final JLabel getArrowLabel(ImageIcon icon) {
		final JLabel res = new JLabel(icon);
		res.setMinimumSize(new Dimension(16, 16));
		res.setMaximumSize(new Dimension(16, 16));
		res.setFont(new Font(Font.MONOSPACED, Font.PLAIN, 9));
		res.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseEntered(MouseEvent e) {
				if (res.isEnabled()) {
					res.setBorder(BorderFactory.createLineBorder(Color.blue));
				}
			}

			@Override
			public void mouseExited(MouseEvent e) {
				if (res.isEnabled()) {
					res.setBorder(null);
				}
			}
		});

		return res;
	}

}
