package tecgraf.openbus.browser.scs_offers;

import java.util.Enumeration;
import java.util.WeakHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;

import javax.swing.SwingUtilities;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;

import scs.core.ComponentId;
import scs.core.FacetDescription;
import scs.core.IMetaInterface;
import scs.core.IMetaInterfaceHelper;
import scs.core.IReceptacles;
import scs.core.IReceptaclesHelper;
import scs.core.ReceptacleDescription;
import tecgraf.openbus.Connection;
import tecgraf.openbus.browser.ManagedConnection;
import tecgraf.openbus.browser.scs_offers.basic_nodes.ComponentNodeBean;
import tecgraf.openbus.browser.scs_offers.basic_nodes.ComponentVersionNodeBean;
import tecgraf.openbus.browser.scs_offers.basic_nodes.ErrorNodeBean;
import tecgraf.openbus.browser.scs_offers.basic_nodes.FacetNodeBean;
import tecgraf.openbus.browser.scs_offers.basic_nodes.ReceptacleNodeBean;
import tecgraf.openbus.browser.scs_offers.basic_nodes.ServiceOfferNode;
import tecgraf.openbus.core.v2_0.services.offer_registry.OfferRegistry;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOfferDesc;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceProperty;

/**
 * Algoritmos para a montagem da estrutura da rvore que pode ser vista quando
 * se faz uma consulta ao servio de ofertas.
 * 
 * @author Daltro Gama (daltro@tecgraf.puc-rio.br)
 */
final class TreeStructureBuilder {

	private static final String INVALID_OFFERS_LABEL = "[OFERTAS INVLIDAS]";

	/**
	 * Interface de callback para que o processo assncrono de montagem da
	 * rvore notifique sobre o progresso. Graas a este callback, o status
	 * da busca no servio de ofertas (ou notificao de observadores remotos)
	 *  informado graficamente na tela, em tempo real.
	 */
	protected interface TreeLoadStatusListener {
		/**
		 * Indica que uma consulta ao servio de ofertas foi iniciada.
		 */
		public void startedQuery();

		/**
		 * Indica que j se o "find" j foi executado e, assim, j se sabe exatamente
		 * quantas ofertas devero ser processadas.
		 * @param total Total de ofertas identificadas.
		 */
		public void startedProcessing(int total);

		/**
		 * Indica que ofertas foram processadas, e o nmero de ofertas devidamente
		 * resolvidas foi atualizado.
		 * @param progress Nmero de ofertas processadas.
		 */
		public void setProgress(int progress);

		/**
		 * Notifica que houve um erro durante a avaliao das ofertas encontradas.
		 * @param error Qual foi o erro.
		 */
		public void error(Throwable error);

		/**
		 * Indica que uma nova oferta foi descoberta por fora do processo de consulta
		 * padro, ou seja, atravs do notificador remoto do servio de ofertas.
		 */
		public void outOfQueryNewOffer();

		/**
		 * Indica que uma oferta foi removida do servio de ofertas, sendo esta
		 * descoberta feita pelo notificador remoto do servio de ofertas.
		 */
		public void outOfQueryRemovedOffer();
	}

	private final ExecutorService threadPool = Executors.newFixedThreadPool(80);
	
	/**
	 * Instncia do callback usado para que o componente da rvore interfira
	 * programaticamente na mesma interface visual que o usurio utiliza para
	 * definir os critrios da busca.
	 */
	private final FindServicesQueryControlInterface queryControl;
	
	private final SCSTree tree;
	private final TreeLoadStatusListener status;
	private final AtomicInteger serialQuery = new AtomicInteger(0);
	private final WeakHashMap<ServiceOfferDesc, IReceptacles> receptaclesCache =
	  new WeakHashMap<ServiceOfferDesc, IReceptacles>();

	/**
	 * Criar instncia do algoritmo, vinculada a um componente de rvore e ao provedor
	 * do filtro da consulta.
	 * @param queryControl 
	 * @param treeComponents
	 * @param status
	 */
	protected TreeStructureBuilder(FindServicesQueryControlInterface queryControl, SCSTree treeComponents,
	  TreeLoadStatusListener status) {
		super();
		this.tree = treeComponents;
		this.status = status;
		this.queryControl = queryControl;
	}

	/**
	 * Iniciar o processo da busca no servio de ofertas propriamente dito.
	 * 
	 * <p>
	 * Este mtodo no  bloqueante. O processamento da busca  inteiramente
	 * feito em threads separadas, e as notificaes de atualizao do status
	 * da busca sero feitos atravs do callback {@link TreeLoadStatusListener}
	 * informado.
	 * 
	 * @param cnn Conexo Openbus a ser usada para realizar a busca e todas as
	 *                    chamadas remotas envolvidas.
	 * @param properties Filtro a ser usado (vide {@link OfferRegistry#findServices(ServiceProperty[])}).
	 */
	protected void startLoadingTree(final Connection cnn, final ServiceProperty properties[]) {
		new Thread() {
			@Override
			public void run() {
				status.startedQuery();
				try {
					ManagedConnection.setContextCurrentConnection(cnn);
					OfferRegistry registryService = ManagedConnection.getContext(cnn).getOfferRegistry();

					ServiceOfferDesc[] services;
					if (properties == null || properties.length == 0)
						services = registryService.getAllServices();
					else
						services = registryService.findServices(properties);

					status.startedProcessing(services.length);
					AtomicInteger queryCounter = new AtomicInteger(0);
					NodeWithTreeReference root = (NodeWithTreeReference)
					  tree.getModel().getRoot();
					int serial = serialQuery.incrementAndGet();
					for (ServiceOfferDesc service : services) {
						processServiceOffer(cnn, serial, service, queryCounter, root);
					}

				}
				catch (Throwable e) {
					status.error(e);
				}
			}
		}.start();
	};

	/**
	 * Adicionar na rvore de resultados uma dada oferta, individualmente.
	 * 
	 * @param cnn Conexo Openbus a ser usada para realizar as chamadas remotas
	 *            relacionadas.
	 * @param service Descrio da oferta de servio a ser adicionada.
	 * @param root Instncia do n raz da rvore, para garantir que se est operando
	 *             na rvore correta.
	 */
	public void addServiceOffer(final Connection cnn, final ServiceOfferDesc service,
	  final NodeWithTreeReference root) {
		processServiceOffer(cnn, serialQuery.get(), service, null, root);
	}

	private void processServiceOffer(final Connection cnn, final int serial, final ServiceOfferDesc service,
	  final AtomicInteger queryCounter, final NodeWithTreeReference root) {
		threadPool.execute(new Runnable() {
			@Override
			public void run() {

				try {

					ManagedConnection.setContextCurrentConnection(cnn);

					Throwable error = null;
					ServiceOfferManagedBean offerBean = null;
					try {
						offerBean = ServiceOfferRegistryObserversPool.getSingleton().addNewServiceOfferBean(cnn, service.ref);
					}
					catch (Throwable e) {
						error = e;
					}

					ComponentId componentId = service.service_ref.getComponentId();

					Object offerObserverBean;
					if (offerBean != null)
						offerObserverBean = offerBean;
					else
						offerObserverBean = error;

					org.omg.CORBA.Object metaInterfaceObject = service.service_ref.getFacetByName("IMetaInterface");
					IMetaInterface metaInterface = IMetaInterfaceHelper.narrow(metaInterfaceObject);

					{
						FacetDescription[] facets = null;
						if (metaInterface != null)
							facets = metaInterface.getFacets();

						if (facets != null) {
							if (serial != serialQuery.get())
								return;
							for (FacetDescription facet : facets) {
								processFacet(cnn, serial, service, offerObserverBean, componentId, facet, root);
							}
						}
					}

					{
						ReceptacleDescription[] receptacles = null;
						if (metaInterface != null)
							receptacles = metaInterface.getReceptacles();

						if (receptacles != null) {
							if (serial != serialQuery.get())
								return;
							for (ReceptacleDescription receptacle : receptacles) {
								processFacet(cnn, serial, service, offerObserverBean, componentId, receptacle, root);
							}
						}
					}
				}
				catch (Throwable e) {
					if (serial != serialQuery.get())
						return;

					processInvalidOffer(cnn, serial, service, e, root);
				}
				finally {
					if (serial != serialQuery.get())
						return;

					if (queryCounter != null) {
						int counter = queryCounter.incrementAndGet();
						status.setProgress(counter);
					}
					else
						status.outOfQueryNewOffer();
				}
			}
		});
	}

	private void processInvalidOffer(final Connection cnn, final int serial, final ServiceOfferDesc service,
	  final Throwable e, final NodeWithTreeReference root) {
		Runnable task = new Runnable() {
			@Override
			public void run() {

				ManagedConnection.setContextCurrentConnection(cnn);

				if (serial != serialQuery.get())
					return;

				DefaultTreeModel model = (DefaultTreeModel) tree.getModel();

				// Nvel 1: Raz das ofertas invlidas
				NodeWithTreeReference componentNode = findChildNode(root, INVALID_OFFERS_LABEL);
				if (componentNode == null) {
					componentNode = new NodeWithTreeReference(tree, INVALID_OFFERS_LABEL);
					insertNodeSorted(model, componentNode, root);
				}

				// Nvel 3: Oferta (identificada por suas propriedades)
				ServiceOfferNode serviceBean = new ServiceOfferNode(tree, queryControl, cnn, service);
				NodeWithTreeReference serviceNode = findChildNode(componentNode, serviceBean);
				if (serviceNode == null) {
					serviceNode = serviceBean;
					insertNodeSorted(model, serviceNode, componentNode);
				}

				// Nvel 4 em diante: Mensagem de erro
				Throwable error = e;
				NodeWithTreeReference parent = serviceNode;
				while (error != null) {
					NodeWithTreeReference newChild = new NodeWithTreeReference(tree, new ErrorNodeBean(error));
					model.insertNodeInto(newChild, parent, 0);
					parent = newChild;
					error = e.getCause();
				}

				if (queryControl.isOffersDefaultExpanded())
					tree.expandPath(new TreePath(componentNode.getPath()));
				else
					tree.expandPath(new TreePath(root.getPath()));

			}
		};
		if (SwingUtilities.isEventDispatchThread())
			task.run();
		else
			SwingUtilities.invokeLater(task);
	}

	private void processFacet(final Connection cnn, final int serial, final ServiceOfferDesc service,
	  final Object offerObserverBeanOrThrowable, final ComponentId componentId,
	  final Object facetOrReceptacleDescription, final NodeWithTreeReference root) {

		Runnable task = new Runnable() {
			@Override
			public void run() {

				ManagedConnection.setContextCurrentConnection(cnn);

				if (serial != serialQuery.get())
					return;

				DefaultTreeModel model = (DefaultTreeModel) tree.getModel();

				// Nvel 1: Componente
				ComponentNodeBean componentBean = new ComponentNodeBean(componentId);
				NodeWithTreeReference componentNode = findChildNode(root, componentBean);
				if (componentNode == null) {
					componentNode = new NodeWithTreeReference(tree, componentBean);
					insertNodeSorted(model, componentNode, root);
				}

				// Nvel 2: Verso do Componente
				ComponentVersionNodeBean versionBean = new ComponentVersionNodeBean(componentId, service.service_ref);
				NodeWithTreeReference versionNode = findChildNode(componentNode, versionBean);
				if (versionNode == null) {
					versionNode = new NodeWithTreeReference(tree, versionBean, true);
					insertNodeSorted(model, versionNode, componentNode);
				}

				// Nvel 3: Oferta (identificada por suas propriedades)
				ServiceOfferNode serviceBean = new ServiceOfferNode(tree, queryControl, cnn, service);
				NodeWithTreeReference serviceNode = findChildNode(versionNode, serviceBean);
				if (serviceNode == null) {

					if (offerObserverBeanOrThrowable != null && offerObserverBeanOrThrowable instanceof ServiceOfferManagedBean)
						serviceBean.setOfferPoolBean((ServiceOfferManagedBean) offerObserverBeanOrThrowable);

					serviceNode = serviceBean;
					insertNodeSorted(model, serviceNode, versionNode);

					if (offerObserverBeanOrThrowable != null && offerObserverBeanOrThrowable instanceof Throwable) {
						serviceNode
						  .add(new NodeWithTreeReference(tree, new ErrorNodeBean((Throwable) offerObserverBeanOrThrowable)));
					}
				}

				// Nvel 4: Receptculo ou Faceta
				Object facetBean;
				if (facetOrReceptacleDescription instanceof FacetDescription) {
					FacetDescription facetDescription = (FacetDescription) facetOrReceptacleDescription;
					facetBean = new FacetNodeBean(cnn, facetDescription);
				}
				else {
					IReceptacles iReceptacles = getIReceptacle(service);
					ReceptacleDescription receptacleDescription = (ReceptacleDescription) facetOrReceptacleDescription;
					facetBean = new ReceptacleNodeBean(cnn, service, receptacleDescription, iReceptacles);
				}
				NodeWithTreeReference facetNode = findChildNode(serviceNode, facetBean);
				if (facetNode == null) {
					facetNode = FacetNodeBeanFactory.getBeanNode(tree, cnn, facetBean);
					facetNode.setUserObject(facetBean);
					insertNodeSorted(model, facetNode, serviceNode);
				}
				
				if (queryControl.isOffersDefaultExpanded())
					tree.expandPath(new TreePath(versionNode.getPath()));
				else
					tree.expandPath(new TreePath(root.getPath()));
				
			}
		};
		if (SwingUtilities.isEventDispatchThread())
			task.run();
		else
			SwingUtilities.invokeLater(task);
	}

	/**
	 * Remover uma dada oferta de servio da rvore de resultados, de forma
	 * individual.
	 * 
	 * <p>
	 * Este mtodo ir remover os ns pai da rvore sempre que o n removido
	 * for o ltimo filho, mantendo o aspecto esttico da rvore de resultados.
	 * 
	 * @param offerNode N que representa a oferta propriemente dita.
	 */
	public void removeOffer(final NodeWithTreeReference offerNode) {
		Runnable r = new Runnable() {
			@Override
			public void run() {
				NodeWithTreeReference rootNode = (NodeWithTreeReference) offerNode.getRoot();
				if (rootNode == null)
					return;

				SCSTree tree = offerNode.getMyTree();
				if (tree.getModel().getRoot() != rootNode)
					return;

				DefaultTreeModel model = (DefaultTreeModel) tree.getModel();
				NodeWithTreeReference parent = (NodeWithTreeReference) offerNode.getParent();
				NodeWithTreeReference removedParent = parent;
				if (parent != null) {
					
					int idxRemovedFromParent = parent.getIndex(offerNode);
					parent.remove(offerNode);
					model.nodesWereRemoved(parent
					  , new int[] { idxRemovedFromParent }
					  , new Object[] { offerNode });
					
					while (parent != null) {
						NodeWithTreeReference grandParent = (NodeWithTreeReference) parent.getParent();
						if (grandParent != null && parent.getChildCount() == 0) {
							removedParent = grandParent;
							idxRemovedFromParent = removedParent.getIndex(parent);
							grandParent.remove(parent);
							model.nodesWereRemoved(grandParent
							  , new int[] { idxRemovedFromParent }
							  , new Object[] { parent });
							parent = grandParent;
						}
						else {
							break;
						}
					}
				}
			}
		};
		if (SwingUtilities.isEventDispatchThread())
			r.run();
		else
			SwingUtilities.invokeLater(r);

		if (offerNode instanceof ServiceOfferNode) {
			status.outOfQueryRemovedOffer();
		}

	}

	private IReceptacles getIReceptacle(ServiceOfferDesc service) {
		synchronized (receptaclesCache) {
			IReceptacles res = receptaclesCache.get(service);
			if (res != null)
				return res;
			res = IReceptaclesHelper.narrow(service.service_ref.getFacet(IReceptaclesHelper.id()));
			receptaclesCache.put(service, res);
			return res;
		}
	}

	private final void insertNodeSorted(DefaultTreeModel model, NodeWithTreeReference newChild,
	  NodeWithTreeReference parent) {
		int pos = 0;
		final String myLbl = newChild.getUserObject().toString();

		if (!myLbl.equals(INVALID_OFFERS_LABEL)) {
			final Enumeration<?> enumChild = parent.children();
			while (enumChild.hasMoreElements()) {
				final NodeWithTreeReference node = (NodeWithTreeReference) enumChild.nextElement();
				final String otherLbl = node.getUserObject().toString();
				if (myLbl.compareTo(otherLbl) < 0 && !otherLbl.equals(INVALID_OFFERS_LABEL))
					break;
				pos += 1;
			}
		}
		model.insertNodeInto(newChild, parent, pos);
	}

	/**
	 * Procura no <param>parentNode</param> por um node com o texto passado
	 * 
	 * @param parentNode Node cujos filhos sero escopo da pesquisa
	 * @param textNode Nome do node procurado
	 * @return <code>NodeWithTreeReference</code> correspondente, caso contrrio,
	 *         retornar NULL
	 */
	private NodeWithTreeReference findChildNode(
	  NodeWithTreeReference parentNode, Object textNode) {
		int childCount = parentNode.getChildCount();
		for (int i = 0; i < childCount; i++) {
			NodeWithTreeReference node =
			  (NodeWithTreeReference) parentNode.getChildAt(i);
			if (node.getUserObject().equals(textNode)) {
				return node;
			}
		}
		return null;
	}

}
