/*
 * $Id$
 */

package csbase.client.algorithms;

import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.swing.ComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.ScrollPaneConstants;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.HistoryProvider;
import csbase.logic.algorithms.HistoryRecord;

/**
 * Painel de seleo de algoritmo e sua respectiva verso.
 */
public final class AlgorithmChooserPanel extends JPanel {
  /**
   * Combo de algortimos
   */
  private final JComboBox algorithmComboBox;
  /**
   * Set de AlgorithmInfo
   */
  private SortedSet<AlgorithmInfo> algorithmInfoSet;

  /**
   * Label de algoritmos
   */
  private final JLabel algorithmLabel;

  /**
   * rea de texto de descrio
   */
  private JTextArea descriptionTextArea;
  /**
   * Ao de abertura de help
   */
  private HelpAlgorithmVersionAction helpVersionAction;
  /**
   * Boto de help
   */
  private JButton helpVersionButton;
  /**
   * Lista de listeners de escolha
   */
  private final List<AlgorithmChooserListener> listenerList;

  /**
   * Ao de exibio de histrico
   */
  private ShowHistoryAction showHistoryAction;

  /**
   * Boto de histrico
   */
  private JButton showHistoryButton;

  /**
   * Combo de verso de algoritmo
   */
  private final JComboBox versionComboBox;

  /**
   * Label de verso de algoritmo.
   */
  private final JLabel versionLabel;

  /**
   * Cria uma painel de seleo de algoritmos. O painel exibe uma lista de
   * algoritmos e suas verses. Tambm exibe a descrio e, opcionalmente, o
   * histrico.
   * 
   * @param window a janela.
   * @param showHistory flag que indica se deve ou no exibir o boto que
   *        permite que o usurio exiba o histrico.
   */
  public AlgorithmChooserPanel(final Window window, final boolean showHistory) {
    this(window, showHistory, true);
  }

  /**
   * Cria uma painel de seleo de algoritmos. O painel exibe uma lista de
   * algoritmos e suas verses. Tambm exibe a descrio e, opcionalmente, o
   * histrico.
   * 
   * @param window a janela.
   * @param showHistory flag que indica se deve ou no exibir o boto que
   *        permite que o usurio exiba o histrico.
   * @param showDetails flag que indica se deve ou no exibir o boto de help e
   *        a descrio do algoritmo.
   */
  public AlgorithmChooserPanel(final Window window, final boolean showHistory,
    final boolean showDetails) {
    this.listenerList = new LinkedList<AlgorithmChooserListener>();
    this.algorithmInfoSet = new TreeSet<AlgorithmInfo>();
    this.algorithmLabel = new JLabel(getString("algorithmLabel"));
    this.algorithmComboBox = makeAlgorithmComboBox();
    this.versionLabel = new JLabel(getString("versionLabel"));
    this.versionComboBox = makeAlgorithmVersionComboBox();
    if (showDetails) {
      this.descriptionTextArea = makeDescriptionTextArea();
      this.helpVersionAction = new HelpAlgorithmVersionAction(window);
      this.helpVersionAction.hideName();
      this.helpVersionButton = new JButton(this.helpVersionAction);
      GUIUtils.trimImageButton(this.helpVersionButton);
    }
    if (showHistory) {
      this.showHistoryAction = new ShowHistoryAction(window);
      this.showHistoryButton = new JButton(this.showHistoryAction);
      this.showHistoryButton.setText("");
      this.showHistoryButton.setToolTipText(getString("showHistoryButton"));
      GUIUtils.trimImageButton(this.showHistoryButton);
    }
    populatePanel();
  }

  /**
   * Adiciona um observador a lista de observadores deste painel. O observador 
   * notificado sempre que um novo algoritmo ou uma nova verso forem
   * selecionados. A ordem de insero dos observadores ser respeitada, logo, o
   * primeiro observador inserido ser o primeiro a receber eventos.
   * 
   * @param listener o observador (no pode ser nulo).
   * 
   * @throws IllegalArgumentException se o listener for nulo.
   */
  public void addAlgorithmChooserListener(
    final AlgorithmChooserListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listenerList.add(listener);
  }

  /**
   * Obtm o algoritmo de est selecionado.
   * 
   * @return o algoritmo selecionado ou null caso no haja nenhum.
   */
  public AlgorithmInfo getSelectedAlgorithmInfo() {
    return (AlgorithmInfo) this.algorithmComboBox.getSelectedItem();
  }

  /**
   * Obtm a verso do algoritmo que est selecionada.
   * 
   * @return a verso selecionada ou null caso no haja nenhum.
   */
  public AlgorithmVersionInfo getSelectedVersionInfo() {
    return (AlgorithmVersionInfo) versionComboBox.getSelectedItem();
  }

  /**
   * Verifica se existem algoritmos disponveis.
   * 
   * @return <code>true</code> se existe ou <code>false</code> caso contrrio.
   */
  public boolean hasAlgorithms() {
    return algorithmInfoSet.size() != 0;
  }

  /**
   * Verifica se existem algoritmos disponveis.
   * 
   * @return <code>true</code> se existe ou <code>false</code> caso contrrio.
   */
  public boolean hasVersions() {
    final ComboBoxModel model = versionComboBox.getModel();
    return model.getSize() != 0;
  }

  /**
   * Seleciona um algoritmo.
   * 
   * @param algorithmName o nome do algoritmo a ser selecionado
   * 
   * @return {@code true} em caso de sucesso ou {@code false} se no houver o
   *         algoritmo.
   */
  public boolean selectAlgorithm(final String algorithmName) {
    for (final AlgorithmInfo algorithmInfo : algorithmInfoSet) {
      final String algName = algorithmInfo.getName();
      if (algName.equals(algorithmName)) {
        algorithmComboBox.setSelectedItem(algorithmInfo);
        return true;
      }
    }
    return false;
  }

  /**
   * Seleciona a verso do algoritmo.
   * 
   * @param versionId o identificador da verso.
   * 
   * @return {@code true} em caso de sucesso ou {@code false} se no houver o
   *         algoritmo ou a verso.
   */
  public boolean selectVersion(final AlgorithmVersionId versionId) {
    final AlgorithmInfo algorithmInfo = getSelectedAlgorithmInfo();
    if (algorithmInfo == null) {
      return false;
    }
    final AlgorithmVersionInfo versionInfo =
      algorithmInfo.getVersionInfo(versionId);
    if (versionInfo == null) {
      return false;
    }
    versionComboBox.setSelectedItem(versionInfo);
    return true;
  }

  /**
   * Estabelece um conjunto de algoritmos que sero exibidos na lista de
   * seleo.
   * 
   * @param algorithmInfoSet um conjunto de <code>AlgorithmInfo</code>
   */
  public void setAlgorithmSet(final Set<AlgorithmInfo> algorithmInfoSet) {
    this.algorithmInfoSet = new TreeSet<AlgorithmInfo>(algorithmInfoSet);
    loadAlgorithms();
  }

  /**
   * Estabelece um conjunto de verses que devem ser exibidas na lista de
   * seleo de verses de um algoritmo.
   * 
   * @param algorithmVersionSet um conjunto de <code>AlgorithmVersionInfo</code>
   */
  public void setAlgorithmVersion(
    final Set<AlgorithmVersionInfo> algorithmVersionSet) {
    loadVersions(new TreeSet<AlgorithmVersionInfo>(algorithmVersionSet));
  }

  /**
   * Atribui uma descrio para ser exibida na rea de texto correspondente.
   * 
   * @param description o texto da descrio
   */
  public void setDescription(final String description) {
    if (descriptionTextArea != null) {
      descriptionTextArea.setText(description);
      descriptionTextArea.setCaretPosition(0);
      descriptionTextArea.setEnabled(isEnabled());
    }
  }

  /**
   * Notifica aos listeners que um novo algoritmo foi selecionado.
   */
  private void fireChangedAlgorithm() {
    final Iterator<AlgorithmChooserListener> listenerIterator =
      this.listenerList.iterator();
    while (listenerIterator.hasNext()) {
      final AlgorithmChooserListener listener = listenerIterator.next();
      listener.wasChangedAlgorithm(new AlgorithmChooserEvent(this));
    }
  }

  /**
   * Notifica aos listeners que uma nova verso foi selecionada.
   */
  private void fireChangedVersion() {
    final Iterator<AlgorithmChooserListener> listenerIterator =
      this.listenerList.iterator();
    while (listenerIterator.hasNext()) {
      final AlgorithmChooserListener listener = listenerIterator.next();
      listener.wasChangedVersion(new AlgorithmChooserEvent(this));
    }
  }

  /**
   * Internacionalizao pela chave
   * 
   * @param key a chave
   * @return a chave traduzida
   */
  private String getString(final String key) {
    final String className = AlgorithmChooserPanel.class.getName();
    return LNG.get(className + "." + key);
  }

  /**
   * Carrega o combo box dos algoritmos.
   */
  private void loadAlgorithms() {
    algorithmComboBox.removeAllItems();
    for (final AlgorithmInfo algorithm : this.algorithmInfoSet) {
      algorithmComboBox.addItem(algorithm);
    }
  }

  /**
   * Carrega o combo box das verses, de acordo com o algoritmo selecionado.
   */
  private void loadVersions() {
    final AlgorithmInfo info = this.getSelectedAlgorithmInfo();
    final SortedSet<AlgorithmVersionInfo> versions =
      new TreeSet<AlgorithmVersionInfo>(info.getVersions());
    versionComboBox.removeAllItems();
    for (final AlgorithmVersionInfo version : versions) {
      versionComboBox.addItem(version);
    }
  }

  /**
   * Carrega um determinado conjunto de verses que devem ser exibidas na lista
   * de seleo de verses de um algoritmo.
   * 
   * @param versions conjunto de verses a serem exibidas para o algoritmo
   */
  private void loadVersions(final Set<AlgorithmVersionInfo> versions) {
    versionComboBox.removeAllItems();
    for (final AlgorithmVersionInfo version : versions) {
      versionComboBox.addItem(version);
    }
  }

  /**
   * Cria um componente para exibir a lista dos algoritmos disponveis para
   * seleo.
   * 
   * @return o combo box para a lista de algoritmos
   */
  private JComboBox makeAlgorithmComboBox() {
    final JComboBox combo = new JComboBox();
    combo.addItemListener(new AlgorithmComboBoxItemListener());
    return combo;
  }

  /**
   * Cria um componente para exibir a lista das verses do algoritmo
   * selecionado.
   * 
   * @return o combo box para a lista das verses do algoritmo
   */
  private JComboBox makeAlgorithmVersionComboBox() {
    final JComboBox combo = new JComboBox();
    combo.addItemListener(new VersionComboBoxItemListener());
    return combo;
  }

  /**
   * Cria o componente para exibir a descrio da verso do algoritmo.
   * 
   * @return o componente de descrio da verso do algoritmo.
   */
  private JTextArea makeDescriptionTextArea() {
    // Os valores para linhas e colunas visam garantir uma uniformidade das
    // dimenses do componente em todos os pontos do sistema em que este 
    // usado.
    final JTextArea descriptionTxtArea = new JTextArea(3, 35);
    descriptionTxtArea.setLineWrap(true);
    descriptionTxtArea.setWrapStyleWord(true);
    descriptionTxtArea.setBackground(getBackground());
    descriptionTxtArea.setEditable(false);
    descriptionTxtArea.setFont(new Font(Font.MONOSPACED, Font.PLAIN, 12));
    return descriptionTxtArea;
  }

  /**
   * Arruma os componentes no painel.
   */
  private void populatePanel() {
    final JPanel middle = new JPanel(new GridBagLayout());
    middle.add(versionComboBox, new GBC(0, 0).horizontal());
    int px = 1;
    if (showHistoryButton != null) {
      middle.add(showHistoryButton, new GBC(px, 0).none().insets(0, 5, 0, 0));
      px++;
    }
    if (helpVersionButton != null) {
      middle.add(helpVersionButton, new GBC(px, 0).none().insets(0, 5, 0, 0));
      px++;
    }

    final JComponent[][] rows =
      new JComponent[][] { { algorithmLabel, algorithmComboBox },
          { versionLabel, middle } };
    final JPanel header = GUIUtils.createBasicGridPanel(rows);

    setLayout(new GridBagLayout());
    add(header, new GBC(0, 0).horizontal());

    if (descriptionTextArea != null) {
      final JScrollPane sPane = new JScrollPane(descriptionTextArea);
      sPane
        .setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
      BorderUtil.setTitledBorder(sPane, getString("descriptionLabel"));
      add(sPane, new GBC(0, 1).both());
    }
  }

  /**
   * Atualiza a ao ajuda com a verso selecionada.
   */
  private void updateHelpAction() {
    if (helpVersionAction != null) {
      this.helpVersionAction.setAlgorithmVersionInfo(getSelectedVersionInfo());
    }
  }

  /**
   * Atualiza a ao de exibio de histrico com a verso selecionada.
   */
  private void updateShowHistoryAction() {
    if (this.showHistoryAction != null) {
      final AlgorithmVersionInfo vInfo = getSelectedVersionInfo();
      final HistoryProvider provider =
        new AlgorithmVersionHistoryProvider(vInfo);
      showHistoryAction.setHistoryProvider(provider);
    }
  }

  /**
   * Trata eventos de seleo de um algoritmo.  o listener usado no componente
   * de exibio da lista de algoritmos.
   */
  private final class AlgorithmComboBoxItemListener implements ItemListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void itemStateChanged(final ItemEvent e) {
      if (e.getStateChange() == ItemEvent.SELECTED) {
        fireChangedAlgorithm();
        loadVersions();
      }
    }
  }

  /**
   * Objeto provedor de histrico para verses de algoritmos.
   */
  private final class AlgorithmVersionHistoryProvider implements
    HistoryProvider {
    /**
     * Verso da qual pretende-se obter o histrico
     */
    private final AlgorithmVersionInfo versionInfo;

    /**
     * Cria o objeto provedor de histrico.
     * 
     * @param versionInfo verso da qual pretende-se obter o histrico.
     */
    public AlgorithmVersionHistoryProvider(
      final AlgorithmVersionInfo versionInfo) {
      this.versionInfo = versionInfo;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<HistoryRecord> getHistory() {
      final String[] path = getVersionPath();
      return AlgorithmManagementProxy.retrieveHistory(path);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getName() {
      return FileUtils.joinPath('/', getVersionPath());
    }

    /**
     * Obtm o caminho da verso.
     * 
     * @return caminho da verso.
     */
    private String[] getVersionPath() {
      return this.versionInfo.getDirPath().split("/");
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setEnabled(boolean enabled) {
    super.setEnabled(enabled);
    algorithmComboBox.setEnabled(enabled);
    algorithmLabel.setEnabled(enabled);
    versionComboBox.setEnabled(enabled);
    versionLabel.setEnabled(enabled);
    if (descriptionTextArea != null) {
      descriptionTextArea.setEnabled(enabled);
    }
    if (helpVersionButton != null) {
      helpVersionButton.setEnabled(enabled);
    }
    if (showHistoryButton != null) {
      showHistoryButton.setEnabled(enabled);
    }
  }

  /**
   * Trata eventos de seleo de uma verso de algoritmo.  o listener usado no
   * componente de exibio da lista de verses.
   */
  private final class VersionComboBoxItemListener implements ItemListener {
    /**
     * 
     * @see ItemListener#itemStateChanged(ItemEvent)
     */
    @Override
    public void itemStateChanged(final ItemEvent e) {
      if (e.getStateChange() == ItemEvent.SELECTED) {
        updateHelpAction();
        updateShowHistoryAction();
        fireChangedVersion();
      }
    }
  }
}
