/*
 * $Id: HelpAlgorithmVersionAction.java 150777 2014-03-19 14:16:56Z oikawa $
 */
package csbase.client.algorithms;

import java.awt.Window;
import java.awt.event.ActionEvent;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.MessageFormat;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Action;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.pdfviewer.core.PDFDocument;
import tecgraf.javautils.pdfviewer.core.PDFReader;
import tecgraf.javautils.pdfviewer.dialog.PDFDialog;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.externalresources.ExternalResources;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.CSBaseException;
import csbase.logic.FileInfo;
import csbase.logic.RemoteFileInputStream;
import csbase.logic.Utilities;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.remote.AlgorithmServiceInterface;
import csbase.remote.ClientRemoteLocator;

/**
 * Ao utilizada para exibir a documentao de uma verso de algoritmo.
 * 
 * @author Tecgraf/PUC-Rio.
 */
public final class HelpAlgorithmVersionAction extends AbstractAction {
  /**
   * A verso do algoritmo.
   */
  private AlgorithmVersionInfo versionInfo;

  /**
   * A janela.
   */
  private final Window window;

  /**
   * Cria uma ao.
   * 
   * @param window A janela que est requisitando a ao.
   * 
   * @throws IllegalArgumentException Se window for nulo.
   */
  public HelpAlgorithmVersionAction(Window window) {
    if (window == null) {
      throw new IllegalArgumentException("O parmetro window est nulo.");
    }
    this.window = window;
    showName();
    showIcon();
    putValue(Action.SHORT_DESCRIPTION, LNG.get("algorithms.full_description"));
    setEnabled(false);
  }

  /**
   * Cria uma ao.
   * 
   * @param window A janela que est requisitando a ao.
   * @param versionInfo A verso do algoritmo.
   * 
   * @throws IllegalArgumentException Se window ou versionInfo forem nulos.
   */
  public HelpAlgorithmVersionAction(Window window,
    AlgorithmVersionInfo versionInfo) {
    this(window);
    if (versionInfo == null) {
      throw new IllegalArgumentException("O parmetro versionInfo est nulo.");
    }
    setAlgorithmVersionInfo(versionInfo);
  }

  /**
   * 
   * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
   */
  @Override
  public void actionPerformed(ActionEvent ev) {
    if (versionInfo == null) {
      throw new IllegalStateException("No existe verso corrente, ou seja, "
        + "o atributo versionInfo est nulo.\n"
        + "Use setAlgorithmVersionInfo ou o construtor que recebe "
        + "algorithmVersionInfo.");
    }

    final String simplePath;
    final List<FileInfo> documentation = versionInfo.getDocumentation();
    if (documentation.size() == 1) {
      simplePath = documentation.get(0).getPath();
    }
    else {
      simplePath = AlgorithmVersionInfo.HTML_FILE;
    }

    // Testa o caso do resource ser aberto com o visualizador PDF interno.
    if (isResourcePDF(simplePath) && usesPDFViewer()) {
      try {
        final RemoteFileInputStream stream = getStreamForPDF(simplePath);
        if (stream == null) {
          return;
        }

        RemoteTask<PDFDocument> openPdfDocTask = new RemoteTask<PDFDocument>() {
          @Override
          protected void performTask() throws Exception {
            PDFDocument document = new PDFReader().read(stream);
            setResult(document);
          }
        };

        String msgPrefix = HelpAlgorithmVersionAction.class.getSimpleName();
        String taskTitle = LNG.get(msgPrefix + ".task.title");
        String taskDescription = LNG.get(msgPrefix + ".task.message");
        openPdfDocTask.execute(window, taskTitle, taskDescription);
        if (openPdfDocTask.getStatus()) {
          final PDFDialog pdfDialog =
            new PDFDialog(window, simplePath, LNG.getLocale());
          pdfDialog.loadDocument(openPdfDocTask.getResult());
          pdfDialog.setVisible(true);
        }
        else if (openPdfDocTask.getError() != null) {
          showError(openPdfDocTask.getError());
        }

      }
      catch (Exception e) {
        showError(e);
      }
      return;
    }

    // Caso tpico aonde o resource (seja de que tipo) ser enviado
    // para o browser.
    String htmlDirPath =
      versionInfo.getVersionsDirName() + "/" + versionInfo.getDirectory() + "/"
        + versionInfo.getDocumentationDirName();
    final String filePath = htmlDirPath + "/" + simplePath;
    final String[] filePathArray = Utilities.splitProjectPath(filePath);
    final String urlStr =
      AlgorithmManagementProxy.retrieveDownloadURL(this.versionInfo.getInfo()
        .getId(), filePathArray, this.window);

    if (urlStr == null) {
      String msg = LNG.get("algorithms.error.no_help_for_version");
      Object[] args = { this.versionInfo.getInfo(), this.versionInfo };
      msg = MessageFormat.format(msg, args);
      showError(msg);
      return;
    }

    try {
      final URL url = new URL(urlStr);
      final ExternalResources extResources = ExternalResources.getInstance();
      if (extResources.isEnabled()) {
        extResources.showDocument(url);
        return;
      }
    }
    catch (MalformedURLException e) {
      showError(e);
      return;
    }
    catch (CSBaseException e) {
      showError(e);
    }
  }

  /**
   * Analisa se um path parece apontar par aum resourc PDF.
   * 
   * @param filePath path
   * @return indicativo.
   */
  private boolean isResourcePDF(final String filePath) {
    return filePath.endsWith(".pdf") || filePath.endsWith(".PDF");
  }

  /**
   * Busca um stream remoto para o PDF
   * 
   * @param filePath path
   * @return stream
   * @throws Exception
   */
  private RemoteFileInputStream getStreamForPDF(final String filePath)
    throws Exception {
    final AlgorithmServiceInterface service =
      ClientRemoteLocator.algorithmService;
    final Task<RemoteFileInputStream> task = new Task<RemoteFileInputStream>() {
      @Override
      protected void performTask() throws Exception {
        final RemoteFileChannelInfo channel =
          service.prepareDownloadDocFile(versionInfo.getInfo().getId(),
            versionInfo.getId(), filePath);
        final RemoteFileInputStream stream = new RemoteFileInputStream(channel);
        setResult(stream);
      }
    };

    task.execute(window, null, filePath);
    final Exception error = task.getError();
    if (error != null) {
      throw error;
    }

    if (task.wasCancelled()) {
      return null;
    }

    return task.getResult();
  }

  /**
   * Indica se o help deve ser usado no modo PDF interno.
   * 
   * @return indicativo
   */
  private boolean usesPDFViewer() {
    final boolean defaultValue = false;
    final ConfigurationManager confManager = ConfigurationManager.getInstance();
    final Configuration configuration;
    try {
      configuration = confManager.getConfiguration(getClass());
    }
    catch (ConfigurationManagerException e) {
      return defaultValue;
    }
    final String propName = "use.pdf.viewer";
    return configuration.getOptionalBooleanProperty(propName, defaultValue);
  }

  /**
   * Oculta o cone.
   */
  public void hideIcon() {
    putValue(Action.SMALL_ICON, null);
  }

  /**
   * Oculta o nome.
   */
  public void hideName() {
    putValue(Action.NAME, null);
  }

  /**
   * Associa uma verso de algoritmo a esta ao.
   * 
   * @param versionInfo A verso do algoritmo.
   */
  public void setAlgorithmVersionInfo(AlgorithmVersionInfo versionInfo) {
    this.versionInfo = versionInfo;
    if (this.versionInfo == null) {
      setEnabled(false);
      return;
    }

    final List<FileInfo> documentation = this.versionInfo.getDocumentation();
    if (documentation == null || documentation.isEmpty()) {
      setEnabled(false);
      return;
    }

    if (containsSinglePDFFile(documentation) && usesPDFViewer()) {
      setEnabled(true);
      return;
    }

    boolean hasHtmlFile = containsHtmlFile(documentation);
    final ExternalResources extResources = ExternalResources.getInstance();
    final boolean hasExtResources = extResources.isEnabled();
    if (documentation.size() == 1) {
      setEnabled(hasExtResources);
      return;
    }
    else if (hasHtmlFile) {
      setEnabled(hasExtResources);
      return;
    }
  }

  /**
   * Verifica se nos arquivos de documentao de uma verso de algoritmo, h um
   * arquivo com nome igual ao especificado em AlgorithmVersionInfo.HTML_FILE.
   * 
   * @param documentation lista de arquivos de documentao de uma verso de
   *        algoritmo
   * @return retorna true se existir um arquivo "index.html", caso contrrio,
   *         retorna false
   */
  private boolean containsHtmlFile(List<FileInfo> documentation) {
    for (FileInfo fileInfo : documentation) {
      //Devemos amarrar ao arquivo "index.html"? E ser for .htm ou .pdf?
      if (fileInfo.getName().equals(AlgorithmVersionInfo.HTML_FILE)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se nos arquivos de documentao de uma verso de algoritmo, h um
   * nico arquivo PDF; o que potencializa o uso de um visualizador interno PDF.
   * 
   * @param documentation lista de arquivos de documentao
   * @return indicativo.
   */
  private boolean containsSinglePDFFile(List<FileInfo> documentation) {
    return documentation != null && documentation.size() == 1
      && isResourcePDF(documentation.get(0).getName());
  }

  /**
   * Mostra o cone.
   */
  public void showIcon() {
    putValue(Action.SMALL_ICON, ApplicationImages.ICON_HELP_24);
  }

  /**
   * Mostra o nome.
   */
  public void showName() {
    putValue(Action.NAME, LNG.get("algorithms.help"));
  }

  /**
   * Exibe um erro.
   * 
   * @param throwable O erro.
   */
  private void showError(Throwable throwable) {
    StandardErrorDialogs.showErrorDialog(this.window, throwable);
  }

  /**
   * Exibe um erro.
   * 
   * @param message A mensagem de erro.
   */
  private void showError(String message) {
    StandardErrorDialogs.showErrorDialog(this.window, message);
  }
}
