package csbase.client.algorithms.commands.newview;

import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.algorithms.parameters.ComponentProperties;
import csbase.client.algorithms.parameters.ParameterView.Mode;
import csbase.client.kernel.ClientException;
import csbase.logic.CommandInfo;
import csbase.logic.ProgressData;
import csbase.logic.User;
import csbase.logic.algorithms.flows.Flow;

/**
 * Aba de informaes gerais sobre um fluxo.
 * 
 * @author Tecgraf
 */
public class FlowInfoTab extends AbstractTab {

  /**
   * Painel principal da aba.
   */
  private JScrollPane mainPanel;
  /**
   * Barra de progresso do comando.
   */
  private JProgressBar commandProgress;
  /**
   * Indica se o progresso do fluxo deve ser mostrado
   */
  private boolean showProgress;

  /**
   * Construtor.
   * 
   * @param command o comando sendo visualizado (No aceita {@code null}).
   * @param flow fluxo a ser visualizado.
   * @param showProgress indica se o progresso do fluxo deve ser mostrado.
   * @param title ttulo da aba.
   * @param owner janela dona da aba.
   */
  public FlowInfoTab(CommandInfo command, Flow flow, boolean showProgress,
    String title, Window owner) {
    super(TabType.STATIC, title, owner);
    this.showProgress = showProgress;
    JPanel internalPane = createInfoPanel(command, flow);
    mainPanel = new JScrollPane(internalPane);
  }

  /**
   * Cria painel de informaes sobre o fluxo.
   * 
   * @param command o comando sendo visualizado (No aceita {@code null}).
   * 
   * @param flow fluxo a ser visualizado.
   * @return o painel de informaes.
   */
  private JPanel createInfoPanel(CommandInfo command, Flow flow) {
    JPanel internalPane = new JPanel();
    internalPane.setLayout(new GridBagLayout());

    internalPane.add(createCommandPanel(command, flow), new GBC().northwest()
      .both().pushy());

    return internalPane;
  }

  /**
   * Cria o painel com informaes do fluxo.
   * 
   * @param command o comando sendo visualizado (No aceita {@code null}).
   * @param flow fluxo a ser visualizado.
   * @return o painel com informaes do fluxo.
   */
  private JPanel createCommandPanel(CommandInfo command, Flow flow) {
    JPanel commandPanel = new JPanel();
    commandPanel.setLayout(new GridBagLayout());

    String flowLabel =
      LNG.get(FlowInfoTab.class.getSimpleName() + ".label.flow.name");
    JComponent[] flowRow =
      { new JLabel(flowLabel), createTextField(flow.getName()) };

    String idLabel =
      LNG.get(FlowInfoTab.class.getSimpleName() + ".label.command.id");
    JComponent[] idRow =
      { new JLabel(idLabel), createTextField(command.getId()) };

    String descLabel =
      LNG.get(FlowInfoTab.class.getSimpleName() + ".label.command.description");
    JComponent[] descRow =
      { new JLabel(descLabel), createTextField(command.getDescription()) };

    String userLabel =
      LNG.get(FlowInfoTab.class.getSimpleName() + ".label.command.user");
    String userName = null;
    try {
      Object userId = command.getUserId();
      if (userId != null) {
        User user = User.getUser(userId);
        if (user != null) {
          userName = user.getName();
        }
      }
    }
    catch (Exception e) {
      userName = "---";
    }
    JComponent[] userRow = { new JLabel(userLabel), createTextField(userName) };

    String projLabel =
      LNG.get(FlowInfoTab.class.getSimpleName() + ".label.command.project");
    String projectId = command.getProjectId().toString();
    String projectName =
      projectId.replaceAll("^[/\\\\]?[^/\\\\]+[/\\\\](.*)$", "$1");
    JComponent[] projRow =
      { new JLabel(projLabel), createTextField(projectName) };

    String dateLabel =
      LNG.get(FlowInfoTab.class.getSimpleName() + ".label.command.date");
    DateFormat formatter = new SimpleDateFormat("dd/MM/yyyy - HH:mm:ss");
    Date date = command.getSubmittedDate();
    String strDate = formatter.format(date);
    JComponent[] dateRow = { new JLabel(dateLabel), createTextField(strDate) };

    String sgaLabel =
      LNG.get(FlowInfoTab.class.getSimpleName() + ".label.command.sga");
    JComponent[] sgaRow =
      { new JLabel(sgaLabel), createTextField(command.getSGAName()) };

    if (showProgress) {
      String progressLabel =
        LNG.get(FlowInfoTab.class.getSimpleName() + ".label.command.progress");
      commandProgress = new JProgressBar();
      commandProgress.setStringPainted(true);
      setProgress(command.getProgressData());
      JComponent[] progressRow = { new JLabel(progressLabel), commandProgress };
      GUIUtils.mountBasicGridPanel(commandPanel, new JComponent[][] { flowRow,
          idRow, descRow, userRow, projRow, dateRow, sgaRow, progressRow });
    }
    else {
      GUIUtils.mountBasicGridPanel(commandPanel, new JComponent[][] { flowRow,
          idRow, descRow, userRow, projRow, dateRow, sgaRow });
    }

    return commandPanel;
  }

  /**
   * Cria um campo de texto com valor indicado.
   * 
   * @param text o valor.
   * @return o campo.
   */
  private JTextField createTextField(String text) {
    ComponentProperties props = ComponentProperties.getInstance(Mode.REPORT);
    JTextField field = new JTextField();
    field.setText(text);
    field.setEditable(false);
    field.setOpaque(false);
    field.setBorder(props.getBorder());
    return field;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getMainComponent() throws ClientException {
    return mainPanel;
  }

  /**
   * Atualiza o valor do progresso do comando.
   * 
   * @param progress a informao de progresso
   */
  public void setProgress(ProgressData progress) {
    if (showProgress) {
      if (progress != null) {
        Double progressValue = progress.getValue();
        commandProgress.setValue(progressValue.intValue());
        commandProgress.setString(progress.getFormattedValue());
        commandProgress.setToolTipText(progress.getDescription());
      }
      else {
        commandProgress.setValue(0);
        commandProgress.setString("---");
        commandProgress.setToolTipText(null);
      }
    }
  }

}
