package csbase.client.algorithms.commands.view;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.StatusBar;
import csbase.client.algorithms.commands.cache.CommandsCache;
import csbase.client.algorithms.commands.cache.events.AbstractCommandUpdatedEventListener;
import csbase.client.algorithms.commands.cache.events.CommandUpdatedEvent;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.kernel.ClientException;
import csbase.client.project.tasks.GetFileTask;
import csbase.client.util.event.EventListener;
import csbase.client.util.gui.log.LogPanel;
import csbase.client.util.gui.log.LogPanel.FileEvent;
import csbase.client.util.gui.log.LogPanel.ThrowableEvent;
import csbase.client.util.gui.log.LogPanelReloader.StatusChangedEvent;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandFinalizationInfo.FinalizationInfoType;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;
import csbase.logic.CommonClientProject;
import csbase.logic.ExtendedCommandFinalizationInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.FileParameterValue;

/**
 * Dilogo abstrato de visualizao de {@link CommandInfo}.
 * 
 * @author Tecgraf / PUC-Rio
 */
abstract class AbstractAlgorithmCommandView extends DesktopComponentFrame {

  /**
   * Dimenso mnima preferida. S deve ser menor que isso se no couber na
   * tela.
   * 
   * Altura mnima
   */
  private static final long MINIMUN_PREFERRED_HEIGHT = 250;
  /**
   * Largura mnina
   */
  private static final long MINIMUN_PREFERRED_WIDTH = 800;

  /**
   * Aba selecionada no momento.
   */
  private Tab selected;
  /**
   * Painel que contm as abas.
   */
  private JTabbedPane tabbedPane;

  /**
   * Aba de parmetros do comando.
   */
  private Tab parametersTab;

  /**
   * Lista de abas de log.
   */
  private List<Tab> logTabs = new ArrayList<Tab>();

  /**
   * Recebe notificaes sobre atualizaces feitas no comando.
   */
  private AbstractCommandUpdatedEventListener commandListener;

  /**
   * Comando representado nessa viso.
   */
  private CommandInfo command;

  /**
   * O configurador a ter seus parmetros apresentados. Deve-se ignorar o
   * configurador do comando pois pode ser diferente deste. Isso ocorrerira no
   * caso de estar representando um n de um comando de fluxo (No aceita
   * {@code null}).
   */
  private AlgorithmConfigurator configurator;

  /**
   * Identificador do algoritmo a ser visualizado (caso seja n de um fluxo).
   */
  private Integer nodeId;

  /**
   * Tipo de log a ser mostrado nessa viso do comando. Pode ser
   * {@value CommandViewType#FLOW}, para mostrar todos os logs de um fluxo (de
   * forma agregada); {@value CommandViewType#NODE}, para os logs de um n
   * especfico; ou {@value CommandViewType#SIMPLE}, para os logs de um
   * algoritmo simples.
   */
  private CommandViewType viewType;

  /**
   * @param index identificador da view. Usado para garantir a unicidade.
   * @param owner A janela pai deste dilogo. No pode ser nula e tem que ser
   *        uma instncia de {@link JFrame} ou {@link JDialog}.
   * @param command O comando (No aceita {@code null}).
   * @param configurator O configurador a ter seus parmetros apresentados.
   *        Deve-se ignorar o configurador do comando pois pode ser diferente
   *        deste. Isso ocorrerira no caso de estar representando um n de um
   *        comando de fluxo (No aceita {@code null}).
   * @param viewType Tipo de log a ser mostrado nessa viso do comando (No
   *        aceita {@code null}).
   * @param nodeId Identificador do algoritmo (caso seja n de um fluxo).
   * 
   * @throws ClientException Erro ao obter informaes sobre o comando.
   */
  public AbstractAlgorithmCommandView(final Object index,
    final DesktopComponentFrame owner, final CommandInfo command,
    final AlgorithmConfigurator configurator, final CommandViewType viewType,
    final Integer nodeId) throws ClientException {

    super(index, owner, createTitle(command));

    this.command = command;
    this.configurator = configurator;
    this.viewType = viewType;
    this.nodeId = nodeId;
    this.commandListener = createCommandListener(command);
    CommandsCache.getInstance().addEventListener(commandListener);

    this.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
    this.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosed(WindowEvent e) {
        /**
         * Desseleciona a tab que estiver selecionada, para que esta se
         * finalize.
         */
        if (null != selected) {
          selected.setSelected(false);
        }
        stop();
        removeDesktopComponentFrame(getIndex());
      }
    });

    initialize(configurator);
    initializeGui(configurator);
  }

  /**
   * Mtodo que possibilita classes filhas a inicializarem atributos antes da
   * inicializao dos componentes grficos da classe.
   * 
   * @param cnf O configurador a ter seus parmetros apresentados. Deve-se
   *        ignorar o configurador do comando pois pode ser diferente deste.
   *        Isso ocorrerira no caso de estar representando um n de um comando
   *        de fluxo (No aceita {@code null}).
   * 
   * @throws ClientException Erro ao obter informaes sobre o comando.
   */
  protected void initialize(AlgorithmConfigurator cnf) throws ClientException {
    /*
     * Implementao "dummy" para no obrigar todas as subclasses a
     * implementarem o mtodo.
     */
  }

  /**
   * Mostra essa janela dando preferncia para abrir com a tab escolhida.
   * 
   * @param preferredTab Qual a tab gostaria que estivesse selecionada quando
   *        esta janela for criada. No  garantido que esta janela abra com a
   *        tab escolhida selecionada, pois aquela tab pode nem existir caso
   *        seja a tab de log.
   */
  public void show(final TabType preferredTab) {
    if (null != preferredTab) {
      Component component = null;
      switch (preferredTab) {
        case LOG:
          if (!logTabs.isEmpty()) {
            component = logTabs.get(0);
          }
          break;
        case PARAMETERS:
          component = parametersTab;
      }
      if (null != component) {
        tabbedPane.setSelectedComponent(component);
      }
    }
    setVisible(true);
  }

  /** Remove o observador do comando. */
  protected void stop() {
    CommandsCache.getInstance().removeEventListener(commandListener);
  }

  /**
   * @see java.awt.Window#pack()
   */
  @Override
  public void pack() {
    super.pack();
    updateSize();
    center(owner);
    updateLocation();
  }

  /**
   * Retorna comando representado por esta viso.
   * 
   * @return O comando representado pela viso.
   */
  protected CommandInfo getCommand() {
    return command;
  }

  /**
   * Determina o comando a ser visualizado por esta viso.
   * 
   * @param command O comando a ser visualizado.
   */
  protected void setCommand(CommandInfo command) {
    this.command = command;
    if (command.getStatus() == CommandStatus.FINISHED) {
      stop();
    }
  }

  /**
   * Retorna um listener para o comando a ser visualizado, para que a viso
   * possa ser atualizada.
   * 
   * @param command O comando a ser visualizado.
   * @return O listener para o comando.
   */
  private AbstractCommandUpdatedEventListener createCommandListener(
    CommandInfo command) {
    return new AbstractCommandUpdatedEventListener(command.getProjectId(),
      command.getId()) {
      @Override
      public void eventFired(CommandUpdatedEvent.Type type, CommandInfo cmd) {
        // Atualiza a instncia com o valor recebido pelo evento.
        setCommand(cmd);
        updateStatusBar();
      }

      @Override
      protected void eventInterrupted(Exception exception, String description) {
        getStatusBar().setError(description);
      }
    };
  }

  /**
   * Cria a tab de relatrio do configurador.
   * 
   * @param configurator O configurador a ter seus parmetros apresentados.
   *        Deve-se ignorar o configurador do comando pois pode ser diferente
   *        deste. Isso ocorrerira no caso de estar representando um n de um
   *        comando de fluxo (No aceita {@code null}).
   * 
   * @return a tab de relatrio do configurador.
   * 
   * @throws ClientException Erro ao obter informaes sobre o comando.
   */
  protected abstract ConfigurationReportTab createConfigurationReportTab(
    final AlgorithmConfigurator configurator) throws ClientException;

  /**
   * Cria os componentes e inicializa os listeners desta janela.
   * 
   * @param configurator O configurador a ter seus parmetros apresentados.
   *        Deve-se ignorar o configurador do comando pois pode ser diferente
   *        deste. Isso ocorrerira no caso de estar representando um n de um
   *        comando de fluxo (No aceita {@code null}).
   * 
   * @throws ClientException Erro ao obter informaes sobre o comando.
   */
  private void initializeGui(final AlgorithmConfigurator configurator)
    throws ClientException {

    String title = createTitle(command);
    this.setTitle(title);

    this.setLayout(new BorderLayout());

    final JPanel buttonPanel = createButtonPanel();
    StatusBar statusBar = getStatusBar();
    statusBar.showStatusBar();

    JPanel southPane = new JPanel(new BorderLayout());
    southPane.add(buttonPanel, BorderLayout.NORTH);
    southPane.add(statusBar, BorderLayout.SOUTH);
    add(southPane, BorderLayout.SOUTH);

    /** Deve-se criar a tab configurationViewTab antes de adicionar o evento. */
    this.tabbedPane = createTabbedPane(configurator);
    add(this.tabbedPane, BorderLayout.CENTER);

    /** Atualiza a barra de estado com o estado inicial do comando. */
    updateStatusBar();

    this.pack();
  }

  /**
   * Cria um ttulo que contm um texto fixo mais a descrio do comando e a
   * hora em que ele foi submetido.
   * 
   * @param command O comando (No aceita {@code null}).
   * 
   * @return o ttulo criado.
   */
  private static String createTitle(CommandInfo command) {
    String title = LNG.get("AbstractAlgorithmCommandView.title");
    String description = command.getDescription();
    if (description != null && description.length() != 0) {
      title += " : " + description;
    }
    Date startDate = command.getSubmittedDate();
    if (startDate != null) {
      title += " : " + FormatUtils.format(startDate);
    }
    return title;
  }

  /**
   * Cria o painel de tabs contento a tab de relatrio do configurador e a de
   * log.
   * 
   * @param configurator O configurador a ter seus parmetros apresentados.
   *        Deve-se ignorar o configurador do comando pois pode ser diferente
   *        deste. Isso ocorrerira no caso de estar representando um n de um
   *        comando de fluxo (No aceita {@code null}).
   * 
   * @return o painel de tabs.
   * 
   * @throws ClientException Erro ao obter informaes sobre o comando.
   */
  private JTabbedPane createTabbedPane(final AlgorithmConfigurator configurator)
    throws ClientException {

    final JTabbedPane tabbedPane = new JTabbedPane();
    tabbedPane.addChangeListener(new ChangeListener() {
      @Override
      public void stateChanged(ChangeEvent e) {
        Tab newSelected = (Tab) tabbedPane.getSelectedComponent();

        if (newSelected.equals(selected)) {
          return;
        }

        if (null != selected) {
          selected.setSelected(false);
        }

        selected = newSelected;
        newSelected.setSelected(true);
      }
    });

    createTabs(tabbedPane, configurator);

    return tabbedPane;
  }

  /**
   * Cria no painel de tabs, as tabs de relatrio do configurador e a de log.
   * 
   * @param tabbedPane Painel que receber as tabs.
   * 
   * @param configurator O configurador a ter seus parmetros apresentados.
   *        Deve-se ignorar o configurador do comando pois pode ser diferente
   *        deste. Isso ocorrerira no caso de estar representando um n de um
   *        comando de fluxo (No aceita {@code null}).
   * 
   * @throws ClientException Erro ao obter informaes sobre o comando.
   */
  private void createTabs(final JTabbedPane tabbedPane,
    final AlgorithmConfigurator configurator) throws ClientException {

    ConfigurationReportTab configurationViewTab =
      createConfigurationReportTab(configurator);
    addTab(tabbedPane, configurationViewTab);
    this.parametersTab = configurationViewTab;

    /**
     * Utiliza o tamanho da tab de configurao como o tamanho para as tabs de
     * log. Caso no fosse definido um tamanho para as tabs de log, estas teriam
     * o tamanho definido pela pgina do arquivo de log.
     */
    Dimension tabsDimension = new Dimension(600, 300);
    if (hasLogsTab()) {
      List<Tab> logsTabs = createLogsTab(tabsDimension);
      if (logsTabs != null && !logsTabs.isEmpty()) {
        for (Tab tab : logsTabs) {
          addTab(tabbedPane, tab);
          this.logTabs.add(tab);
        }
      }
    }

  }

  /**
   * Retorna a lista de configuraes de arquivos de log do comando.
   * 
   * @return A lista de configuraes.
   */
  private List<LogTabConfiguration> getLogTabsConfiguration() {
    String idForLogPattern = null;
    if (nodeId != null) {
      idForLogPattern = String.valueOf(nodeId);
    }
    GetLogTabConfigurationsTask task =
      new GetLogTabConfigurationsTask(getCommand(), viewType, idForLogPattern);
    if (task.execute(SwingUtilities.getWindowAncestor(this), getTitle(), LNG
      .get("AbstractAlgorithmCommandView.loading.logs"))) {
      return task.getResult();
    }
    else {
      StandardDialogs.showErrorDialog(this.getParent(), getTitle(), String
        .format(LNG
          .get("AbstractAlgorithmCommandView.loading.logs.error.message"),
          getCommand().getId()));
      return null;
    }
  }

  /**
   * Adiciona uma nova aba a esta viso.
   * 
   * @param tabbedPane O painel que armazena as abas.
   * @param tab Aba a ser adicionada.
   */
  private void addTab(final JTabbedPane tabbedPane, final Tab tab) {
    tabbedPane.addTab(tab.getTitle(), tab);
  }

  /**
   * Cria as abas de logs (configurveis por sistema).
   * 
   * @param preferredSize O tamanho preferencial da aba ou nulo.
   * @return a lista de abas de logs.
   */
  protected List<Tab> createLogsTab(final Dimension preferredSize) {
    List<LogTabConfiguration> configurations = getLogTabsConfiguration();
    List<Tab> logTabs = new ArrayList<Tab>();
    for (LogTabConfiguration conf : configurations) {
      Set<ClientProjectFile> matchingFiles = conf.getFiles();
      String title = conf.getTitle();
      Tab logTab = null;
      if (matchingFiles.size() == 1) {
        logTab = new LogTab(matchingFiles.iterator().next(), title);
      }
      else {
        logTab = new ConsolidatedLogsTab(getCommand(), matchingFiles, title);
      }
      logTab.setPreferredSize(preferredSize);
      logTabs.add(logTab);
    }
    return logTabs;
  }

  /**
   * Cria um painel com botes para a viso.
   * 
   * @return O painel com botes.
   */
  private JPanel createButtonPanel() {
    final JButton closeButton =
      new JButton(LNG.get("AbstractAlgorithmCommandView.button.close.label"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        close();
      }
    });
    final JPanel buttonPanel = new JPanel();
    buttonPanel.setLayout(new FlowLayout(FlowLayout.CENTER));
    buttonPanel.add(closeButton);

    return buttonPanel;
  }

  /**
   * Atualiza a barra de status com o estado do comando.
   */
  protected void updateStatusBar() {
    StatusBar statusBar = getStatusBar();
    switch (command.getStatus()) {
      case SCHEDULED:
        statusBar.setStatus(LNG
          .get("AbstractAlgorithmCommandView.status.scheduled"));
        break;
      case FINISHED:
        statusBar.setStatus(getFinalizationTypeDescription());
        break;
      case SYSTEM_FAILURE:
        statusBar.setStatus(LNG
          .get("AbstractAlgorithmCommandView.status.system_failure"));
        break;
      case EXECUTING:
      case INIT:
      case UPLOADING:
      case DOWNLOADING:
        if (command.isValid()) {
          if (command.isQueued()) {
            statusBar.setStatus(LNG
              .get("AbstractAlgorithmCommandView.status.queued"));
          }
          else {
            statusBar.setStatus(LNG
              .get("AbstractAlgorithmCommandView.status.running"));
          }
        }
        else {
          statusBar.setStatus(LNG
            .get("AbstractAlgorithmCommandView.status.disconnected"));
        }
        break;
      default:
        statusBar.setStatus(LNG
          .get("AbstractAlgorithmCommandView.status.unknown"));
        break;
    }
  }

  /**
   * Retorna a descrio do tipo de finalizao do comando ou de um algoritmo,
   * no caso de ser um n de um fluxo.
   * 
   * @return A descrio do tipo de finalizao de comando.
   */
  protected String getFinalizationTypeDescription() {
    CommandFinalizationInfo finalizationInfo = command.getFinalizationInfo();
    CommandFinalizationType finalizationType =
      finalizationInfo.getFinalizationType();
    if (nodeId == null
      || finalizationInfo.getInfoType() == FinalizationInfoType.SIMPLE) {
      Integer exitCode = finalizationInfo.getExitCode();
      return getFinalizationTypeDescription(finalizationType, exitCode);
    }
    else {
      ExtendedCommandFinalizationInfo flowfinalizationInfo =
        (ExtendedCommandFinalizationInfo) finalizationInfo;
      CommandFinalizationInfo nodeFinalizationInfo =
        flowfinalizationInfo.getFinalizationInfoForNode(nodeId);
      switch (nodeFinalizationInfo.getFinalizationType()) {
        case EXECUTION_ERROR:
          return getFinalizationTypeDescription(
            CommandFinalizationType.EXECUTION_ERROR, nodeFinalizationInfo
              .getExitCode());
        case SUCCESS:
          return getFinalizationTypeDescription(
            CommandFinalizationType.SUCCESS, null);
        case NO_EXIT_CODE:
          return getFinalizationTypeDescription(
            CommandFinalizationType.NO_EXIT_CODE, null);
        default:
          return getFinalizationTypeDescription(CommandFinalizationType.END,
            null);
      }
    }
  }

  /**
   * Retorna a descrio textual do tipo de finalizao.
   * 
   * @param finalizationType Tipo de finalizao.
   * @param exitCode Cdigo de retorno.
   * @return a descrio.
   */
  protected String getFinalizationTypeDescription(
    CommandFinalizationType finalizationType, Integer exitCode) {
    switch (finalizationType) {
      case END:
        return LNG.get("AbstractAlgorithmCommandView.status.finished");
      case EXECUTION_ERROR:
        if (exitCode != null) {
          return LNG.get(
            "AbstractAlgorithmCommandView.status.finished.error.code",
            new Object[] { exitCode });
        }
        else {
          return LNG.get("AbstractAlgorithmCommandView.status.finished.error");
        }
      case NO_EXIT_CODE:
        return (LNG.get("AbstractAlgorithmCommandView.status.finished.no_code"));
      case SUCCESS:
        return (LNG.get("AbstractAlgorithmCommandView.status.finished.success"));
      case FAILED:
        return (LNG.get("AbstractAlgorithmCommandView.status.finished.failed"));
      case KILLED:
        return (LNG.get("AbstractAlgorithmCommandView.status.finished.killed"));
      case LOST:
        return (LNG.get("AbstractAlgorithmCommandView.status.finished.lost"));
      default:
        return (LNG.get("AbstractAlgorithmCommandView.status.finished"));
    }
  }

  /**
   * Retorna o {@link ClientProjectFile} que corresponde ao
   * {@link FileParameterValue} que armazena o caminho de um arquivo.
   * 
   * @param fileParameter Parmetro que armazena o caminho do arquivo.
   * @return O {@link ClientProjectFile} correspondente.
   */
  protected ClientProjectFile getClientProjectFile(
    final FileParameterValue fileParameter) {
    if (fileParameter != null) {
      CommonClientProject project = DesktopFrame.getInstance().getProject();
      return GetFileTask.runTask(project, fileParameter.getPathAsArray());
    }
    return null;
  }

  /**
   * Ajusta o tamanho do dilogo para que ele no passe dos limites do monitor.
   */
  private void updateSize() {

    Dimension screenSize = GUIUtils.getScreenDimension();
    Dimension currentSize = getSize();

    double currentWidth = currentSize.getWidth();
    double newWidth = Math.max(currentWidth, MINIMUN_PREFERRED_WIDTH);
    double usuableWidth = screenSize.getWidth();
    newWidth = Math.min(newWidth, usuableWidth);

    double currentHeight = currentSize.getHeight();
    double newHeight = Math.max(currentHeight, MINIMUN_PREFERRED_HEIGHT);
    double usuableHeight = screenSize.getHeight();
    newHeight = Math.min(newHeight, usuableHeight);

    Dimension newSize = new Dimension();
    newSize.setSize(newWidth, newHeight);
    setSize(newSize);
  }

  /**
   * Ajusta a localizao do dilogo para que ele no passe dos limites do
   * monitor.
   */
  private void updateLocation() {
    Dimension screenSize = GUIUtils.getScreenDimension();
    Dimension currentSize = getSize();
    double usuableWidth = screenSize.getWidth();
    double currentWidth = currentSize.getWidth();
    double usuableHeight = screenSize.getHeight();
    double currentHeight = currentSize.getHeight();
    Point currentLocation = getLocation();
    double currentLeftmostX = currentLocation.getX();
    double availableWidth = usuableWidth - currentLeftmostX;
    double currentUpperY = currentLocation.getY();
    double availableHeight = usuableHeight - currentUpperY;
    double newLeftmostX;
    if (availableWidth < currentWidth) {
      newLeftmostX = currentLeftmostX - (currentWidth - availableWidth);
      newLeftmostX = Math.max(newLeftmostX, 0);
    }
    else {
      newLeftmostX = currentLeftmostX;
    }
    double newUpperY;
    if (availableHeight < currentHeight) {
      newUpperY = currentUpperY - (currentHeight - availableHeight);
      newUpperY = Math.max(newUpperY, 0);
    }
    else {
      newUpperY = currentUpperY;
    }
    Point newLocation = new Point();
    newLocation.setLocation(newLeftmostX, newUpperY);
    setLocation(newLocation);
  }

  /**
   * Retorna o configurador do comando apresentado por essa viso.
   * 
   * @return configurator O configurador do comando apresentado por essa viso.
   */
  public AlgorithmConfigurator getConfigurator() {
    return configurator;
  }

  /**
   * Aba que apresenta os parmetros do configurador do comando executado.
   * 
   * @author Tecgraf
   */
  protected abstract class ConfigurationReportTab extends Tab {

    /**
     * {@inheritDoc}
     */
    @Override
    protected void setSelected(boolean selected) {
      super.setSelected(selected);
      if (selected) {
        updateStatusBar();
      }
    }
  }

  /**
   * Indica se essa viso possui abas de log (configurvel por sistema).
   * 
   * @return verdadeiro se a viso possui abas de log ou falso, caso contrrio.
   */
  private boolean hasLogsTab() {
    ConfigurationManager cnfManager = ConfigurationManager.getInstance();
    final boolean defaultValue = false;
    if (cnfManager == null) {
      return defaultValue;
    }
    try {
      Class<ConsolidatedLogsTab> propClass = ConsolidatedLogsTab.class;
      Configuration cnf = cnfManager.getConfiguration(propClass);
      String propName = "enabled";
      Boolean isEnabled =
        cnf.getOptionalBooleanProperty(propName, defaultValue);
      return isEnabled;
    }
    catch (Exception e) {
      return defaultValue;
    }
  }

  /**
   * Aba que apresenta o log de execuo do comando
   */
  private class LogTab extends Tab {

    /**
     * Ttulo da aba
     */
    private final String title;

    /**
     * Viso atual da aba
     */
    private AbstractView view;

    /**
     * Construtor.
     * 
     * @param file Arquivo de log
     * @param title Ttulo da aba
     */
    LogTab(final ClientProjectFile file, final String title) {

      setLayout(new GridBagLayout());
      this.title = title;

      if (null == file) {
        showOpenLogView(file);
      }
      else {
        showLogPanelView(file);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getTitle() {
      return title;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setSelected(boolean selected) {
      view.setSelected(selected);
    }

    /**
     * Cria uma viso para a tab com um painel de log para o arquivo passado
     * como parmetro.
     * 
     * @param file Arquivo de log.
     */
    public void showLogPanelView(final ClientProjectFile file) {
      LogPanelView logPanelView = new LogPanelView(file);
      showView(logPanelView, new GBC(0, 0).both());
    }

    /**
     * Cria uma viso para a tab com um boto centralizado que permite que o
     * usurio force a abertura do painel de log do arquivo.
     * 
     * @param file Arquivo de log.
     */
    public void showOpenLogView(final ClientProjectFile file) {
      OpenLogView openLogView = new OpenLogView(file);
      showView(openLogView, null);
    }

    /**
     * Mostra uma determinada viso na aba.
     * 
     * @param view Viso a ser mostrada.
     * @param obj Objeto que contm configurao de layout para a viso.
     */
    private void showView(AbstractView view, Object obj) {
      setVisible(false);

      removeAll();
      add(view, obj);
      this.view = view;
      if (null != selected && selected.equals(this)) {
        this.view.setSelected(true);
      }
      setVisible(true);
    }

    /**
     * Viso de uma aba.
     */
    abstract class AbstractView extends JPanel {

      /**
       * Determina se esta aba est selecionada.
       * 
       * @param selected Indica se a viso est selecionada.
       */
      public abstract void setSelected(boolean selected);
    }

    /**
     * Viso que apresenta um painel de visualizao do log de execuo do
     * comando.
     */
    private class LogPanelView extends AbstractView {

      /**
       * Painel de visualizao do log.
       */
      private final LogPanel logPanel;
      /**
       * Indica se o painel no foi selecionado ainda.
       */
      private final AtomicBoolean wasNeverSelected;
      /**
       * Indica se o comando j entrou em execuo.
       */
      private final AtomicBoolean wasRunningStatus;
      /**
       * Indica se o evento de atualizao de status deve ser ignorado.
       */
      private final AtomicBoolean ignoreReloaderStatusChangedEvent;

      /**
       * Construtor
       * 
       * @param file Arquivo com o contedo do log.
       */
      LogPanelView(final ClientProjectFile file) {
        final int pageSizeKb = LogPanel.DEFAULT_PAGE_SIZE_KB;

        logPanel =
          LogPanel.createLogPanelWithToolBar(LogPanel.RELOAD | LogPanel.PAGING,
            pageSizeKb);

        logPanel.addThrowableEventListener(new EventListener<ThrowableEvent>() {
          @Override
          public void eventFired(ThrowableEvent event) {
            getStatusBar().setError(event.getMessage());
          }
        });

        logPanel.addFileEventListener(new EventListener<FileEvent>() {
          @Override
          public void eventFired(FileEvent event) {
            if (FileEvent.Type.NOT_FOUND == event.getType()) {
              logPanel.getReloader().stop();
              logPanel.closeFile();
              LogTab.this.showOpenLogView(file);
            }
            updateStatusBar();
          }
        });

        wasNeverSelected = new AtomicBoolean(true);

        ignoreReloaderStatusChangedEvent = new AtomicBoolean(false);

        /** Armazena o estado do boto do reload. */
        wasRunningStatus =
          new AtomicBoolean(logPanel.getReloader().isRunning());

        logPanel.getReloader().addStatusChangedEventListener(
          new EventListener<StatusChangedEvent>() {
            @Override
            public void eventFired(StatusChangedEvent event) {
              if (!ignoreReloaderStatusChangedEvent.compareAndSet(true, false)) {
                wasRunningStatus.set(event.isRunning());
              }
            }
          });

        CommandsCache.getInstance().addEventListener(
          new AbstractCommandUpdatedEventListener(getCommand().getProjectId(),
            getCommand().getId()) {
            @SuppressWarnings("incomplete-switch")
            @Override
            protected void eventFired(CommandUpdatedEvent.Type type,
              CommandInfo command) {

              /*
               * Caso o commando tenha terminado sua execuo (independente do
               * porqu),  preciso parar de recarregar o arquivo de log
               */
              switch (type) {
                case end:
                case success:
                case error:
                case failed:
                case killed:
                case lost:
                case no_code:
                case removed: {
                  logPanel.getReloader().stop();
                }
              }
            }
          });

        setLayout(new GridBagLayout());
        add(logPanel, new GBC(0, 0).both());
        logPanel.openFile(file);
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public void setSelected(boolean selected) {
        /*
         * Se a tab de log est sendo selecionada pela primeira vez e o comando
         * ainda no terminou, a recarga de log deve estar ligada. Se este no
         * for o caso, a recarga deve ficar desligada se esta tab estiver sendo
         * desselecionada ou deve ficar no ltimo estado em que o usurio a
         * deixou. <br>
         * 
         * O ignoreReloaderStatusChangedEvent permite que ao sair desta aba, o
         * reloader seja desligado, mas que o estado escolhido pelo usurio
         * atravs do boto de reload ou pelo fato do comando estar em execuo,
         * seja armazenado no wasRunningStatus. Assim, os eventos gerados pela
         * ativao ou desativao do reload a partir da seleo de abas so
         * ignorados e os demais - usurio escolhendo o estado atravs do boto
         * de recarga - so pegos e o estado da recarga  salvo. Porm, como a
         * lgica da 1a seleo  diferente, este evento no deve ser ignorado.
         */

        boolean ignoreReloader = !wasNeverSelected.get();

        if (wasNeverSelected.compareAndSet(true, !selected)
          && !getCommand().getStatus().equals(CommandStatus.FINISHED)) {
          // O if-elseif desta forma pois fica mais legvel assim.
        }
        else if (!wasRunningStatus.get()) {
          // A recarga estava desligada, ento deve continuar assim.
          return;
        }

        ignoreReloaderStatusChangedEvent.set(ignoreReloader);

        if (selected) {
          logPanel.getReloader().start();
        }
        else {
          logPanel.getReloader().stop();
        }
      }
    }

    /**
     * Viso que permite a abertura de um arquivo de log.
     */
    private class OpenLogView extends AbstractView {

      /**
       * Construtor
       * 
       * @param file Arquivo com o contedo do log.
       */
      OpenLogView(final ClientProjectFile file) {
        super();
        /**
         * Caso o arquivo no tenha sido encontrado, talvez por ainda no ter
         * sido criado, devemos por um boto para que o usurio consiga forar o
         * sistema a abri-lo de novo.
         */
        class OpenAction extends AbstractAction {
          OpenAction() {
            super(LNG
              .get("AbstractAlgorithmCommandView.tab.log.button.reload.label"),
              ApplicationImages.ICON_REFRESH_24);
          }

          @Override
          public void actionPerformed(ActionEvent e) {
            if (null == file) {
              String warningMsg =
                LNG
                  .get("AbstractAlgorithmCommandView.tab.log.file.notFound.dialog");
              String[] buttons =
                new String[] { LogPanel.getString("button.ok") };
              JOptionPane.showOptionDialog(getOwner(), warningMsg, getTitle(),
                JOptionPane.DEFAULT_OPTION, JOptionPane.WARNING_MESSAGE, null,
                buttons, buttons[0]);
            }
            else {
              LogTab.this.showLogPanelView(file);
              updateStatusBar();
            }
          }
        }
        final JButton openLogBtn = new JButton(new OpenAction());
        openLogBtn.setToolTipText(LNG
          .get("AbstractAlgorithmCommandView.tab.log.button.reload.tooltip"));

        setLayout(new BorderLayout());
        add(openLogBtn, BorderLayout.CENTER);
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public void setSelected(boolean selected) {
        if (selected) {
          getStatusBar()
            .setWarning(
              LNG
                .get("AbstractAlgorithmCommandView.tab.log.file.notFound.statusbar"));
        }
        else {
          updateStatusBar();
        }
      }
    }
  }
}
