package csbase.client.applications.commandviewer;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.core.lng.LNG;
import csbase.client.algorithms.commands.CommandViewerPanel;
import csbase.client.algorithms.commands.cache.CommandsCache;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationProject;
import csbase.client.applications.executor.ExecutorFrame;
import csbase.client.applications.flowapplication.FlowApplication;
import csbase.client.desktop.RemoteTask;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.CommandInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;

/**
 * <p>
 * Aplicao Visualizadora de Comandos
 * </p>
 * 
 * @author lmoreira
 */
public final class CommandViewerApplication extends ApplicationProject {

  /**
   * O painl de visualizao de comandos.
   */
  private CommandViewerPanel panel;

  /**
   * Cria a aplicao.
   * 
   * @param id O identificador da aplicao (No aceita {@code null}).
   */
  public CommandViewerApplication(String id) {
    super(id);
    createGui();
  }

  /**
   * @see csbase.client.applications.Application#killApplication()
   */
  @Override
  public void killApplication() {
    panel.stop();
  }

  /**
   * @see csbase.client.applications.Application#startApplication()
   */
  @Override
  public void startApplication() throws ApplicationException {
    super.startApplication();
    panel.start();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    return true;
  }

  /**
   * Cria o menu Comandos.
   * 
   * @return O menu.
   */
  private JMenu createCommandMenu() {
    JMenu commandMenu = new JMenu(getMessage("label_command_menu"));
    JMenuItem openCommandMenuItem = new JMenuItem(panel.getShowCommandAction());
    commandMenu.add(openCommandMenuItem);
    JMenuItem openCommandLogMenuItem =
      new JMenuItem(panel.getShowCommandLogAction());
    commandMenu.add(openCommandLogMenuItem);
    JMenuItem exportCommandMenuItem =
      new JMenuItem(panel.getExportCommandAction());
    commandMenu.add(exportCommandMenuItem);
    JMenuItem removeCommandMenuItem =
      new JMenuItem(panel.getRemoveCommandAction());
    commandMenu.add(removeCommandMenuItem);

    JMenuItem killCommandMenuItem = new JMenuItem(panel.getKillCommandAction());
    commandMenu.add(killCommandMenuItem);

    if (hasAdvancedMenu()) {
      commandMenu.addSeparator();
      JMenu advancedSubMenu = createAdvancedMenu();
      commandMenu.add(advancedSubMenu);
    }

    commandMenu.addSeparator();
    commandMenu.add(new ApplicationExitAction(this));

    return commandMenu;
  }

  /**
   * Testa se o menu avanado est habilitado na configurao do sistema.
   * 
   * @return verdadeiro se o menu deve ser mostrado ou falso, caso contrrio.
   */
  private boolean hasAdvancedMenu() {
    final ConfigurationManager cnfManager = ConfigurationManager.getInstance();
    if (cnfManager == null) {
      return false;
    }
    try {
      final Class<CommandViewerApplication> propClass =
        CommandViewerApplication.class;
      final Configuration cnf = cnfManager.getConfiguration(propClass);
      final String propName = "enable.advanced.menu";
      return cnf.getOptionalBooleanProperty(propName, false);
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Cria o menu Avanado.
   * 
   * @return O menu.
   */
  private JMenu createAdvancedMenu() {
    JMenu advancedSubMenu = new JMenu(getMessage("label_advanced_menu"));
    advancedSubMenu.add(createRefreshCommandsAction());
    return advancedSubMenu;
  }

  /**
   * Cria item de menu de recarga dos comandos.
   * 
   * @return O item de menu.
   */
  private JMenuItem createRefreshCommandsAction() {
    JMenuItem refreshMenuItem =
      new JMenuItem(getMessage("label_refresh_menu_item"));
    refreshMenuItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent event) {
        RemoteTask<Void> task = new RemoteTask<Void>() {
          @Override
          public void performTask() throws Exception {
            CommandsCache.getInstance().hardReload(
              getApplicationProject().getId());
          }
        };
        String msg = getMessage("message_refreshing");
        if (!task.execute(getApplicationFrame(), msg, msg)) {
          if (task.getError() != null) {
            showExceptionStack(task.getError());
          }
        }
      }
    });
    return refreshMenuItem;
  }

  /**
   * Cria a interface grfica.
   */
  private void createGui() {
    buildApplicationFrame();
    ApplicationFrame applicationFrame = getApplicationFrame();
    panel =
      new CommandViewerPanel(getApplicationFrame(), getApplicationProject()
        .getId()) {
        @Override
        protected boolean exportCommand(final CommandInfo command) {
          try {
            RemoteTask<AlgorithmConfigurator> task =
              new RemoteTask<AlgorithmConfigurator>() {
                @Override
                public void performTask() throws Exception {
                  AlgorithmConfigurator configurator =
                    command.getConfigurator();
                  setResult(configurator);
                }
              };
            String msg = getMessage("loading_configurator");
            if (!task.execute(getApplicationFrame(), msg, msg)) {
              return false;
            }
            AlgorithmConfigurator configurator = task.getResult();
            if (null == configurator) {
              showError(getMessage("error_loading_configurator"));
              return false;
            }

            /*
             * Quando o configurador no tem nome nem verso, abre a aplicao
             * de fluxo para visualizar o comando. Note que um configurador de
             * fluxo PODE ter nome e verso, mas nesse caso ele deve ser aberto
             * pelo ExecutorFrame!
             */
            if (null == configurator.getAlgorithmName()
              && null == configurator.getAlgorithmVersionId()
              && configurator.getConfiguratorType() == ConfiguratorType.FLOW) {

              ApplicationManager applicationManager =
                ApplicationManager.getInstance();
              FlowApplication application =
                applicationManager.runApplication(FlowApplication.class);
              return application.editConfigurator(
                (FlowAlgorithmConfigurator) configurator,
                command.getDescription());
            }
            try {
              String algName = configurator.getAlgorithmName();
              AlgorithmVersionId algVersionId =
                configurator.getAlgorithmVersionId();
              Map<String, String> parameters = new HashMap<String, String>();
              Set<String> names = configurator.getParameterNames();
              for (String name : names) {
                String value = configurator.getParameterValue(name);
                parameters.put(name, value);
              }

              ExecutorFrame executorFrame =
                (ExecutorFrame) ApplicationManager.getInstance()
                  .runApplication("executor");
              executorFrame.showConfigurator(algName, algVersionId, parameters);
              return true;
            }
            catch (ParameterNotFoundException e) {
              showExceptionStack(e);
              return false;
            }
          }
          catch (ApplicationException e) {
            showExceptionStack(e);
            return false;
          }
        }
      };
    applicationFrame.add(panel);
    JMenuBar menuBar = createMenuBar();
    applicationFrame.setJMenuBar(menuBar);
    applicationFrame.pack();
    applicationFrame.pack();
    applicationFrame.setTitle(getMessage("title"));
  }

  /**
   * Cria a barra de menu.
   * 
   * @return A barra de menu.
   */
  private JMenuBar createMenuBar() {
    JMenuBar menuBar = new JMenuBar();
    menuBar.add(createCommandMenu());
    return menuBar;
  }

  /**
   * Obtm uma mensagem no mecanismo de internacionalizao.
   * 
   * @param keySuffix O sufixo da chave (o nome da classe ser pr-fixado ao
   *        sufixo).
   * 
   * @return A mensagem.
   */
  private String getMessage(String keySuffix) {
    return LNG.get(getClass().getSimpleName() + "." + keySuffix);
  }

}
