package csbase.client.applications.flowapplication.zoom;

import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.JSpinner;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;

import csbase.client.applications.ApplicationImages;
import csbase.client.applications.flowapplication.graph.actions.Action;

/**
 * <p>
 * Controle de zoom.
 * </p>
 * 
 * <p>
 * Painel com os widgets que controlam o zoom.
 * </p>
 * 
 * @author lmoreira
 */
public final class ZoomControl extends JToolBar {
  /**
   * Modelo de zoom.
   */
  private ZoomModel model;

  /**
   * Ao de zoom in.
   */
  private ZoomInAction zoomInAction;

  /**
   * Ao de zoom out.
   */
  private ZoomOutAction zoomOutAction;

  /**
   * Ao de zoom out.
   */
  private ZoomFitAction zoomFitAction;

  /**
   * Cria um controle de zoom para o modelo de zoom dado.
   * 
   * @param model O modelo de zoom.
   */
  public ZoomControl(final ZoomModel model) {
    setLayout(new FlowLayout(FlowLayout.RIGHT));
    setFloatable(false);
    setModel(model);
    buildZoomInAction();
    buildZoomOutAction();
    buildZoomFitAction();
    buildSlider();
    buildSpinner();
  }

  /**
   * Obtm a ao de zoom in.
   * 
   * @return A ao de zoom in.
   */
  public Action getZoomInAction() {
    return zoomInAction;
  }

  /**
   * Obtm a ao de fit.
   * 
   * @return a ao de fit.
   */
  public Action getZoomFitAction() {
    return zoomFitAction;
  }

  /**
   * Obtm a ao de zoom out.
   * 
   * @return A ao de zoom out.
   */
  public Action getZoomOutAction() {
    return zoomOutAction;
  }

  /**
   * Constri o slider.
   */
  private void buildSlider() {
    final ZoomSliderModel sliderModel = new ZoomSliderModel(model);
    final JSlider slider = new JSlider(sliderModel);
    slider.setMajorTickSpacing(sliderModel.getMajorTicks());
    slider.setMinorTickSpacing(sliderModel.getMinorTicks());
    slider.setOrientation(SwingConstants.HORIZONTAL);
    slider.setPaintTicks(true);
    add(slider);
  }

  /**
   * Constri o spinner.
   */
  private void buildSpinner() {
    final JPanel panel = new JPanel();
    panel.add(new JLabel("%"));
    panel.add(new JSpinner(new ZoomSpinnerModel(model)));
    add(panel);
  }

  /**
   * Constri a ao zoom in e os widgets que a utilizam.
   */
  private void buildZoomInAction() {
    zoomInAction = new ZoomInAction();
    final JButton zoomInButton = new JButton(zoomInAction);
    zoomInButton.setText("");
    zoomInButton.setToolTipText(zoomInAction.getToolTipText());
    add(zoomInButton);
  }

  /**
   * Constri a ao fit in e os widgets que a utilizam.
   */
  private void buildZoomFitAction() {
    zoomFitAction = new ZoomFitAction();
    final JButton zoomFitButton = new JButton(zoomFitAction);
    zoomFitButton.setText("");
    zoomFitButton.setToolTipText(zoomFitAction.getToolTipText());
    add(zoomFitButton);
  }

  /**
   * Constri a ao zoom out e os widgets que a utilizam.
   */
  private void buildZoomOutAction() {
    zoomOutAction = new ZoomOutAction();
    final JButton zoomOutButton = new JButton(zoomOutAction);
    zoomOutButton.setText("");
    zoomOutButton.setToolTipText(zoomOutAction.getToolTipText());
    add(zoomOutButton);
  }

  /**
   * Atribui o modelo de zoom a este controle de zoom.
   * 
   * @param zoomModel O modelo de zoom.
   */
  private void setModel(final ZoomModel zoomModel) {
    model = zoomModel;
    model.addListener(new ZoomControlListener());
  }

  /**
   * Zoom in.
   */
  private void zoomIn() {
    model.setValue(model.getNextBlockValue());
  }

  /**
   * Zoom out.
   */
  private void zoomOut() {
    model.setValue(model.getPreviousBlockValue());
  }

  /**
   * Zoom out.
   */
  private void zoomFit() {
    model.setValue(model.getFitValue());
  }

  /**
   * O listener de zoom que comunica ao controle de zoom sobre mudanas de zoom.
   */
  private final class ZoomControlListener implements ZoomListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void wasChanged(final ZoomModel zoomModel) {
      zoomInAction.setEnabled(zoomModel.hasNextBlockValue());
      zoomOutAction.setEnabled(zoomModel.hasPreviousBlockValue());

      final double value = zoomModel.getValue();
      final double defaultValue = zoomModel.getStandardValue();
      zoomFitAction.setEnabled(value != defaultValue);
    }
  }

  /**
   * Ao zoom in.
   */
  private final class ZoomInAction extends Action {
    /**
     * Cria a ao.
     */
    ZoomInAction() {
      super(
        "csbase.client.applications.flowapplication.zoom.ZoomControl.zoomIn",
        ApplicationImages.ICON_ZOOMIN_16, KeyStroke.getKeyStroke(
          KeyEvent.VK_ADD, 0));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(final ActionEvent e) {
      zoomIn();
    }
  }

  /**
   * Ao zoom out.
   */
  private final class ZoomOutAction extends Action {
    /**
     * Cria a ao.
     */
    ZoomOutAction() {
      super(
        "csbase.client.applications.flowapplication.zoom.ZoomControl.zoomOut",
        ApplicationImages.ICON_ZOOMOUT_16, KeyStroke.getKeyStroke(
          KeyEvent.VK_SUBTRACT, 0));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(final ActionEvent e) {
      zoomOut();
    }
  }

  /**
   * Ao zoom out.
   */
  private final class ZoomFitAction extends Action {
    /**
     * Cria a ao.
     */
    ZoomFitAction() {
      super(
        "csbase.client.applications.flowapplication.zoom.ZoomControl.zoomFit",
        ApplicationImages.ICON_ZOOM_16, KeyStroke.getKeyStroke(
          KeyEvent.VK_HOME, 0));
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(final ActionEvent e) {
      zoomFit();
    }
  }
}
