/*
 * $Id:
 */

package csbase.client.externalresources;

import java.awt.print.PageFormat;
import java.awt.print.Pageable;
import java.awt.print.Printable;

import javax.jnlp.PrintService;
import javax.jnlp.ServiceManager;
import javax.jnlp.UnavailableServiceException;

import csbase.exception.CSBaseException;

/**
 * Classe para uso do servio de impresso do JNLP.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ExternalPrintService {
  /**
   * Mensagem de erro para excees de servio indisponvel.
   */
  private static final String UNAVAILABLE_SERVICE_MESSAGE =
    "O servio {0} no est disponvel: {1}";

  /**
   * O nome da classe que representa o servio de impresso.
   */
  private static final String PRINT_SERVICE_CLASS_NAME =
    "javax.jnlp.PrintService";

  /**
   * A instncia nica.
   */
  private static ExternalPrintService instance;

  /**
   * Construtor privado para evitar a criao de objetos.
   */
  private ExternalPrintService() {
    //Sem implementao
  }

  /**
   * Obtm a instncia nica.
   * 
   * @return A instncia nica.
   */
  public static ExternalPrintService getInstance() {
    if (instance == null) {
      instance = new ExternalPrintService();
    }
    return instance;
  }

  /**
   * Verifica se o servio de impresso est disponvel.
   * 
   * @return true, caso o servio de impresso esteja disponvel, ou false, caso
   *         contrrio.
   */
  public boolean isAvailable() {
    try {
      ServiceManager.lookup(PRINT_SERVICE_CLASS_NAME);
      return true;
    }
    catch (final UnavailableServiceException e) {
      return false;
    }
  }

  /**
   * Consulta ao servio de impresso.
   * 
   * @return o servio.
   * @throws CSBaseException
   */
  private PrintService getPrintService() throws CSBaseException {
    try {
      return (PrintService) ServiceManager.lookup(PRINT_SERVICE_CLASS_NAME);
    }
    catch (final UnavailableServiceException e) {
      throw new CSBaseException(UNAVAILABLE_SERVICE_MESSAGE, new Object[] {
          PRINT_SERVICE_CLASS_NAME, e.getMessage() });
    }
  }

  /**
   * Consulta o formato de pgina default.
   * 
   * @return a pgina
   * @throws CSBaseException
   */
  public PageFormat getDefaultPage() throws CSBaseException {
    final PrintService printService = this.getPrintService();
    return printService.getDefaultPage();
  }

  /**
   * Abtrao para consulta e exibio do formato de pgina (ver
   * {@link PrintService#showPageFormatDialog(PageFormat)}).
   * 
   * @param page formato de pgina
   * @return objeto de formato de pgina.
   * @throws CSBaseException
   */
  public PageFormat showPageFormatDialog(final PageFormat page)
    throws CSBaseException {
    final PrintService printService = this.getPrintService();
    return printService.showPageFormatDialog(page);
  }

  /**
   * Imprime um documento usando o objeto recebido.
   * 
   * @param document A pgina a ser impressa.
   * 
   * @return true, se a impresso ocorreu, ou false, caso contrrio.
   * 
   * @throws CSBaseException Caso ocorra algum erro durante o procedimento.
   */
  public boolean print(final Pageable document) throws CSBaseException {
    final PrintService printService = this.getPrintService();
    return printService.print(document);
  }

  /**
   * Imprime um documento usando o objeto recebido.
   * 
   * @param painter O objeto utilizado para renderizar cada pgina do documento.
   * 
   * @return true, se a impresso ocorreu, ou false, caso contrrio.
   * 
   * @throws CSBaseException Caso ocorra algum erro durante o procedimento.
   */
  public boolean print(final Printable painter) throws CSBaseException {
    final PrintService printService = this.getPrintService();
    return printService.print(painter);
  }
}
