/*
 * $Id$
 */
package csbase.client.openbus;

import java.rmi.RemoteException;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Properties;
import java.util.logging.Level;

import org.omg.CORBA.COMM_FAILURE;
import org.omg.CORBA.NO_PERMISSION;
import org.omg.CORBA.ORB;
import org.omg.CORBA.TRANSIENT;
import org.omg.CORBA.ORBPackage.InvalidName;
import org.omg.PortableServer.POA;
import org.omg.PortableServer.POAHelper;
import org.omg.PortableServer.POAManagerPackage.AdapterInactive;
import org.omg.PortableServer.POAManagerPackage.State;

import scs.core.IComponent;
import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.openbus.CallerChain;
import tecgraf.openbus.Connection;
import tecgraf.openbus.InvalidLoginCallback;
import tecgraf.openbus.OpenBusContext;
import tecgraf.openbus.SharedAuthSecret;
import tecgraf.openbus.core.ORBInitializer;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;
import tecgraf.openbus.core.v2_0.services.access_control.AccessDenied;
import tecgraf.openbus.core.v2_0.services.access_control.InvalidLogins;
import tecgraf.openbus.core.v2_0.services.access_control.LoginInfo;
import tecgraf.openbus.core.v2_0.services.access_control.NoLoginCode;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOfferDesc;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceProperty;
import tecgraf.openbus.exception.AlreadyLoggedIn;
import tecgraf.openbus.exception.InvalidEncodedStream;
import tecgraf.openbus.exception.InvalidLoginProcess;
import tecgraf.openbus.exception.InvalidPropertyValue;
import tecgraf.openbus.session_service.v1_05.SessionEventSinkPOA;
import csbase.client.Client;
import csbase.client.login.LoginInterface;
import csbase.client.login.PreLogin;
import csbase.client.login.UserPasswordLogin;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.exception.CSBaseException;
import csbase.exception.OperationFailureException;
import csbase.logic.BusInfo;
import csbase.logic.Session;
import csbase.logic.User;
import csbase.logic.openbus.OpenBusLoginToken;
import csbase.remote.Authenticator;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.OpenBusServiceInterface;
import csbase.remote.UserPasswordAuthenticator;

/**
 * Responsvel por realizar a conexo com o OpenBus.
 *
 * @author Tecgraf/PUC-Rio
 */
public final class OpenBusAccessPoint {
  private static final String OPENBUS_OFFER_ID_KEY = "openbus.offer.id";

  /**
   * A instncia nica.
   */
  private static OpenBusAccessPoint instance;

  /**
   * Indica se o acesso ao OpenBus j foi iniciado.
   */
  private volatile Boolean inited = false;

  /**
   * <i>Thread</i> responsvel pela execuo do ORB.
   */
  private ORBThread orbThread;

  /**
   * A configurao desta classe.
   */
  final private Configuration configuration;

  /**
   * O host do OpenBus
   */
  private String host;

  /**
   * A porta do OpenBus
   */
  private int port;

  /**
   * As propriedades do ORB
   */
  private Properties orbProperties;

  /**
   * Sesso.
   */
  private OpenBusSession session;

  /**
   * Contexto do OpenBus
   */
  private OpenBusContext context;

  /**
   * Propriedades da conexo
   */
  private Properties connectionProps;

  /**
   * Conexo com o barramento
   */
  private Connection connection;

  /**
   * Construtor.
   */
  private OpenBusAccessPoint() {
    setInited(false);
    try {
      final ConfigurationManager configurationManager =
        ConfigurationManager.getInstance();
      this.configuration = configurationManager.getConfiguration(getClass());
    }
    catch (ConfigurationManagerException e) {
      throw new IllegalStateException(e);
    }

    PrivilegedAction<Object> action = new PrivilegedAction<Object>() {
      @Override
      public Object run() {
        System.setSecurityManager(null);
        return null;
      }
    };
    AccessController.doPrivileged(action);
  }

  /**
   * Finaliza o acesso ao barramento (se estiver iniciado e/ou conectado)
   * {@link OpenBusAccessPoint#shutdown()}
   */
  public static void destroy() {
    if (instance == null) {
      return;
    }
    instance.shutdown();
  }

  /**
   * Obtm a instncia nica.
   *
   * @return A instncia nica.
   */
  public static OpenBusAccessPoint getInstance() {
    if (instance == null) {
      instance = new OpenBusAccessPoint();
    }
    return instance;
  }

  /**
   * Inicializa o acesso ao barramento.
   *
   * @throws CSBaseException Caso ocorra algum erro na inicializao.
   * @throws RemoteException Caso haja falha na comunicao com o servidor
   *         CSBase.
   */
  public void init() throws CSBaseException, RemoteException {
    final OpenBusServiceInterface openBusService =
      ClientRemoteLocator.openBusService;
    if (openBusService == null) {
      final String err = getString("openbus.service.null.error");
      throw new CSBaseException(err);
    }
    if (!openBusService.isActive()) {
      final String err = getString("openbus.service.inactive.error");
      throw new CSBaseException(err);
    }

    BusInfo busInfo = openBusService.getBusInfo();

    host = busInfo.getHost();
    port = busInfo.getPort();
    orbProperties = new Properties();
    orbProperties.setProperty("org.omg.CORBA.ORBClass", "org.jacorb.orb.ORB");

    /*
     * Configura o nvel de Logging do SDK OpenBus usando a API do
     * java.util.logging
     */
    java.util.logging.Logger SDKLogger =
      java.util.logging.Logger.getLogger("tecgraf.openbus");
    SDKLogger.setLevel(Level.WARNING);

    orbThread = new ORBThread(ORBInitializer.initORB(null, orbProperties));
    orbThread.start();
    try {
      this.context =
        (OpenBusContext) orbThread.orb
          .resolve_initial_references("OpenBusContext");
    }
    catch (InvalidName e) {
      throw new CSBaseException("Falha ao obter o OpenBusContext.", e);
    }

    this.connectionProps = new Properties();
    connectionProps.put("legacy.disable", "false");
    connectionProps.put("legacy.delegate", "originator");
    try {
      this.connection =
        this.context.createConnection(host, port, this.connectionProps);
    }
    catch (InvalidPropertyValue e) {
      throw new CSBaseException("Falha ao criar a conexo com o barramento.", e);
    }

    this.context.setDefaultConnection(this.connection);
    this.connection.onInvalidLoginCallback(new OnInvalidLogin());

    connect();

    setInited(true);
  }

  /**
   * Ajuste o flag interno sincronizado de inicializao.
   *
   * @param flag flag.
   */
  private void setInited(final boolean flag) {
    synchronized (inited) {
      this.inited = flag;
    }
  }

  /**
   * Verifica se o acesso ao OpenBus j foi iniciado.
   *
   * @return {@code true} caso o acesso ao OpenBus j tenha sido iniciado ou,
   *         {@code false}, caso contrrio.
   */
  final public boolean isInited() {
    synchronized (inited) {
      return this.inited;
    }
  }

  /**
   * Conexo.
   *
   * @throws CSBaseException em caso de erro no OpenBus
   * @throws RemoteException em caso de falha de comunicao.
   */
  void connect() throws RemoteException, CSBaseException {
    final Client client = Client.getInstance();
    final LoginInterface loginObject = client.getLoginObject();
    if (loginObject instanceof OpenBusLogin) {
      final ClientRemoteMonitor clientRemoteMonitor =
        ClientRemoteMonitor.getInstance();
      final Session session = clientRemoteMonitor.getSession();
      OpenBusLoginToken token =
        (OpenBusLoginToken) session.getAttribute("CLIENT_TOKEN");

      connectWithToken(token);
    }
    else if (loginObject instanceof UserPasswordLogin) {
      final User loggedUser = User.getLoggedUser();
      if (loggedUser.getId().equals(User.getAdminId())) {
        // Usurio Admin no pode ser autenticado no barramento
      }
      else {
        connectForUser(loggedUser);
      }
    }
    else if (loginObject instanceof PreLogin) {
      final String err =
        "No  possvel fazer a conexo com o barramento usando pre-login.";
      throw new CSBaseException(err);
    }
    else {
      final String err =
        "No foi possvel fazer a conexo com o barramento (forma de logindescohecida).";
      throw new CSBaseException(err);
    }
  }

  /**
   * Conexo usando um token.
   *
   * @param token token.
   * @throws CSBaseException
   */
  private void connectWithToken(final OpenBusLoginToken token)
    throws CSBaseException {
    try {
      SharedAuthSecret secret =
        getOpenBusContext().decodeSharedAuth(token.secret);
      this.connection.loginBySharedAuth(secret);
    }
    catch (InvalidEncodedStream e) {
      throw new CSBaseException(
        "Erro durante a autenticao no barramento usando toke: erro na decodificao do segredo .",
        e);
    }
    catch (AlreadyLoggedIn e) {
      //Nada a fazer
    }
    catch (InvalidLoginProcess e) {
      throw new CSBaseException(
        "Erro durante a autenticao no barramento usando toke: login invlido.",
        e);
    }
    catch (AccessDenied e) {
      throw new CSBaseException(
        "Erro durante a autenticao no barramento usando toke: acesso negado.",
        e);
    }
    catch (ServiceFailure e) {
      throw new CSBaseException(
        "Erro durante a autenticao no barramento usando toke: erro inesperado no barramento.",
        e);
    }
  }

  /**
   * Conexo para usurio.
   *
   * @param loggedUser usurio loggado.
   * @throws CSBaseException em caso de erro no OpenBus.
   */
  private void connectForUser(final User loggedUser) throws CSBaseException {
    final ClientRemoteMonitor clientRemoteMonitor =
      ClientRemoteMonitor.getInstance();
    final Authenticator authenticator = clientRemoteMonitor.getAuthenticator();
    final boolean isUserAuthenticator =
      authenticator instanceof UserPasswordAuthenticator;
    if (!isUserAuthenticator) {
      final String key = "no.user.authenticator.error";
      final String err = getString(key);
      throw new CSBaseException(err);
    }

    final String login = clientRemoteMonitor.getLogin();
    if (login == null) {
      final String key = "login.error";
      final String err = getString(key);
      throw new CSBaseException(err);
    }
    final String password = clientRemoteMonitor.getPassword();
    if (password == null) {
      final String key = "password.error";
      final String err = getString(key);
      throw new CSBaseException(err);
    }
    connectWithPassword(login, password);
  }

  /**
   * Conecta ao barramento a partir de um par usurio/senha.
   *
   * @param user O usurio.
   * @param password A senha.
   * @throws CSBaseException
   */
  private void connectWithPassword(String user, String password)
    throws CSBaseException {
    try {
      this.connection.loginByPassword(user, password.getBytes());
    }
    catch (AccessDenied e) {
      throw new CSBaseException(
        "Erro durante a autenticao no barramento usando toke: acesso negado.",
        e);
    }
    catch (AlreadyLoggedIn e) {
      // Nada a fazer
    }
    catch (ServiceFailure e) {
      throw new CSBaseException(
        "Erro durante a autenticao no barramento usando toke: erro inesperado no barramento.",
        e);
    }
  }

  /**
   * Consulta texto internacionalizado.
   *
   * @param key chave
   * @return texto.
   */
  private String getString(String key) {
    final String prefix = this.getClass().getSimpleName();
    return LNG.get(prefix + "." + key);
  }

  /**
   * Finaliza o acesso ao barramento.
   */
  private void shutdown() {
    try {
      this.connection.logout();
    }
    catch (ServiceFailure e) {
      // Nada a fazer
    }
    setInited(false);
  }

  /**
   * Obtm o token para login no OpenBus.
   *
   * @return o token
   *
   * @throws CSBaseException em caso de erro no OpenBus.
   */
  public OpenBusLoginToken getLoginToken() throws CSBaseException {
    if (!isInited()) {
      throw new CSBaseException("OpenBusAccessPoint no iniciado.");
    }
    String userName = (String) User.getLoggedUser().getId();

    SharedAuthSecret authSecret = null;
    try {
      authSecret = this.connection.startSharedAuth();
    }
    catch (ServiceFailure e) {
      throw new CSBaseException(
        "Erro ao obter token de login no barramento: erro inesperado no barramento.",
        e);
    }
    catch (TRANSIENT e) {
      throw new CSBaseException(String.format(
        "O barramento em %s:%s esta inacessvel no momento.", host, port), e);
    }
    catch (COMM_FAILURE e) {
      throw new CSBaseException(
        "Falha de comunicao ao acessar servios ncleo do barramento.", e);
    }
    catch (NO_PERMISSION e) {
      if (e.minor == NoLoginCode.value) {
        throw new CSBaseException(String.format(
          "No h um login de '%s' vlido no momento.", userName), e);
      }
    }

    return new OpenBusLoginToken(userName, getOpenBusContext()
      .encodeSharedAuth(authSecret));
  }

  /**
   * Cria uma cadeia de chamadas tendo como destino o servidor CSBase.
   *
   * @return a cadeia na representao de bytes
   *
   * @throws RemoteException em caso de erro de comunicao.
   * @throws CSBaseException em caso de erro no OpenBus.
   */
  public byte[] makeChain() throws RemoteException, CSBaseException {
    if (!isInited()) {
      throw new CSBaseException("OpenBusAccessPoint no iniciado.");
    }
    BusInfo busInfo = null;

    try {
      busInfo = ClientRemoteLocator.openBusService.getBusInfo();
    }
    catch (RemoteException e) {
      throw e;
    }

    String serverLoginId = busInfo.getLoginId();

    OpenBusContext context = null;
    try {
      context = getOpenBusContext();
    }
    catch (OperationFailureException e) {
      throw new CSBaseException(e);
    }
    CallerChain chain = null;
    try {
      chain = context.makeChainFor(serverLoginId);
    }
    catch (InvalidLogins e) {
      throw new CSBaseException(e);
    }
    catch (ServiceFailure e) {
      throw new CSBaseException(e);
    }

    return context.encodeChain(chain);
  }

  /**
   * Busca um servio publicado no barramento.
   *
   * @param properties as propriedades do servio
   *
   * @return o servio ou {@code null} caso no seja encontrado servio com as
   *         propriedades especificadas.
   *
   * @throws CSBaseException caso ocorra algum erro na busca do servio.
   */
  public IComponent findService(Properties properties) throws CSBaseException {
    ServiceProperty[] serviceProps = new ServiceProperty[properties.size()];
    int i = 0;
    for (String key : properties.stringPropertyNames()) {
      serviceProps[i] = new ServiceProperty(key, properties.getProperty(key));
      i++;
    }
    ServiceOfferDesc[] services;
    try {
      services = this.context.getOfferRegistry().findServices(serviceProps);
    }
    catch (Exception e) {
      throw new CSBaseException(
        "Falha ao buscar servios com as seguintes propriedades: \n"
          + properties.toString(), e);
    }

    for (int j = 0; j < services.length; j++) {
      IComponent service = services[j].service_ref;
      String offerId = "";
      for (ServiceProperty prop : services[j].properties) {
        if (prop.name.equals(OPENBUS_OFFER_ID_KEY)) {
          offerId = prop.value;
          break;
        }
      }
      try {
        if (!service._non_existent()) {
          return service;
        }
      }
      catch (TRANSIENT e) {
        System.err.println("WARNING: Oferta de servio encontrada (" + offerId
          + ") mas servio est inalcanvel.");
      }
      catch (COMM_FAILURE e) {
        System.err.println("WARNING: Oferta de servio encontrada (" + offerId
          + ") mas houve falha na comunicao com servio encontrado.");
      }
      catch (Exception e) {
        System.err.println("WARNING: Oferta de servio encontrada (" + offerId
          + ") mas houve falha ao testar a comunicao.");
        e.printStackTrace();
      }
    }
    return null;
  }

  /**
   * Inicia a sesso do barrramento com o tratador de eventos padro.
   *
   * @throws CSBaseException caso ocorra alguem erro ao obter a sesso.
   */
  public void initSession() throws CSBaseException {
    Class<SessionEventSinkPOA> sessionEventSinkClass = null;
    try {
      final String propSinkName = "session.event.sink.class";
      sessionEventSinkClass =
        this.configuration
          .<SessionEventSinkPOA> getOptionalClassProperty(propSinkName);
    }
    catch (ClassNotFoundException e) {
      final String key = "no.sink.class.error";
      final String err = getString(key);
      throw new CSBaseException(err);
    }
    if (sessionEventSinkClass == null) {
      initSession(OpenBusEventSink.class);
    }
    else {
      initSession(sessionEventSinkClass);
    }
  }

  /**
   * Inicia a sesso do barrramento.
   *
   * @param sessionEventSinkClass a classe do tratador de eventos da sesso.
   *
   * @throws CSBaseException caso ocorra alguem erro ao obter a sesso.
   */
  public void initSession(
    Class<? extends SessionEventSinkPOA> sessionEventSinkClass)
    throws CSBaseException {
    this.session = new OpenBusSession(sessionEventSinkClass);
  }

  /**
   * Obtm a sesso do barrramento.
   *
   * @return a sesso.
   */
  public OpenBusSession getSession() {
    return this.session;
  }

  /**
   * Obtm o ORB utilizado pelo servio.
   *
   * @return o ORB
   */
  public ORB getORB() {
    return orbThread.orb;
  }

  /**
   * Obtm o POA raiz.
   *
   * @return o POA raiz
   *
   * @throws OperationFailureException caso ocorra algum erro na obteno do POA
   */
  public POA getRootPOA() throws OperationFailureException {
    try {
      POA poa =
        POAHelper.narrow(getORB().resolve_initial_references("RootPOA"));
      if (!poa.the_POAManager().get_state().equals(State.ACTIVE)) {
        poa.the_POAManager().activate();
      }
      return poa;
    }
    catch (InvalidName e) {
      String msg = "Erro ao obter o POA.";
      throw new OperationFailureException(msg);
    }
    catch (AdapterInactive e) {
      String msg = "Erro ao obter o POA.";
      throw new OperationFailureException(msg);
    }
  }

  /**
   * Obtm o contexto de chamadas do barramento.
   *
   * @return o contexto
   *
   * @throws OperationFailureException
   */
  private OpenBusContext getOpenBusContext() throws OperationFailureException {
    OpenBusContext context = null;
    try {
      context =
        (OpenBusContext) getORB().resolve_initial_references("OpenBusContext");
    }
    catch (InvalidName e) {
      throw new OperationFailureException(
        "Erro ao obter o contexto da conexo com o o barramento.");
    }

    return context;
  }

  /**
   * Responsvel por executar e finalizar o ORB.
   *
   * @author Tecgraf/PUC-Rio
   */
  private static class ORBThread extends Thread {
    /**
     * O ORB.
     */
    private final ORB orb;

    /**
     * Cria a <i>thread</i> para execuo do ORB.
     *
     * @param orb O ORB.
     */
    ORBThread(ORB orb) {
      this.orb = orb;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      orb.run();
    }
  }
}

/**
 * Callback chamada quando o login  invalidado.
 *
 * @author Tecgraf
 */
class OnInvalidLogin implements InvalidLoginCallback {

  /**
   * {@inheritDoc}
   */
  @Override
  public void invalidLogin(Connection conn, LoginInfo loginInfo) {
    try {
      OpenBusAccessPoint.getInstance().connect();
    }
    catch (Exception e) {
      System.err.println("bus login failure: " + e);
    }
  }
}
