package csbase.client.preferences;

import java.awt.Window;
import java.io.ByteArrayOutputStream;
import java.io.StringReader;
import java.rmi.RemoteException;
import java.util.Map.Entry;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.transform.stream.StreamSource;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.RemoteTask;
import csbase.client.preferences.model.Category;
import csbase.client.preferences.model.Preference;
import csbase.exception.CSBaseRuntimeException;
import csbase.exception.ServiceFailureException;
import csbase.logic.SharedObject;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.NotificationServiceInterface;
import csbase.remote.SharedObjectServiceInterface;

/**
 * Classe encarregada de se comunicar com o servidor e fazer a
 * leitura/persistncia das preferncias do usurio.
 * 
 * @see PreferenceManager DefinitionLoader PreferenceCategory
 * 
 * @author Tecgraf
 */
class PreferencePersistence {

  /** Diretrio no servidor onde as preferncias do usuario sero salvas. */
  private static String USER_PREFERENCE_CATEGORY = "UserPreferences";

  /**
   * Nome do arquivo que armazena as preferncias persistido em
   * {@link SharedObject}.
   */
  private static String PREFERENCES = "preferences";

  /** Path do modelo da rvore de preferncias usada pelo JAXB. */
  private static String CLASSPATH = "csbase.client.preferences.model";

  /** Referncia para a instncia nica desta classe. */
  private static PreferencePersistence instance;

  /**
   * Retorna a instncia nica desta classe.
   * 
   * @return instncia nica desta classe.
   */
  static PreferencePersistence getInstance() {
    if (instance == null) {
      instance = new PreferencePersistence();
    }
    return instance;
  }

  /**
   * Carrega as preferncias do usurio e seta a rvore recebida por parmetro
   * com os respectivos valores.
   * 
   * @param parent janela principal que est manipulando as preferncias.
   * @param root raz da rvore de preferncias.
   */
  void load(Window parent, PreferenceCategory root) {
    String preferenceAsStr = loadSharedObject(parent);
    Category tree = xml2Tree(preferenceAsStr);
    setUserPreferences(tree, root);
  }

  /**
   * Salva a rvore de preferncias do usurio.
   * 
   * @param parent janela principal que est manipulando as preferncias.
   * @param root raz da rvore de preferncias.
   */
  void save(Window parent, PreferenceCategory root) {
    Category tree = new Category();
    convert2Tree(tree, root);
    String xml = tree2Xml(tree);
    saveSharedObject(xml, parent);
  }

  /**
   * Mtodo auxiliar que constroi recursivamente a rvore do JAXB. Este mtodo 
   * usado na escrita do XML de preferncias no servidor.
   * 
   * @param category objeto alvo.
   * @param pc objeto fonte.
   */
  private void convert2Tree(Category category, PreferenceCategory pc) {
    category.setId(pc.getId());

    for (Entry<String, PreferenceValue<?>> entry : pc.getPreferencesMap()
      .entrySet()) {
      String prefName = entry.getKey();
      PreferenceValue<?> prefValue = entry.getValue();

      Preference p = new Preference();
      p.setName(prefName);
      p.setValue(prefValue.toString());
      p.setType(prefValue.getClassName());
      p.setPolicy(prefValue.getPolicy().toString());

      category.getPreferences().add(p);
    }

    for (PreferenceCategory aux : pc.getCategories()) {
      Category c = new Category();
      convert2Tree(c, aux);
      category.getCategories().add(c);
    }
  }

  /**
   * Percorre recursivamente pela rvore JAXB e troca os valores da rvore de
   * preferncias do usurio.
   * 
   * @param tree rvore JAXB.
   * @param node rvore de preferncias do usurio.
   */
  private void setUserPreferences(Category tree, PreferenceCategory node) {
    for (Preference p : tree.getPreferences()) {

      PreferenceValue<?> pv = node.getPreference(p.getName());
      if (pv != null && pv.getClassName().equals(p.getType())) {
        pv.setValueAsStr(p.getValue());
      }
    }

    for (Category c : tree.getCategories()) {
      PreferenceCategory pc = node.getCategory(c.getId());
      if (pc != null) {
        setUserPreferences(c, pc);
      }
    }
  }

  /**
   * Dado a string que representa as preferncias em XML, retorna a rvore de
   * objetos equivalente. Este mtodo  usado na carga das preferncias do
   * usurio que esto salvas no servidor.
   * 
   * @param preferences preferncias em XML.
   * @return rvore sinttica equivalente.
   */
  private Category xml2Tree(String preferences) {
    try {
      JAXBContext context = JAXBContext.newInstance(CLASSPATH);

      Unmarshaller unmarshaller = context.createUnmarshaller();
      StreamSource ss = new StreamSource(new StringReader(preferences));
      JAXBElement<Category> result = unmarshaller.unmarshal(ss, Category.class);
      return result.getValue();
    }
    catch (JAXBException e) {
      throw new PreferenceException(
        "No foi possvel construir as preferncias dado o XML:" + preferences,
        e);
    }
  }

  /**
   * Converte a rvore de objetos de preferncias em um XML. Este mtodo  usado
   * na escrita do XML de preferncias no servidor.
   * 
   * @param root rvore que representa as preferncias do usurio.
   * 
   * @return String (XML) que representa as preferncias.
   */
  private String tree2Xml(Category root) {
    try {
      JAXBContext context = JAXBContext.newInstance(CLASSPATH);

      Marshaller marshaller = context.createMarshaller();
      marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT,
        new Boolean(true));

      ByteArrayOutputStream os = new ByteArrayOutputStream();

      marshaller.marshal(root, os);

      return os.toString();
    }
    catch (JAXBException e) {
      throw new PreferenceException(
        "No foi possvel converter o objeto de preferncias para XML.", e);
    }
  }

  /**
   * Faz a carga da preferncia do usurio persistido como {@link SharedObject}.
   * 
   * @param parent janela principal que est manipulando as preferncias.
   * 
   * @return String que representa as preferncias do usurio em XML.
   */
  private String loadSharedObject(Window parent) {
    RemoteTask<String> task = new ReadSharedObjectTask();

    boolean result =
      task.execute(parent, getString("title"), getString("load"));
    if (result) {
      return task.getResult();
    }
    throw new PreferenceException(
      "No foi possvel carregar o sharedObject de preferncias.");
  }

  /**
   * Faz a persistncia da preferncia do usurio como {@link SharedObject}.
   * 
   * @param preferences String que representa a preferncia do usurio em
   *        formato XML.
   * @param parent janela principal que est manipulando as preferncias.
   */
  private void saveSharedObject(final String preferences, Window parent) {
    final SharedObjectServiceInterface sharedObjectService =
      ClientRemoteLocator.sharedObjectService;

    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        SharedObject sharedObject =
          new SharedObject(USER_PREFERENCE_CATEGORY, User.getLoggedUser()
            .getId(), PREFERENCES, false, preferences);
        sharedObjectService.saveSharedObject(sharedObject);
      }
    };

    task.execute(parent, getString("title"), getString("save"));
  }

  /**
   * Mtodo auxiliar que obtem a string internacionalizada.
   * 
   * @param key chave da string.
   * @return string internacionalizada.
   */
  private String getString(String key) {
    return LNG.get(getClass().getSimpleName() + "." + key);
  }

  /** Construtor privador. */
  private PreferencePersistence() {
  }

  /**
   * Task que se encarrega de fazer a leitura do {@link SharedObject} que
   * armazena as preferncias do usurio. Esta classe tambm se encarrega de
   * implementar a estratgia de auto-healing quendo h um erro na leitura do
   * arquivo, por exemplo, arquivo corrompido.
   * 
   * @author Tecgraf
   */
  private class ReadSharedObjectTask extends RemoteTask<String> {

    /** {@inheritDoc} */
    @Override
    protected void performTask() throws Exception {
      SharedObjectServiceInterface sharedObjectService =
        ClientRemoteLocator.sharedObjectService;

      SharedObject sharedObject =
        sharedObjectService.getSharedObject(USER_PREFERENCE_CATEGORY, User
          .getLoggedUser().getId(), PREFERENCES);

      if (sharedObject == null) {
        setResult(null);
      }
      else {
        setResult(sharedObject.getContents().toString());
      }
    }

    /** {@inheritDoc} */
    @Override
    protected void handleServerError(CSBaseRuntimeException cserror) {
      if (cserror instanceof ServiceFailureException) {
        try {
          SharedObjectServiceInterface sharedObjectService =
            ClientRemoteLocator.sharedObjectService;

          sharedObjectService.removeSharedObject(USER_PREFERENCE_CATEGORY,
            PREFERENCES);

          // Notifica o usurio que suas preferncias foram resetadas.
          NotificationServiceInterface notificationService =
            ClientRemoteLocator.notificationService;

          Object[] userIds = new Object[] { User.getLoggedUser().getId() };

          notificationService.notifyTo(userIds,
            LNG.get("notification.data.reset.preference"), false, false);
        }
        catch (RemoteException e) {
          super.handleError(e);
        }
      }
      else {
        super.handleServerError(cserror);
      }
    }
  };
}
