/*
 * $id$
 */
package csbase.client.project;

import java.awt.Window;

import javax.swing.JOptionPane;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.desktop.ProjectFileNameAndTypeFilter;
import csbase.client.kernel.ClientException;
import csbase.client.project.tasks.GetChildFromNameTask;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileFilter;
import csbase.logic.ProjectFileType;
import csbase.logic.filters.ProjectFileCompositeFilter;
import csbase.logic.filters.ProjectFileCompositeOrFilter;
import csbase.logic.filters.ProjectFileDirectoryFilter;
import csbase.logic.filters.ProjectFileNotDirectoryFilter;
import csbase.logic.filters.ProjectFileTypeFilter;

/**
 * Essa classe implementa um navegador para salvamento de um arquivo em um
 * projeto. Pode-se configurar se o navegador vai permitir, por exemplo, a
 * seleo somente de arquivos na rvore e no de diretrios.
 */
public final class ProjectFileChooserSave extends ProjectFileChooser {
  /** Ttulo do chooser */
  private static final String TITLE = LNG.get("PRJ_FILE_CHOOSER_SAVE_TITLE");

  /** Extenso a ser inserida no nome do arquivo, caso j no a tenha. */
  private final String forceExtension;

  /**
   * Contri um navegador para salvamento de um arquivo em um projeto, de acordo
   * com o modo de seleo especificado para o navegador.
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param mode modo de seleo no navegador [FILE_ONLY ou DIRECTORY_ONLY].
   *        (@see ProjectFileChooser)
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserSave(Window owner, CommonClientProject project,
    int mode) throws ClientException {
    this(owner, project, mode, null, null, null, false);
  }

  /**
   * Contri um navegador para salvamento de um arquivo em um projeto, de acordo
   * com o modo de seleo especificado para o navegador.
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param mode modo de seleo no navegador [FILE_ONLY ou DIRECTORY_ONLY].
   *        (@see ProjectFileChooser)
   * @param defaultFileType tipo de arquivo default no navegador
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserSave(Window owner, CommonClientProject project,
    int mode, String defaultFileType) throws ClientException {
    this(owner, project, mode, defaultFileType, null, null, false);
  }

  /**
   * Contri um navegador para salvamento de um arquivo em um projeto, de acordo
   * com o modo de seleo especificado para o navegador. Nesse construtor, 
   * possvel especificar um valor default para o nome do arquivo a ser salvo.
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param mode modo de seleo no navegador [FILE_ONLY ou DIRECTORY_ONLY].
   *        (@see ProjectFileChooser)
   * @param defaultFileType tipo de arquivo default no navegador
   * @param defaultName valor sugerido inicialmente para o nome do arquivo
   * @param fileExtension - extenso a ser includa no nome do arquivo, caso ele
   *        no a tenha. Null para ignorar extenso.
   * @param useFilter se um panel de filtro deve ser exibido
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserSave(Window owner, CommonClientProject project,
    int mode, String defaultFileType, String defaultName, String fileExtension,
    boolean useFilter) throws ClientException {
    this(owner, project, mode, new String[] { defaultFileType },
      defaultFileType, defaultName, fileExtension, useFilter);
  }

  /**
   * Contri um navegador para salvamento de um arquivo em um projeto, de acordo
   * com o modo de seleo especificado para o navegador. Nesse construtor, 
   * possvel especificar um valor default para o nome do arquivo a ser salvo ou
   * diretrio corrente para o navegador.
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param mode modo de seleo no navegador [FILE_ONLY ou DIRECTORY_ONLY].
   *        (@see ProjectFileChooser)
   * @param defaultFileType tipo de arquivo default no navegador
   * @param defaultName valor sugerido inicialmente para o nome do arquivo, pode
   *        ser null
   * @param currentDirectory diretrio corrente para abertura do navegador, pode
   *        ser null
   * @param fileExtension - extenso a ser includa no nome do arquivo, caso ele
   *        no a tenha. Null para ignorar extenso.
   * @param useFilter se um panel de filtro deve ser exibido
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserSave(Window owner, CommonClientProject project,
    int mode, String defaultFileType, String defaultName,
    String[] currentDirectory, String fileExtension, boolean useFilter)
    throws ClientException {
    this(owner, project, mode, new String[] { defaultFileType },
      defaultFileType, defaultName, currentDirectory, fileExtension, useFilter);
  }

  /**
   * Contri um navegador para salvamento de um arquivo em um projeto, de acordo
   * com o modo de seleo especificado para o navegador. Nesse construtor, 
   * possvel especificar um valor default para o nome do arquivo a ser salvo.
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param mode modo de seleo no navegador [FILE_ONLY ou DIRECTORY_ONLY].
   *        (@see ProjectFileChooser)
   * @param fileTypes os tipos de arquivos aceitos
   * @param defaultFileType tipo de arquivo default no navegador
   * @param defaultName valor sugerido inicialmente para o nome do arquivo
   * @param fileExtension - extenso a ser includa no nome do arquivo, caso ele
   *        no a tenha. Null para ignorar extenso.
   * @param useFilter se um panel de filtro deve ser exibido
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserSave(Window owner, CommonClientProject project,
    int mode, String[] fileTypes, String defaultFileType, String defaultName,
    String fileExtension, boolean useFilter) throws ClientException {
    this(owner, project, mode, fileTypes, defaultFileType, defaultName, null,
      fileExtension, useFilter);
  }

  /**
   * Contri um navegador para salvamento de um arquivo em um projeto, de acordo
   * com o modo de seleo especificado para o navegador. Nesse construtor, 
   * possvel especificar um valor default para o nome do arquivo a ser salvo ou
   * diretrio corrente para o navegador.
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param mode modo de seleo no navegador [FILE_ONLY ou DIRECTORY_ONLY].
   *        (@see ProjectFileChooser)
   * @param fileTypes os tipos de arquivos aceitos
   * @param defaultFileType tipo de arquivo default no navegador
   * @param defaultName valor sugerido inicialmente para o nome do arquivo
   * @param currentDirectory diretrio corrente para abertura do navegador, pode
   *        ser null
   * @param fileExtension - extenso a ser includa no nome do arquivo, caso ele
   *        no a tenha. Null para ignorar extenso.
   * @param useFilter se um panel de filtro deve ser exibido
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserSave(Window owner, CommonClientProject project,
    int mode, String[] fileTypes, String defaultFileType, String defaultName,
    String[] currentDirectory, String fileExtension, boolean useFilter)
    throws ClientException {
    super(owner, ProjectFileChooserSave.TITLE, project, false, mode,
      defaultFileType, getFilter(mode, fileTypes), getFilter(mode, fileTypes),
      true, defaultName, currentDirectory, useFilter);

    if (mode == FILE_AND_DIRECTORY) {
      throw new IllegalArgumentException(
        "Erro: ProjectFileChooserSave no deve"
          + " ser usado com modo = FILE_AND_DIRECTORY");
    }

    this.forceExtension = fileExtension;
    this.dialog.setVisible(true);
  }

  /**
   * Cria um filtro visual de um determinado tipo de arquivo, de acordo com o
   * modo de seleo especificado para o navegador.
   * 
   * @param mode modo de seleo no navegador
   * @param fileTypes os tipos de arquivo permitidos
   * @return o filtro criado
   */
  private static ProjectFileFilter getFilter(int mode, String[] fileTypes) {
    ProjectFileCompositeFilter projectFileCompositeFilter =
      new ProjectFileCompositeOrFilter();
    projectFileCompositeFilter.addChild(ProjectFileDirectoryFilter
      .getInstance());
    if ((mode == ProjectFileChooser.FILE_ONLY)
      || (mode == ProjectFileChooser.FILE_AND_DIRECTORY)) {
      if (fileTypes == null || fileTypes.length == 0) {
        projectFileCompositeFilter.addChild(ProjectFileNotDirectoryFilter
          .getInstance());
      }
      else {
        projectFileCompositeFilter
          .addChild(new ProjectFileTypeFilter(fileTypes));
      }
    }
    return projectFileCompositeFilter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getVisualFilter() {
    return ProjectFileChooserSave.getFilter(this.mode, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getVisualFilter(String fileType) {
    return ProjectFileChooserSave.getFilter(this.mode,
      new String[] { fileType });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getVisualFilter(final String nameFilter,
    final String fileType) {
    if (nameFilter == null || nameFilter.isEmpty()) {
      return this.getVisualFilter(fileType);
    }
    ProjectFileCompositeFilter projectFileCompositeFilter =
      new ProjectFileCompositeOrFilter();
    projectFileCompositeFilter.addChild(ProjectFileDirectoryFilter
      .getInstance());
    if ((mode == ProjectFileChooser.FILE_ONLY)
      || (mode == ProjectFileChooser.FILE_AND_DIRECTORY)) {
      projectFileCompositeFilter.addChild(new ProjectFileNameAndTypeFilter(
        nameFilter, fileType));
    }
    return projectFileCompositeFilter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getSelectionFilter() {
    return ProjectFileChooserSave.getFilter(this.mode, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getSelectionFilter(String fileType) {
    return ProjectFileChooserSave.getFilter(this.mode,
      new String[] { fileType });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getSelectionFilter(final String nameFilter,
    final String fileType) {
    if (nameFilter == null || nameFilter.isEmpty()) {
      return this.getSelectionFilter(fileType);
    }
    ProjectFileCompositeFilter projectFileCompositeFilter =
      new ProjectFileCompositeOrFilter();
    projectFileCompositeFilter.addChild(ProjectFileDirectoryFilter
      .getInstance());
    if ((mode == ProjectFileChooser.FILE_ONLY)
      || (mode == ProjectFileChooser.FILE_AND_DIRECTORY)) {
      projectFileCompositeFilter.addChild(new ProjectFileNameAndTypeFilter(
        nameFilter, fileType));
    }
    return projectFileCompositeFilter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean handleAction() {
    // boto 'confirmar' foi acionado.  garantido que temos uma seleo na
    // rvore e contedo no campo de texto [para arquivos]:
    // lembretes:
    // - 'save' permite apenas seleo simples, no mltipla
    // - 'save' no contempla FILE_AND_DIRECTORY
    ClientProjectFile selectedFile = this.getSelectedFile();
    if (mode == DIRECTORY_ONLY) {
      // diretrios: ver apenas a seleo da rvore ... campo de texto est
      // escondido mesmo, e o filtro deve mostrar apenas diretrios
      setSelection(selectedFile);
      return true;
    }

    // seleo de arquivo ... verificar campo de texto, cujo contedo 
    // considerado um nome de arquivo escolhido para o diretrio correspondente
    //  seleco atual na rvore
    String name = this.fileNameText.getText().trim();

    if (!ClientUtilities.isValidFileName(name)) {
      showFilenameError();
      return false;
    }

    ClientProjectFile targetDirectory;
    if (selectedFile.isDirectory()) {
      targetDirectory = selectedFile;
    }
    else {
      targetDirectory = selectedFile.getParent();
    }

    // verificar existncia do arquivo:
    ClientProjectFile newFile = null;

    if (forceExtension != null) {
      name = FileUtils.addFileExtension(name, forceExtension);
    }

    // se o texto for um diretrio dentro do diretrio atualmente selecionado,
    // selecionamos tal diretrio na rvore ...
    newFile = GetChildFromNameTask.runTask(targetDirectory, name);
    if (newFile != null) {
      // j existe ... se diretrio, entramos nele ... se arquivo, pedimos
      // confirmao de sobrescrita
      if (newFile.isDirectory()) {
        this.projectTree.setSelectedFile(newFile);
        return false;
      }
      if (confirmOverwrite(newFile)) {
        setSelection(newFile);
        return true;
      }
      return false;
    }

    // arquivo no existe ... criar objeto para ele e retornar ok
    setSelection(targetDirectory, name);
    return true;
  }

  /**
   * Atribui a seleo.
   * 
   * @param file o arquivo selecionado.
   */
  private void setSelection(ClientProjectFile file) {
    this.selectedPaths = new ProjectTreePath[1];
    if (file != null) {
      this.selectedPaths[0] = new ProjectTreePath(file);
    }
    else {
      this.selectedPaths[0] = null;
    }
  }

  /**
   * Atribui a seleo.
   * 
   * @param targetDirectory diretrio do arquivo selecionado.
   * @param filename nome do arquivo selecionado.
   */
  private void setSelection(ClientProjectFile targetDirectory, String filename) {
    String[] dirPath = targetDirectory.getPath();
    String[] newPath = new String[dirPath.length + 1];
    System.arraycopy(dirPath, 0, newPath, 0, dirPath.length);
    newPath[newPath.length - 1] = filename;
    this.selectedPaths = new ProjectTreePath[1];
    this.selectedPaths[0] = new ProjectTreePath(newPath, targetDirectory);
  }

  /**
   * Mostra uma janela de erro para caracteres invlidos no nome do arquivo
   * selecionado.
   */
  private void showFilenameError() {
    StandardErrorDialogs.showErrorDialog(this.dialog,
      ProjectFileChooserSave.TITLE, LNG
        .get("PRJ_PROJECT_FILE_NAME_CHARACTERES_ERROR"));
  }

  /**
   * Pergunta ao usurio se ele deseja substituir o item j existente.
   * 
   * @param newFile Arquivo selecionado para possvel sobrescrita.
   * 
   * @return TRUE se o usurio clicou em 'SIM', FALSE caso contrrio.
   */
  private boolean confirmOverwrite(ClientProjectFile newFile) {
    if (newFile.isLocked()) {
      StandardDialogs.showErrorDialog(this.dialog, LNG
        .get("PRJ_FILE_CHOOSER_ERROR")
        + " - " + ProjectFileChooserSave.TITLE, LNG
        .get("PRJ_FILE_CHOOSER_OVERWRITE_LOCKED_FILE_ERROR"));
      return false;
    }

    int option =
      StandardDialogs.showYesNoDialog(this.dialog, LNG
        .get("PRJ_FILE_CHOOSER_INFO")
        + " - " + ProjectFileChooserSave.TITLE, LNG
        .get("PRJ_FILE_CHOOSER_OVERWRITE_WARNING"));
    return option == JOptionPane.YES_OPTION;
  }

  /**
   * Devolve o cdigo do tipo de arquivo selecionado. Caso o tipo seja "Todos os
   * Arquivos", retorna o cdigo do tipo desconhecido
   * {@link ProjectFileType#UNKNOWN}.
   * 
   * @return Tipo de arquivo selecionado neste dilogo.
   * @see ProjectFileType#getCode()
   */
  public String getFileType() {
    Object selectedFileType = this.fileTypeList.getSelectedItem();
    if (selectedFileType instanceof String) {
      return ProjectFileType.getFileType(ProjectFileType.UNKNOWN).getCode();
    }
    return ((ProjectFileType) selectedFileType).getCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void makeSelection(ClientProjectFile[] projectFileArray) {
    if (projectFileArray.length == 0) {
      this.fileNameText.setText("");
      this.selectedPaths = null;
    }
    else {
      ClientProjectFile selectedFile = projectFileArray[0];
      if (!selectedFile.isDirectory()) {
        this.fileNameText.setText(selectedFile.getName());
      }
      this.selectedPaths =
        new ProjectTreePath[] { new ProjectTreePath(selectedFile) };
    }
  }
}
