package csbase.command;

import csbase.logic.ClientProjectFile;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.FileParameterValue;
import csbase.server.Server;
import csbase.server.plugin.service.IServiceManager;
import csbase.server.plugin.service.commandpersistenceservice.ICommandInfo;
import csbase.server.plugin.service.commandpersistenceservice.ICommandStatusListener;
import csbase.server.services.commandpersistenceservice.CommandPersistenceService;
import csbase.server.services.projectservice.ProjectService;
import tecgraf.javautils.core.io.FileUtils;

import java.util.*;

/**
 * Created by SINAPAD.
 *
 * Updated by mjulia on 30/04/15.
 */
public class DataProvenanceHandler implements ICommandStatusListener {

    private CommandPersistenceService commandPersistenceService;
    private ProjectService projectService;


    public DataProvenanceHandler(IServiceManager serviceManager) {
        commandPersistenceService = (CommandPersistenceService)serviceManager.getService(CommandPersistenceService.SERVICE_NAME);
        projectService = (ProjectService) serviceManager.getService(ProjectService.SERVICE_NAME);
    }

    @Override
    public void statusChanged(ICommandInfo commandInfo) {
        final AlgorithmConfigurator configurator =
                commandPersistenceService.readAlgorithmConfigurator(
                        commandInfo.getProjectId(), commandInfo.getCommandId());
        if (commandInfo.getStatus().equals(ICommandInfo.CommandStatus.UPLOADING)) {
            Collection<FileParameterValue> inputFileCollection =
                    configurator.getInputFiles();
            Collection<FileParameterValue> inputDirColletion =
                    configurator.getInputDirectories();
            Collection<FileParameterValue> inputs = new TreeSet<FileParameterValue>();
            inputs.addAll(inputDirColletion);
            inputs.addAll(inputFileCollection);
            createCommandInputHistory(commandInfo, inputs);
            return;
        }
        if (commandInfo.getStatus().equals(ICommandInfo.CommandStatus.DOWNLOADING)
                && projectService.existsProject(commandInfo.getProjectId())) {
            Collection<FileParameterValue> outputFileCollection =
                    configurator.getOutputFiles();
            Collection<FileParameterValue> outputDirColletion =
                    configurator.getOutputDirectories();
            Collection<FileParameterValue> outputs =
                    new TreeSet<FileParameterValue>();
            outputs.addAll(outputDirColletion);
            outputs.addAll(outputFileCollection);
            createCommandOutputHistory(commandInfo, outputs);
        }
    }

    /**
     *
     * Cria o diretrio FILES/INPUT_FILES dentro de .cmds para guardar todos os
     * arquivos de entrada da execuo.
     *
     * @param commandInfo Informao do comando usada para obter o projeto e
     *        identificador
     * @param inputFileCollection Lista de arquivos de entrada utilizados pelo
     *        comando
     */
    private void createCommandInputHistory(ICommandInfo commandInfo,
                                           Collection<FileParameterValue> inputFileCollection) {
        Object projectId = commandInfo.getProjectId();
        // cria o diretorio de arquivos de entrada para o historico
        String commandId = FileUtils.fixDirectoryName(commandInfo.getCommandId());
        String[] filesDir = { ".cmds", commandId, "FILES" };
        projectService.createDirectory(projectId, filesDir);
        String[] historyPath = { ".cmds", commandId, "FILES", "INPUT_FILES" };
        projectService.createDirectory(projectId, historyPath);

        // cria o diretorio de arquivos de entrada para o historico
        createCommandHistory(commandInfo, inputFileCollection, historyPath);
    }

    /**
     * Cria o diretrio FILES/OUTPUT_FILES dentro de .cmds para guardar todos os
     * arquivos de sada da execuo.
     *
     * @param commandInfo Informao do comando usada para obter o projeto e
     *        identificador
     * @param outputFileCollection Lista de arquivos de sada gerados pela
     *        execuo do comando
     */
    private void createCommandOutputHistory(ICommandInfo commandInfo,
                                            Collection<FileParameterValue> outputFileCollection) {
        String commandId = FileUtils.fixDirectoryName(commandInfo.getCommandId());
        Object projectId = commandInfo.getProjectId();
        // cria o diretorio de arquivos do historico
        String[] historyPath = { ".cmds", commandId, "FILES", "OUTPUT_FILES" };
        if (!projectService.existsFile(projectId, historyPath)) {
            projectService.createDirectory(projectId, historyPath);
        }
        // cria o diretorio de arquivos de sada para o histrico
        createCommandHistory(commandInfo, outputFileCollection, historyPath);
    }

    /**
     * Cria os arquivos e diretrios utilizados como entrada na rea de histicos
     * dentro do diretrio .cmds.
     *
     * @param commandInfo Informao do comando usada para obter o projeto e
     *        identificador
     * @param fileCollection Os arquivos e diretrios que sero copiados
     * @param historyPath Caminho onde os arquivos sero copiados
     */
    private void createCommandHistory(ICommandInfo commandInfo,
                                      Collection<FileParameterValue> fileCollection, String[] historyPath) {
        if ((fileCollection == null) || (fileCollection.size() == 0)) {
            return;
        }
        Object projectId = commandInfo.getProjectId();

        int numFiles = fileCollection.size();
        Iterator<FileParameterValue> fileIterator = fileCollection.iterator();
        for (int i = 0; i < numFiles; i++) {
            FileParameterValue file = fileIterator.next();
            String[] filePath = file.getPathAsArray();

            if (Arrays.binarySearch(filePath, ".cmds") >= 0) {
                continue;
            }

            String[] fullPath = new String[historyPath.length + filePath.length];
            System.arraycopy(historyPath, 0, fullPath, 0, historyPath.length);
            System.arraycopy(filePath, 0, fullPath, historyPath.length,
                    filePath.length);
            try {
                String[] dirPath = new String[fullPath.length - 1];
                for (int j = 0; j < fullPath.length - 1; j++) {
                    dirPath[j] = fullPath[j];
                }
                if (file.isDirectory()) {
                    if (!projectService.existsFile(projectId, fullPath)) {
                        projectService.createDirectory(projectId, fullPath);
                    }
                    ClientProjectFile[] children =
                            projectService.getChildren(projectId, filePath, true);
                    Collection<FileParameterValue> childrenList =
                            new ArrayList<FileParameterValue>();
                    for (ClientProjectFile c : children) {
                        String[] path = c.getPath();
                        FileParameterValue fpv =
                                new FileParameterValue(FileUtils.joinPath('/', path));
                        childrenList.add(fpv);
                    }
                    createCommandHistory(commandInfo, childrenList, historyPath);
                }
                else {
                    if (!projectService.existsFile(projectId, dirPath)) {
                        projectService.createDirectory(projectId, dirPath);
                    }
                    if (projectService.existsFile(projectId, fullPath)) {
                        projectService.removeFile(projectId, fullPath);
                    }
                    projectService.copyFile(projectId, filePath, dirPath);
                }
            }
            catch (Exception e) {
                Server.logSevereMessage(
                        "Falha na criao do arquivo " + file.getPath(), e);
            }
        }
    }
}
