/**
 * $Id$
 */
package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import csbase.logic.ProjectFileType;
import csbase.logic.algorithms.CommandLineBuilder;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.EnvironmentVariable;
import csbase.logic.algorithms.FileParameterValue;
import csbase.logic.algorithms.parameters.validators.FileParameterValidator;

/**
 * <p>
 * Parmetro do tipo arquivo.
 * </p>
 */
public abstract class FileParameter extends SimpleParameter<FileParameterValue> {

  /**
   * Chave utilizada para informar o <b>caminho</b> do arquivo no mapa de
   * parmetros dos mtodos de importao ({@link #importValue(Map)}) e
   * exportao ({@link #exportValue()}).
   */
  public static final String PATH = ".path";

  /**
   * Chave utilizada para informar o <b>tipo</b> do arquivo no mapa de
   * parmetros dos mtodos de importao ({@link #importValue(Map)}) e
   * exportao ({@link #exportValue()}).
   */

  public static final String TYPE = ".type";

  /**
   * Indica se este parmetro pode aceitar pipe
   * {@value FileParameterPipeAcceptance#TRUE}, no aceita pipe
   * {@value FileParameterPipeAcceptance#FALSE} ou *s* aceita pipe
   * {@value FileParameterPipeAcceptance#ALWAYS}.
   */
  private FileParameterPipeAcceptance usesPipe;

  /**
   * O tipo de arquivo que este parmetro filtra. Se ele se importa com o tipo
   * do arquivo, ele valer {@code null}.
   */
  private String fileType;

  /**
   * Indica se este parmetro est ({@code true}) ou no ({@code false}) est
   * participando de uma conexo entre ns de um fluxo de algoritmos.
   */
  private boolean hasLink;

  /**
   * Associao com os observador do parmetro do tipo arquivo.
   */
  private transient List<FileParameterListener> listeners;

  /**
   * O modo de funcionamento atual (apenas arquivos, apenas diretrios ou
   * ambos).
   */
  private FileParameterMode mode;

  /**
   * Indica se um painel para filtro deve ser exibido.
   */
  private boolean usesFilter;

  /**
   * Indica se o arquivo deve existir.
   */
  private final boolean mustExist;

  /**
   * Cria um parmetro do tipo arquivo.
   * 
   * @param name O nome deste parmetro (No aceita {@code null}).
   * @param label O rtulo deste parmetro (No aceita {@code null}).
   * @param description A descrio deste parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o valor do parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param fileType O tipo dos arquivos aceitos neste parmetro (Aceita
   *        {@code null}).
   * @param mode O modo de funcionamento deste parmetro (No aceita
   *        {@code null}).
   * @param usesPipe Indica se este parmetro pode aceitar pipe
   *        {@value FileParameterPipeAcceptance#TRUE}, no aceita pipe
   *        {@value FileParameterPipeAcceptance#FALSE} ou *s* aceita pipe
   *        {@value FileParameterPipeAcceptance#ALWAYS}.
   * @param usesFilter Indica se o painel de filtro deve ser exibido.
   * @param mustExist Indica se o arquivo deve existir.
   */
  protected FileParameter(String name, String label, String description,
    FileParameterValue defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, String fileType, FileParameterMode mode,
    FileParameterPipeAcceptance usesPipe, boolean usesFilter, boolean mustExist) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern);
    this.listeners = new LinkedList<FileParameterListener>();
    this.fileType = fileType;
    this.hasLink = false;
    this.usesFilter = usesFilter;
    this.mustExist = mustExist;
    setUsesPipe(usesPipe);
    setMode(mode);
  }

  /**
   * Adiciona um observador de parmetros do tipo arquivo a este parmetro.
   * 
   * @param listener O observador (No aceita {@code null}).
   */
  public final void addFileParameterListener(FileParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * Indica se este arquivo pode utilizar <i>pipes</i>.
   * 
   * @return {@value FileParameterPipeAcceptance#TRUE} se este parmetro pode
   *         aceitar pipe, {@value FileParameterPipeAcceptance#FALSE} se no
   *         aceita pipe, ou {@value FileParameterPipeAcceptance#ALWAYS} se *s*
   *         aceita pipe.
   */
  public final FileParameterPipeAcceptance usesPipe() {
    return this.usesPipe;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Object> exportValue() {
    Map<String, Object> valuesByName = new HashMap<String, Object>();
    Collection<Map<String, String>> exportableValues =
      new LinkedList<Map<String, String>>();
    FileParameterValue file = getValue();
    if (file != null) {
      Map<String, String> exportableValue = new HashMap<String, String>();
      exportableValue.put(PATH, file.getPath());
      exportableValue.put(TYPE, file.getType());
      exportableValues.add(Collections.unmodifiableMap(exportableValue));
      valuesByName.put(getName(), Collections
        .unmodifiableCollection(exportableValues));
    }
    return Collections.unmodifiableMap(valuesByName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String getCommandValue(CommandLineContext context) {

    FileParameterValue file = getValue();
    if (file == null) {
      return null;
    }
    StringBuilder commandValue = new StringBuilder();
    char fileSeparator = context.getFileSeparator();
    if (file.getPath().startsWith(CommandLineBuilder.REFERENCE_VAR_CHAR)) {
      commandValue.append(file.getPath());
    }
    else {
      String path = file.getPath(fileSeparator);
      commandValue.append(CommandLineBuilder.makePathWithEnvironmentVariable(
        EnvironmentVariable.PROJECT_DIR, path, fileSeparator));
    }
    return commandValue.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final Object getExpressionValue() {
    return null;
  }

  /**
   * <p>
   * Obtm a lista de observadores deste parmetro.
   * </p>
   * 
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   * 
   * <p>
   * Se no houver observadores a lista retornada estar vazia.
   * </p>
   * 
   * @return A lista de observadores.
   */
  public final List<FileParameterListener> getFileParameterListeners() {
    return Collections.unmodifiableList(this.listeners);
  }

  /**
   * Obtm o tipo de arquivo que este parmetro aceita.
   * 
   * @return O tipo de arquivo ou {@code null} se ele no se importar com um
   *         tipo especfico.
   */
  public final String getFileType() {
    return this.fileType;
  }

  /**
   * Obtm o modo de funcionamento (apenas arquivos, apenas diretrios ou ambos)
   * deste parmetro.
   * 
   * @return O modo de funcionamento.
   */
  public final FileParameterMode getMode() {
    return this.mode;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String getValueAsText() {
    FileParameterValue file = getValue();
    if (file == null) {
      return null;
    }
    String type = file.getType();
    if (type == null || type.trim().isEmpty()) {
      type = ProjectFileType.UNKNOWN;
    }

    String valueAsText = file.getPath() + ":" + type;
    return valueAsText;
  }

  /**
   * Indica se este parmetro participa de uma conexo entre ns de um fluxo de
   * algoritmos.
   * 
   * @return {@code true} se participar ou {@code false} caso contrrio.
   */
  public final boolean hasLink() {
    return this.hasLink;
  }

  /**
   * Indica se um painel de filtro deve ser exibido. ({@code true}), se sim, (
   * {@code false}), caso contrrio.
   * 
   * @return {@code true} se exibir ou {@code false} caso contrrio.
   */
  public boolean usesFilter() {
    return this.usesFilter;
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  public void importValue(Map<String, Object> parameterValues) {
    Collection<Map<String, String>> exportableValues =
      (Collection<Map<String, String>>) parameterValues.get(getName());
    if (exportableValues != null && !exportableValues.isEmpty()) {
      Map<String, String> exportableValue = exportableValues.iterator().next();
      String path = exportableValue.get(PATH);
      String type = exportableValue.get(TYPE);
      FileParameterValue file = new FileParameterValue(path, type);
      setValue(file);
    }
    else {
      setValue(null);
    }
  }

  /**
   * Remove um observador deste parmetro.
   * 
   * @param listener O observador (No aceita {@code null}).
   * 
   * @return {@code true} se ele removeu o observador ou {@code false} se o
   *         observador no estiver neste parmetro.
   */
  public final boolean removeFileParameterListener(
    FileParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    return this.listeners.add(listener);
  }

  /**
   * Atribui um tipo de arquivo especfico a este parmetro.
   * 
   * @param fileType O tipo de arquivo (Aceita {@code null}).
   */
  public final void setFileType(String fileType) {
    this.fileType = fileType;
    for (FileParameterListener listener : this.listeners) {
      listener.typeWasChanged(this);
    }
    fireVisiblityWasChangedEvent();
  }

  /**
   * Marca se este arquivo est ou no participando de uma conexo entre ns de
   * algoritmos em um fluxo de algoritmos.
   * 
   * @param hasLink {@code true} se estiver ou {@code false} caso contrrio.
   */
  public final void setHasLink(boolean hasLink) {
    this.hasLink = hasLink;
    for (FileParameterListener listener : this.listeners) {
      listener.hasLinkWasChanged(this);
    }
    fireVisiblityWasChangedEvent();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final void setValueAsText(String parameterValue) {
    if (parameterValue == null || parameterValue.length() == 0) {
      setValue(null);
      return;
    }
    Pattern withTypePattern = Pattern.compile("^(.+):(.*)$");
    Matcher withTypeMatcher = withTypePattern.matcher(parameterValue);
    String path, type;
    if (withTypeMatcher.matches()) {
      path = withTypeMatcher.group(1);
      type = withTypeMatcher.group(2);
    }
    else {
      path = parameterValue;
      type = null;
    }
    if (type == null || type.trim().isEmpty()) {
      if (mode == FileParameterMode.DIRECTORY) {
        type = ProjectFileType.DIRECTORY_TYPE;
      }
      else {
        type = ProjectFileType.UNKNOWN;
      }
    }
    FileParameterValue file = new FileParameterValue(path, type);
    setValue(file);
  }

  /**
   * Atribui o modo de funcionamento (apenas arquivo, apenas diretrio ou ambos)
   * deste parmetro.
   * 
   * @param mode O modo de funcionamento (No aceita {@code null}).
   */
  private void setMode(FileParameterMode mode) {
    if (mode == null) {
      throw new IllegalArgumentException("O parmetro mode est nulo.");
    }
    this.mode = mode;
  }

  /**
   * Determina se este arquivo pode utilizar <i>pipes</i>.
   * 
   * @param usesPipe {@value FileParameterPipeAcceptance#TRUE} se este parmetro
   *        pode aceitar pipe, {@value FileParameterPipeAcceptance#FALSE} se no
   *        aceita pipe, ou {@value FileParameterPipeAcceptance#ALWAYS} se *s*
   *        aceita pipe.
   */
  private void setUsesPipe(FileParameterPipeAcceptance usesPipe) {
    if (usesPipe == null) {
      throw new IllegalArgumentException("O parmetro usesPipe est nulo.");
    }
    this.usesPipe = usesPipe;
  }

  /**
   * Determina se um painel de filtro pode ser usado.
   * 
   * @param usesFilter True, se o parmetro pode exibir um painel de filtro ou
   *        false, caso contrrio.
   */
  public void setUsesFilter(boolean usesFilter) {
    this.usesFilter = usesFilter;
  }

  /**
   * Cria os atributos transientes.
   * 
   * @param in stream com o objeto serializado.
   * 
   * @throws IOException em caso de erro de I/O.
   * @throws ClassNotFoundException se a classe do objeto serializado no for
   *         encontrada.
   */
  private void readObject(java.io.ObjectInputStream in) throws IOException,
    ClassNotFoundException {
    in.defaultReadObject();
    listeners = new LinkedList<FileParameterListener>();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileParameterValidator createParameterValidator() {
    return new FileParameterValidator(isOptional());
  }

  /**
   * Indica se o arquivo selecionado precisa existir.
   * 
   * @return mustExist verdadeiro se o arquivo precisa existir, ou falso, caso
   *         contrrio.
   */
  public boolean mustExist() {
    return mustExist;
  }
}
