package csbase.logic.algorithms.parameters;

import java.rmi.RemoteException;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import tecgraf.javautils.core.io.FileUtils;
import csbase.exception.ParseException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.CommandLineBuilder;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.EnvironmentVariable;
import csbase.logic.algorithms.ExecutionLocation;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.algorithms.FileParameterValue;
import csbase.logic.algorithms.parameters.triggers.Trigger;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationSuccess;

/**
 * <p>
 * Configurador de algoritmos simples.
 * </p>
 *
 * @author lmoreira
 */
public final class SimpleAlgorithmConfigurator extends AlgorithmConfigurator {

  /**
   * O nome do executvel.
   */
  private String commandBinaryName;

  /**
   * O nome do arquivo de entrada padro.
   */
  private String defaultInputFileParameterName;
  /**
   * As expresses de validao.
   */
  private Set<ValidationExpression> expressions;
  /**
   * Os grupos.
   */
  private List<ParameterGroup> groups;
  /**
   * Os parmetros ocultos.
   */
  private Set<HiddenParameter> hiddenParameters;
  /**
   * Indica se o configurador  editvel.
   */
  private boolean isEnabled;
  /**
   * Um flag que indica se as sadas padro e de erro devem ser exibidas.
   */
  private boolean showOutput;
  /**
   * O shell que ir executar o comando.
   */
  private String shell;
  /**
   * Os gatilhos que modificam propriedades deste configurador.
   */
  private Set<Trigger<?>> triggers;

  /**
   * Cria um configurador de algoritmos simples.
   *
   * @param algorithmVersion A verso do algoritmo (No aceita {@code null}).
   * @param description A descrio (Aceita {@code null}).
   * @param executionType O tipo de execuo (No aceita {@code null}).
   * @param executionLocation O local de execuo do comando.
   * @param abbreviation A abreviatura (Aceita {@code null}).
   * @param provideId Sinal que indica se  necessrio fornecer o identificador
   *        do comando na linha de comando.
   * @param command O nome do executvel (No aceita {@code null}).
   * @param shell O shell que ir executar o comando (Aceita {@code null}).
   * @param defaultInputFileParameterName O arquivo de entrada padro (Aceita
   *        {@code null}).
   * @param showOutput Um flag que indica se as sadas padro e de erro devem
   *        ser exibidas.
   * @param commandLinePattern pattern de linha de comando.
   */
  public SimpleAlgorithmConfigurator(AlgorithmVersionInfo algorithmVersion,
    String description, ExecutionType executionType,
    ExecutionLocation executionLocation, String abbreviation,
    boolean provideId, String command, String shell,
    String defaultInputFileParameterName, boolean showOutput,
    final String commandLinePattern) {
    super(ConfiguratorType.SIMPLE, algorithmVersion, description,
      executionType, executionLocation, abbreviation, provideId,
      commandLinePattern);
    this.isEnabled = true;
    this.expressions = new HashSet<ValidationExpression>();
    this.groups = new LinkedList<ParameterGroup>();
    this.hiddenParameters = new HashSet<HiddenParameter>();
    this.triggers = new HashSet<Trigger<?>>();
    this.shell = shell;
    this.defaultInputFileParameterName = defaultInputFileParameterName;
    this.showOutput = showOutput;
    setCommandBinaryName(command);
  }

  /**
   * Adiciona uma expresso de validao a este configurador.
   *
   * @param expression A expresso (No aceita {@code null}).
   *
   * @return {@code true} sucesso ou {@code false} se no puder adicionar a
   *         expresso, porque existe uma expresso igual a esta j cadastrada.
   */
  public boolean addExpression(ValidationExpression expression) {
    if (expression == null) {
      throw new IllegalArgumentException("O parmetro expression est nulo.");
    }
    return this.expressions.add(expression);
  }

  /**
   * <p>
   * Adiciona um grupo a este configurador.
   * </p>
   *
   * <p>
   * Mantm a ordem de insero.
   * </p>
   *
   * @param group O grupo (No aceita {@code null}).
   *
   * @return {@code true} sucesso ou {@code false} se no puder adicionar o
   *         grupo, porque existe um grupo igual a este j cadastrado.
   */
  public boolean addGroup(ParameterGroup group) {
    if (group == null) {
      throw new IllegalArgumentException("O parmetro group est nulo.");
    }
    if (this.groups.contains(group)) {
      return false;
    }
    this.groups.add(group);
    Iterator<?> simpleParameterIterator =
      group.getSimpleParameters().iterator();
    while (simpleParameterIterator.hasNext()) {
      SimpleParameter<?> parameter =
        (SimpleParameter<?>) simpleParameterIterator.next();
      addParameterListeners(parameter);
    }
    return true;
  }

  /**
   * Adiciona observadores dos parmetros.
   *
   * @param <T> Tipo do parmetro.
   * @param parameter O parmetro.
   */
  @SuppressWarnings("synthetic-access")
  private <T> void addParameterListeners(SimpleParameter<T> parameter) {
    parameter.addSimpleParameterListener(new SimpleParameterListener<T>() {
      @Override
      public void capabilityWasChanged(SimpleParameter<T> param) {
        fireParameterWasSetEnabled(param.getName(), param.isEnabled());
      }

      @Override
      public void defaultValueWasChanged(SimpleParameter<T> param) {
        // no faz nada
      }

      @Override
      public void labelWasChanged(SimpleParameter<T> param) {
        fireParameterLabelWasChanged(param.getName(), param.getLabel());
      }

      @Override
      public void valueWasChanged(SimpleParameter<T> param) {
        fireParameterValueWasChanged(param.getName(), param.getValue());
      }

      @Override
      public void visibilityWasChanged(SimpleParameter<T> param) {
        fireParameterWasSetVisible(param.getName(), param.isVisible());
      }
    });
    parameter.addSimpleParameterListener(new SimpleParameterListener<T>() {
      @Override
      public void capabilityWasChanged(SimpleParameter<T> param) {
        // no faz nada
      }

      @Override
      public void defaultValueWasChanged(SimpleParameter<T> param) {
        // no faz nada
      }

      @Override
      public void labelWasChanged(SimpleParameter<T> param) {
        // no faz nada
      }

      @Override
      public void valueWasChanged(SimpleParameter<T> param) {
        updateTriggers();
      }

      @Override
      public void visibilityWasChanged(SimpleParameter<T> param) {
        // no faz nada
      }
    });
  }

  /**
   * Adiciona um parmetro oculto a este configurador.
   *
   * @param hiddenParameter O parmetro oculto (No aceita {@code null}).
   *
   * @return {@code true} sucesso ou {@code false} se no puder adicionar o
   *         parmetro oculto, porque existe um parmetro oculto igual a este j
   *         cadastrado.
   */
  public boolean addHiddenParameter(HiddenParameter hiddenParameter) {
    if (hiddenParameter == null) {
      throw new IllegalArgumentException(
        "O parmetro hiddenParameter est nulo.");
    }
    return this.hiddenParameters.add(hiddenParameter);
  }

  /**
   * <p>
   * Adiciona um gatilho a este configurador.
   * </p>
   *
   * <p>
   * Quando o gatilho  adicionado ele  atualizado, para que haja uma sincronia
   * entre o estado do configurador e o funcionamento do gatilho.
   * </p>
   *
   * @param trigger O gatilho (No aceita {@code null}).
   *
   * @return {@code true} sucesso ou {@code false} se no puder adicionar o
   *         gatilho, porque existe um gatilho igual a este j cadastrado.
   */
  public boolean addTrigger(Trigger<?> trigger) {
    if (trigger == null) {
      throw new IllegalArgumentException("O parmetro trigger est nulo.");
    }
    if (!this.triggers.add(trigger)) {
      return false;
    }
    for (Trigger<?> aTrigger : getTriggers()) {
      aTrigger.update(this);
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<Object, Object> exportValues() {
    Map<Object, Object> parameterValues = new HashMap<Object, Object>();
    Iterator<ParameterGroup> groupIterator = getGroups().iterator();
    while (groupIterator.hasNext()) {
      ParameterGroup group = groupIterator.next();
      Map<String, Object> values = group.exportValue();
      for (String parameterKey : values.keySet()) {
        parameterValues.put(parameterKey, values.get(parameterKey));
      }
    }
    return Collections.unmodifiableMap(parameterValues);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<String[]> getBinaryDirectoriesAsArray(String platformId) {
    String[] binaryDirectory = getBinaryDirectoryAsArray(platformId);
    Set<String[]> binaryDirectories = new HashSet<String[]>();
    binaryDirectories.add(binaryDirectory);
    return binaryDirectories;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<String> getBinaryDirectories(String platformId, char fileSeparator) {
    String binaryDirectory = getBinaryDirectory(platformId, fileSeparator);
    Set<String> binaryDirectories = new HashSet<String>();
    binaryDirectories.add(binaryDirectory);
    return binaryDirectories;
  }

  /**
   * Obtm o nome do executvel.
   *
   * @return O nome do executvel.
   */
  public String getCommandBinaryName() {
    return this.commandBinaryName;
  }

  /**
   * Obtm o nome do arquivo de entrada padro.
   *
   * @return O nome do arquivo de entrada padro ou {@code null} se no houver
   *         um.
   */
  public String getDefaultInputFileParameterName() {
    return this.defaultInputFileParameterName;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<FileParameterValue> getInputDirectories() {
    Set<FileParameterValue> directories = new HashSet<FileParameterValue>();
    for (FileParameter parameter : getInputFileParameters()) {
      FileParameterValue file = parameter.getValue();
      if (file != null && file.isDirectory()) {
        directories.add(file);
      }
    }
    for (InputFileListParameter inputFileListParameter : getInputFileListParameters()) {
      List<FileParameterValue> files = inputFileListParameter.getValue();
      if (files != null) {
        for (FileParameterValue file : files) {
          if (file.isDirectory()) {
            directories.add(file);
          }
        }
      }
    }
    return Collections.unmodifiableSet(directories);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<FileParameterValue> getOutputDirectories() {
    Set<FileParameterValue> directories = new HashSet<FileParameterValue>();
    for (FileParameter parameter : getOutputFileParameters()) {
      FileParameterValue file = parameter.getValue();
      if (file != null && file.isDirectory()) {
        directories.add(file);
      }
    }
    for (OutputFileListParameter outputFileListParameter : getOutputFileListParameters()) {
      List<FileParameterValue> files = outputFileListParameter.getValue();
      if (files != null) {
        for (FileParameterValue file : files) {
          if (file.isDirectory()) {
            directories.add(file);
          }
        }
      }
    }
    return Collections.unmodifiableSet(directories);
  }

  /**
   * <p>
   * Obtm o conjunto de expresses de validao associadas a este configurador.
   * </p>
   *
   * <p>
   * O conjunto retornado  imutvel (veja
   * {@link Collections#unmodifiableSet(Set)}).
   * </p>
   *
   * @return O conjunto de expresses ou um conjunto vazio se no houver
   *         expresses.
   */
  public Set<ValidationExpression> getExpressions() {
    return Collections.unmodifiableSet(this.expressions);
  }

  /**
   * <p>
   * Obtm a lista de grupos associadas a este configurador.
   * </p>
   *
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return A lista de grupos ou um conjunto vazio se no houver grupos.
   */
  public List<ParameterGroup> getGroups() {
    return Collections.unmodifiableList(this.groups);
  }

  /**
   * <p>
   * Obtm o conjunto de parmetros ocultos associados a este configurador.
   * </p>
   *
   * <p>
   * O conjunto retornado  imutvel (veja
   * {@link Collections#unmodifiableSet(Set)}).
   * </p>
   *
   * @return O conjunto de parmetros ocultos ou um conjunto vazio se no houver
   *         parmetros ocultos.
   */
  public Set<HiddenParameter> getHiddenParameters() {
    return Collections.unmodifiableSet(this.hiddenParameters);
  }

  /**
   * <p>
   * Obtm a lista com os parmetros que aceitam um conjunto de arquivos e/ou
   * diretrios para entrada de dados.
   * </p>
   *
   * <p>
   * Obs.: a relao  imutvel (@see {@link Collections#unmodifiableList(List)}
   * ).
   * </p>
   *
   * @return A relao de parmetros ou uma relao vazia caso no haja
   *         parmetros que aceitem um conjunto de arquivos.
   */
  public List<InputFileListParameter> getInputFileListParameters() {
    List<InputFileListParameter> inputFileListParameters =
      new LinkedList<InputFileListParameter>();
    Iterator<ParameterGroup> groupIterator = getGroups().iterator();
    while (groupIterator.hasNext()) {
      ParameterGroup group = groupIterator.next();
      inputFileListParameters.addAll(group.getInputFileListParameters());
    }
    return Collections.unmodifiableList(inputFileListParameters);
  }

  /**
   * <p>
   * Obtm a lista com os parmetros que aceitam um conjunto de arquivos e/ou
   * diretrios para sada de dados.
   * </p>
   *
   * <p>
   * Obs.: a relao  imutvel (@see {@link Collections#unmodifiableList(List)}
   * ).
   * </p>
   *
   * @return A relao de parmetros ou uma relao vazia caso no haja
   *         parmetros que aceitem um conjunto de arquivos.
   */
  public List<OutputFileListParameter> getOutputFileListParameters() {
    List<OutputFileListParameter> outputFileListParameters =
      new LinkedList<OutputFileListParameter>();
    Iterator<ParameterGroup> groupIterator = getGroups().iterator();
    while (groupIterator.hasNext()) {
      ParameterGroup group = groupIterator.next();
      outputFileListParameters.addAll(group.getOutputFileListParameters());
    }
    return Collections.unmodifiableList(outputFileListParameters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<InputFileParameter> getInputFileParameters() {
    List<InputFileParameter> inputFileParameters =
      new LinkedList<InputFileParameter>();
    Iterator<ParameterGroup> groupIterator = getGroups().iterator();
    while (groupIterator.hasNext()) {
      ParameterGroup group = groupIterator.next();
      inputFileParameters.addAll(group.getInputFileParameters());
    }
    return Collections.unmodifiableList(inputFileParameters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<FileParameterValue> getInputFiles() {
    Set<FileParameterValue> inputFiles = new HashSet<FileParameterValue>();
    List<InputFileParameter> inputFileParameters = getInputFileParameters();
    for (InputFileParameter inputFileParameter : inputFileParameters) {
      FileParameterValue file = inputFileParameter.getValue();
      if (file != null && file.isRegularFile()) {
        inputFiles.add(file);
      }
    }
    for (InputFileListParameter inputFileListParameter : getInputFileListParameters()) {
      List<FileParameterValue> files = inputFileListParameter.getValue();
      if (files != null) {
        for (FileParameterValue file : files) {
          if (file.isRegularFile()) {
            inputFiles.add(file);
          }
        }
      }
    }
    return Collections.unmodifiableSet(inputFiles);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<OutputFileParameter> getOutputFileParameters() {
    List<OutputFileParameter> outputFileParameters =
      new LinkedList<OutputFileParameter>();
    Iterator<ParameterGroup> groupIterator = getGroups().iterator();
    while (groupIterator.hasNext()) {
      ParameterGroup group = groupIterator.next();
      outputFileParameters.addAll(group.getOutputFileParameters());
    }
    return Collections.unmodifiableList(outputFileParameters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<FileParameterValue> getOutputFiles() {
    Set<FileParameterValue> outputFiles = new HashSet<FileParameterValue>();
    List<OutputFileParameter> outputFileParameters = getOutputFileParameters();
    for (OutputFileParameter outputFileParameter : outputFileParameters) {
      if ((outputFileParameter.isVisible() || !outputFileParameter
        .ignoreIfInvisible())
        && (outputFileParameter.isEnabled() || !outputFileParameter
          .ignoreIfDisabled())) {
        FileParameterValue file = outputFileParameter.getValue();
        if (file != null && file.isRegularFile()) {
          outputFiles.add(file);
        }
      }
    }
    for (OutputFileListParameter outputFileListParameter : getOutputFileListParameters()) {
      if ((outputFileListParameter.isVisible() || !outputFileListParameter
        .ignoreIfInvisible())
        && (outputFileListParameter.isEnabled() || !outputFileListParameter
          .ignoreIfDisabled())) {
        List<FileParameterValue> files = outputFileListParameter.getValue();
        if (files != null) {
          for (FileParameterValue file : files) {
            if (file.isRegularFile()) {
              outputFiles.add(file);
            }
          }
        }
      }
    }
    return Collections.unmodifiableSet(outputFiles);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getParameterLabel(String parameterName)
    throws ParameterNotFoundException {
    if (parameterName == null) {
      throw new IllegalArgumentException("O parmetro parameterName est nulo.");
    }
    for (ParameterGroup group : getGroups()) {
      SimpleParameter<?> parameter = group.getSimpleParameter(parameterName);
      if (parameter != null) {
        return parameter.getLabel();
      }
    }
    throw new ParameterNotFoundException(parameterName, toString());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<String> getParameterNames() {
    Set<String> parameterNames = new HashSet<String>();
    for (ParameterGroup group : getGroups()) {
      for (SimpleParameter<?> parameter : group.getSimpleParameters()) {
        parameterNames.add(parameter.getName());
      }
    }
    return Collections.unmodifiableSet(parameterNames);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getParameterType(String parameterName)
    throws ParameterNotFoundException {
    if (parameterName == null) {
      throw new IllegalArgumentException("O parmetro parameterName est nulo.");
    }
    for (ParameterGroup group : getGroups()) {
      SimpleParameter<?> parameter = group.getSimpleParameter(parameterName);
      if (parameter != null) {
        return parameter.getType();
      }
    }
    throw new ParameterNotFoundException(parameterName, toString());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getParameterValue(String parameterName)
    throws ParameterNotFoundException {
    if (parameterName == null) {
      throw new IllegalArgumentException("O parmetro parameterName est nulo.");
    }
    for (ParameterGroup group : getGroups()) {
      SimpleParameter<?> parameter = group.getSimpleParameter(parameterName);
      if (parameter != null) {
        return parameter.getValueAsText();
      }
    }
    throw new ParameterNotFoundException(parameterName, toString());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<String> getPlatforms() {
    Vector<?> platformVector = getAlgorithmVersion().getSupportedPlatforms();
    Set<String> platformSet = new HashSet<String>();
    for (Object platformAsObject : platformVector) {
      String platform = (String) platformAsObject;
      platformSet.add(platform);
    }
    return Collections.unmodifiableSet(platformSet);
  }

  /**
   * Obtm o shell que ir executar o comando deste configurador.
   *
   * @return O shell ou {@code null} se ele no for informado.
   */
  public String getShell() {
    return this.shell;
  }

  /**
   * Obtm um parmetro simples dado o seu nome.
   *
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   *
   * @return O parmetro ou {@code null} se o parmetro no for encontrado.
   */
  public SimpleParameter<?> getSimpleParameter(String parameterName) {
    List<SimpleParameter<?>> parameters = getSimpleParameters();
    for (SimpleParameter<?> parameter : parameters) {
      if (parameter.getName().equals(parameterName)) {
        return parameter;
      }
    }
    return null;
  }

  /**
   * <p>
   * Obtm a lista de todos parmetros simples associados a este configurador de
   * algoritmos.
   * </p>
   *
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return A lista de parmetros ou uma lista vazia se no houver parmetros.
   */
  public List<SimpleParameter<?>> getSimpleParameters() {
    List<SimpleParameter<?>> parameters = new LinkedList<SimpleParameter<?>>();
    for (ParameterGroup group : getGroups()) {
      parameters.addAll(group.getSimpleParameters());
    }
    return Collections.unmodifiableList(parameters);
  }

  /**
   * <p>
   * Obtm o conjunto de gatilhos associados a este configurador.
   * </p>
   *
   * <p>
   * O conjunto retornado  imutvel (veja
   * {@link Collections#unmodifiableSet(Set)}).
   * </p>
   *
   * @return O conjunto de gatilhos ou um conjunto vazio se no houver gatilhos
   *         associados a este configurador.
   */
  public Set<Trigger<?>> getTriggers() {
    return Collections.unmodifiableSet(this.triggers);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isEnabled() {
    return isEnabled;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isSetDefaultValues() {
    for (ParameterGroup group : getGroups()) {
      if (!group.isSetDefaultValue()) {
        return false;
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void importValues(Map<Object, Object> parameterValues) {
    if (parameterValues == null) {
      throw new IllegalArgumentException(
        "O parmetro parameterValues est nulo.");
    }
    Iterator<ParameterGroup> groupIterator = getGroups().iterator();
    while (groupIterator.hasNext()) {
      ParameterGroup group = groupIterator.next();
      group.importValue(parameterValues);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String makeCommandLine(CommandLineContext context) {

    char fileSeparator = context.getFileSeparator();

    CommandLineBuilder builder = new CommandLineBuilder(context.isScript());

    builder.append(makeEnvironmentVariablesDeclaration(context));
    builder.append("( ");
    /*
     * Entra no diretrio de execuo
     */
    String execDirPath =
      CommandLineBuilder
      .makeEnvironmentVariableReference(EnvironmentVariable.EXECUTION_DIR);
    builder.append("cd ").append(execDirPath).append(" && ");

    /*
     * Executa o binrio passando os parmetros especificados
     */
    builder.append("( ");
    if (getShell() != null) {
      builder.append(getShell()).append(" ");
    }
    String binaryPath =
      CommandLineBuilder.makePathWithEnvironmentVariable(
        EnvironmentVariable.BINARY_DIR, getCommandBinaryName(), fileSeparator);
    builder.append(binaryPath);

    StringBuilder parameterListBuilder = new StringBuilder();
    Iterator<ParameterGroup> groupIterator = getGroups().iterator();
    String parameterSeparator = "";
    while (groupIterator.hasNext()) {
      ParameterGroup group = groupIterator.next();
      String groupCommandLine = group.getCommandLine(context);
      if (groupCommandLine != null) {
        parameterListBuilder.append(parameterSeparator);
        parameterListBuilder.append(groupCommandLine);
        parameterSeparator = " ";
      }
    }
    for (HiddenParameter parameter : getHiddenParameters()) {
      parameterListBuilder.append(parameterSeparator);
      parameterListBuilder.append(parameter.getCommandLine());
      parameterSeparator = " ";
    }
    String parameterList = parameterListBuilder.toString();
    if (parameterList.length() != 0) {
      builder.append(" ");
      builder.append(parameterList);
    }

    if (provideId()) {
      builder.append(" cmdId=").append(context.getCommandId());
    }

    /*
     * Direciona a sada da execuo e captura o cdigo de sada (se
     * especificado)
     */
    String standardOutputFile = null;
    if (getStandardOutputFile() != null) {
      standardOutputFile =
        CommandLineBuilder
        .makeEnvironmentVariableReference(EnvironmentVariable.LOG_FILE);
    }

    if (showOutput) {
      if (standardOutputFile != null) {
        builder.append(" )");
        builder.append(getExitCodeCaptureCommand(context));
        builder.append(")");
        builder.redirectStdErrToStdOutput();
        builder.append(" | tee -a ");
        builder.append(standardOutputFile);
      }
      else {
        builder.append(" ))");
        builder.append(getExitCodeCaptureCommand(context));
      }
    }
    else {
      builder.append(" ))");
      if (standardOutputFile != null) {
        builder.redirectStdErrAndStdOutputToFile(standardOutputFile);
      }
      else {
        builder.discardStdErrAndStdOutput();
      }
      builder.append(getExitCodeCaptureCommand(context));
    }

    String commandLine = builder.toString();
    return commandLine;
  }

  /**
   * Cria a parte da linha de comando que declara as variveis de ambiente
   * utilizadas e disponibilizadas pelo comando.
   *
   * @param context Contexto de execuo do comando.
   * @return A declarao de variveis de ambiente para a linha de comando.
   */
  protected String makeEnvironmentVariablesDeclaration(
    CommandLineContext context) {

    char fileSeparator = context.getFileSeparator();
    CommandLineBuilder commandLineBuilder =
      new CommandLineBuilder(context.isScript());

    Integer nodeId = context.getNodeIdForFlow();
    if (nodeId != null) {
      /*
       * Declara o identificador do n do fluxo.
       */
      commandLineBuilder.appendEnvironmentVariableDeclaration(
        EnvironmentVariable.FLOW_NODE_ID, String.valueOf(nodeId));
    }

    /*
     * Declara o diretrio de binrios do algoritmo
     */
    String binDir =
      FileUtils.joinPath(fileSeparator, context.getAlgorithmRootDirectory(),
        getBinaryDirectory(context.getPlatformId(), fileSeparator));
    commandLineBuilder.appendEnvironmentVariableDeclaration(
      EnvironmentVariable.BINARY_DIR, binDir);

    /*
     * Declara o diretrio do projeto do comando
     */
    String projectDir =
      FileUtils.joinPath(fileSeparator, context.getProjectRootDirectory(),
        context.getProjectDirectory());
    commandLineBuilder.appendEnvironmentVariableDeclaration(
      EnvironmentVariable.PROJECT_DIR, projectDir);

    /*
     * Declara a sandbox de execuo do comando
     */
    String sandboxDir =
      FileUtils.joinPath(fileSeparator, context.getSandboxRootDirectory(),
        context.getSandboxDirectory());
    commandLineBuilder.appendEnvironmentVariableDeclaration(
      EnvironmentVariable.SANDBOX_DIR, sandboxDir);

    /*
     * Declara o diretrio de execuo do comando. Pode ser o diretrio de
     * binrios do algoritmo, o diretrio de persistncia do comando (dentro do
     * .cmds) ou a sandbox de execuo do comando.
     */
    String executionDirectory;
    switch (getExecutionLocation()) {
      case BINARY_DIR:
        executionDirectory =
        CommandLineBuilder
        .makeEnvironmentVariableReference(EnvironmentVariable.BINARY_DIR);
        break;
      case PERSISTENCY_DIR:
        executionDirectory = context.getPersistenceDirectory();
        break;
      case SANDBOX:
        executionDirectory =
        CommandLineBuilder
        .makeEnvironmentVariableReference(EnvironmentVariable.SANDBOX_DIR);
        break;
      default:
        throw new IllegalStateException("Esse tipo de execuo no  suportada");
    }
    commandLineBuilder.appendEnvironmentVariableDeclaration(
      EnvironmentVariable.EXECUTION_DIR, executionDirectory);

    FileParameterValue standardOutputFile = getStandardOutputFile();
    if (standardOutputFile != null) {
      String standardOutputPath =
        CommandLineBuilder.makePathWithEnvironmentVariable(
          EnvironmentVariable.PROJECT_DIR, standardOutputFile.getPath(),
          fileSeparator);
      commandLineBuilder.appendEnvironmentVariableDeclaration(
        EnvironmentVariable.LOG_FILE, standardOutputPath);
    }

    /*
     * Declara a lista de arquivos de entrada do comando (se houver).
     */
    StringBuilder inputFilesVarValue = new StringBuilder();
    for (InputFileParameter inputFile : this.getInputFileParameters()) {
      String commandValue = inputFile.getCommandValue(context);
      if (commandValue != null) {
        inputFilesVarValue.append(commandValue);
        inputFilesVarValue.append(FILE_LIST_SEPARATOR);
      }
    }
    for (InputFileListParameter inputFileList : this
      .getInputFileListParameters()) {
      String commandValue = inputFileList.getCommandValue(context);
      if (commandValue != null) {
        inputFilesVarValue.append(commandValue);
        inputFilesVarValue.append(FILE_LIST_SEPARATOR);
      }
    }
    if (inputFilesVarValue.length() > 0) {
      inputFilesVarValue.deleteCharAt(inputFilesVarValue
        .lastIndexOf(FILE_LIST_SEPARATOR));

      commandLineBuilder.appendEnvironmentVariableDeclaration(
        EnvironmentVariable.INPUT_FILES, inputFilesVarValue.toString());
    }

    /*
     * Declara a lista de arquivos de sada do comando (se houver).
     */
    StringBuilder outputFilesVarValue = new StringBuilder();
    for (OutputFileParameter outputFile : this.getOutputFileParameters()) {
      String commandValue = outputFile.getCommandValue(context);
      if (commandValue != null) {
        outputFilesVarValue.append(commandValue);
        outputFilesVarValue.append(FILE_LIST_SEPARATOR);
      }
    }
    for (OutputFileListParameter outputFileList : this
      .getOutputFileListParameters()) {
      String commandValue = outputFileList.getCommandValue(context);
      if (commandValue != null) {
        outputFilesVarValue.append(commandValue);
        outputFilesVarValue.append(FILE_LIST_SEPARATOR);
      }
    }
    if (outputFilesVarValue.length() > 0) {
      outputFilesVarValue.deleteCharAt(outputFilesVarValue
        .lastIndexOf(FILE_LIST_SEPARATOR));

      commandLineBuilder.appendEnvironmentVariableDeclaration(
        EnvironmentVariable.OUTPUT_FILES, outputFilesVarValue.toString());
    }
    return commandLineBuilder.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void resetValues() {
    Iterator<ParameterGroup> parameterGroupIterator = getGroups().iterator();
    while (parameterGroupIterator.hasNext()) {
      ParameterGroup parameterGroup = parameterGroupIterator.next();
      parameterGroup.resetValue();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean setDefaultInputFile(FileParameterValue inputFile) {
    if (getDefaultInputFileParameterName() == null) {
      return false;
    }
    for (InputFileParameter inputFileParameter : getInputFileParameters()) {
      if (inputFileParameter.getName().equals(
        getDefaultInputFileParameterName())) {
        inputFileParameter.setValue(inputFile);
        return true;
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean setEnabled(boolean isEnabled) {
    if (this.isEnabled == isEnabled) {
      return false;
    }
    for (SimpleParameter<?> parameter : getSimpleParameters()) {
      parameter.setEnabled(isEnabled);
    }
    if (isEnabled) {
      updateTriggers();
    }
    this.isEnabled = isEnabled;
    fireWasSetEnabled();
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setParameterValue(String parameterName, String parameterValue)
    throws ParseException, ParameterNotFoundException {
    if (parameterName == null) {
      throw new IllegalArgumentException("O parmetro parameterName est nulo.");
    }
    for (ParameterGroup group : getGroups()) {
      SimpleParameter<?> parameter = group.getSimpleParameter(parameterName);
      if (parameter != null) {
        parameter.setValueAsText(parameterValue);
        return;
      }
    }
    throw new ParameterNotFoundException(parameterName, toString());
  }

  /**
   * Atualiza os gatilhos cujas condies sejam aceitas.
   */
  public void updateTriggers() {
    for (Trigger<?> trigger : this.triggers) {
      trigger.update(this);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getCurrentParameterFileVersion() {
    return "1.0";
  }

  /**
   * Obtm o diretrio do binrio para um SGA especfico.
   *
   * @param platformId O identificador da plataforma (No aceita {@code null}).
   * @param fileSeparator O separador de arquivo (No aceita {@code null}).
   *
   * @return O diretrio do binrio.
   */
  private String getBinaryDirectory(String platformId, char fileSeparator) {
    String[] path = getBinaryDirectoryAsArray(platformId);
    return FileUtils.joinPath(fileSeparator, path);
  }

  /**
   * Obtm o diretrio em formato de array do binrio para um SGA especfico.
   *
   * @param platformId O identificador da plataforma (No aceita {@code null}).
   *
   * @return O diretrio do binrio em formato de array.
   */
  private String[] getBinaryDirectoryAsArray(String platformId) {
    AlgorithmVersionInfo versionInfo = getAlgorithmVersion();
    String algorithmDirectoryName = versionInfo.getInfo().getDirectory();
    String versionsDirectoryName = versionInfo.getVersionsDirName();
    String versionDirectoryName = versionInfo.getDirectory();
    String binariesDirectoryName = versionInfo.getBinDirName();
    String[] binaryDirectory =
      new String[] { algorithmDirectoryName, versionsDirectoryName,
      versionDirectoryName, binariesDirectoryName, platformId };
    return binaryDirectory;
  }

  /**
   * Atribui o nome do executvel deste configurador.
   *
   * @param command O nome do executvel (No aceita {@code null}).
   */
  private void setCommandBinaryName(String command) {
    if (command == null) {
      throw new IllegalArgumentException("O parmetro command est nulo.");
    }
    this.commandBinaryName = command;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Validation validate(ValidationContext context) throws RemoteException {
    for (ParameterGroup groupView : groups) {
      Validation validation = groupView.validate(context);
      if (!validation.isWellSucceded()) {
        return validation;
      }
    }
    return new ValidationSuccess();
  }

}
