package csbase.logic.algorithms.parsers;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.BooleanColumn;
import csbase.logic.algorithms.parameters.Column;
import csbase.logic.algorithms.parameters.DoubleColumn;
import csbase.logic.algorithms.parameters.EnumerationColumn;
import csbase.logic.algorithms.parameters.EnumerationItem;
import csbase.logic.algorithms.parameters.IntegerColumn;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.TableParameter;
import csbase.logic.algorithms.parameters.TextColumn;

/**
 * <p>
 * Analisador de {@link TableParameter}.
 * </p>
 * 
 * <p>
 * Este parser l os atributos de parmetros do tipo tabela. O elemento corrente
 * do {@link XmlParser analisador de XML} precisa ser um elemento
 * {@link TableParameter}.
 * </p>
 * 
 */
public class TableParameterFactory extends
  SimpleParameterParser<TableParameter> {

  /**
   * <p>
   * O elemento {@value #TABLE_PARAMETER_ELEMENT}: descreve as propriedades de
   * um {@link TableParameter parmetro do tipo tabela}.
   * </p>
   * <p>
   *  filho do elemento {@link TableParameter}.
   * </p>
   */
  static final String TABLE_PARAMETER_ELEMENT = "tabela";

  /**
   * O atributo {@value #TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_ATTRIBUTE} do
   * elemento {@link #TABLE_PARAMETER_ELEMENT}: indica se o
   * {@link TableParameter parmetro do tipo tabela} deve delimitar as linhas da
   * tabela na linha de comando.  opcional, o seu valor-padro 
   * {@link #TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_DEFAULT_VALUE} e o seu tipo 
   * booleano.
   */
  protected static final String TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_ATTRIBUTE =
    "delimitar_linhas";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_ATTRIBUTE}.
   * </p>
   * <p>
   * O seu valor  {@value #TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_DEFAULT_VALUE}.
   * </p>
   */
  protected static final boolean TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_DEFAULT_VALUE =
    false;

  /**
   * O atributo {@value #TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_ATTRIBUTE} do
   * elemento {@link #TABLE_PARAMETER_ELEMENT}: indica se o
   * {@link TableParameter parmetro do tipo tabela} deve delimitar a tabela na
   * linha de comando.  opcional, o seu valor-padro 
   * {@link #TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_DEFAULT_VALUE} e o seu tipo 
   * booleano.
   */
  protected static final String TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_ATTRIBUTE =
    "delimitar_tabela";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_ATTRIBUTE}.
   * </p>
   * <p>
   * O seu valor  {@value #TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_DEFAULT_VALUE}
   * .
   * </p>
   */
  protected static final boolean TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_DEFAULT_VALUE =
    false;

  /**
   * O atributo {@value #TABLE_PARAMETER_ELEMENT_VISIBLE_ROW_COUNT_ATTRIBUTE} do
   * elemento {@link #TABLE_PARAMETER_ELEMENT}: indica a quantidade de linhas do
   * {@link TableParameter parmetro do tipo tabela}.  opcional, seu tipo 
   * inteiro e o valor mnimo 
   * {@link #TABLE_PARAMETER_ELEMENT_ROW_COUNT_MINIMUM_VALUE}.
   */
  protected static final String TABLE_PARAMETER_ELEMENT_VISIBLE_ROW_COUNT_ATTRIBUTE =
    "quantidade_de_linhas_visiveis";

  /**
   * O atributo {@value #TABLE_PARAMETER_ELEMENT_MAX_ROW_COUNT_ATTRIBUTE} do
   * elemento {@link #TABLE_PARAMETER_ELEMENT}: indica a quantidade mxima de
   * linhas do {@link TableParameter parmetro do tipo tabela};  opcional.
   */
  protected static final String TABLE_PARAMETER_ELEMENT_MAX_ROW_COUNT_ATTRIBUTE =
    "quantidade_maxima_de_linhas";

  /**
   * O atributo {@value #TABLE_PARAMETER_ELEMENT_MIN_ROW_COUNT_ATTRIBUTE} do
   * elemento {@link #TABLE_PARAMETER_ELEMENT}: indica a quantidade mxima de
   * linhas do {@link TableParameter parmetro do tipo tabela};  opcional.
   */
  protected static final String TABLE_PARAMETER_ELEMENT_MIN_ROW_COUNT_ATTRIBUTE =
    "quantidade_minima_de_linhas";

  /**
   * <p>
   * O valor mnimo para o atributo
   * {@link #TABLE_PARAMETER_ELEMENT_ROW_COUNT_ATTRIBUTE}.
   * </p>
   * <p>
   * O seu valor  {@value #TABLE_PARAMETER_ELEMENT_ROW_COUNT_MINIMUM_VALUE}.
   * </p>
   */
  protected static final int TABLE_PARAMETER_ELEMENT_ROW_COUNT_MINIMUM_VALUE =
    1;

  /**
   * O atributo {@value #TABLE_PARAMETER_ELEMENT_ROW_COUNT_ATTRIBUTE} do
   * elemento {@link #TABLE_PARAMETER_ELEMENT}: indica a quantidade de linhas
   * sero visveis no {@link TableParameter parmetro do tipo tabela}. 
   * opcional, seu tipo  inteiro e o valor mnimo 
   * {@link #TABLE_PARAMETER_ELEMENT_ROW_COUNT_MINIMUM_VALUE}.
   */
  protected static final String TABLE_PARAMETER_ELEMENT_ROW_COUNT_ATTRIBUTE =
    "quantidade_de_linhas";

  /**
   * <p>
   * O elemento {@link #CELL_VALUE_ELEMENT}: indica o valor de uma clula do
   * {@link TableParameter parmetro tipo tabela} indicado pelo elemento que 
   * pai deste elemento.
   * </p>
   * <p>
   *  filho do elemento {@link #TABLE_PARAMETER_ELEMENT}.
   * </p>
   */
  protected static final String CELL_VALUE_ELEMENT = "celula";

  /**
   * O atributo {@value #CELL_VALUE_ELEMENT_COLUMN_ID_ATTRIBUTE} do elemento
   * {@link #CELL_VALUE_ELEMENT}: indica o identificador da coluna, 
   * obrigatrio e  do tipo string.
   */
  protected static final String CELL_VALUE_ELEMENT_COLUMN_ID_ATTRIBUTE =
    "id_da_coluna";

  /**
   * O atributo {@value #CELL_VALUE_ELEMENT_ROW_INDEX_ATTRIBUTE} do elemento
   * {@link #CELL_VALUE_ELEMENT}: indica o ndice da linha,  obrigatrio,  do
   * tipo inteiro e o valor mnimo  0.
   */
  protected static final String CELL_VALUE_ELEMENT_ROW_INDEX_ATTRIBUTE =
    "indice_da_linha";

  /**
   * O atributo {@value #CELL_VALUE_ELEMENT_VALUE_ATTRIBUTE} do elemento
   * {@link #CELL_VALUE_ELEMENT}: indica o valor da clula,  obrigatrio, o
   * tipo depende do tipo da coluna ({@link #BOOLEAN_COLUMN_ELEMENT} - valor
   * booleano).
   */
  protected static final String CELL_VALUE_ELEMENT_VALUE_ATTRIBUTE = "valor";

  /**
   * O atributo {@value #COLUMN_ELEMENT_DEFAULT_VALUE_ATTRIBUTE} dos elementos
   * do tipo coluna: indica o valor-padro da coluna,  opcional e o seu tipo
   * depende do tipo da coluna (Ex.: {@link BooleanColumn coluna de booleanos} -
   * booleano).
   */
  protected static final String COLUMN_ELEMENT_DEFAULT_VALUE_ATTRIBUTE =
    "padrao";

  /**
   * O atributo {@value #COLUMN_ELEMENT_ID_ATTRIBUTE} dos elementos do tipo
   * coluna: define um identificador para a coluna,  opcional e  do tipo
   * string.
   */
  protected static final String COLUMN_ELEMENT_ID_ATTRIBUTE = "id";

  /**
   * O atributo {@value #COLUMN_ELEMENT_IS_EDITABLE_ATTRIBUTE} dos elementos do
   * tipo coluna: indica se a coluna  editvel,  opcional, o valor-padro 
   * {@link #COLUMN_ELEMENT_IS_EDITABLE_DEFAULT_VALUE} e  do tipo booleano.
   */
  protected static final String COLUMN_ELEMENT_IS_EDITABLE_ATTRIBUTE =
    "editavel";

  /**
   * <p>
   * O valor-padro do atributo {@link #COLUMN_ELEMENT_IS_EDITABLE_ATTRIBUTE}
   * dos elementos do tipo coluna.
   * </p>
   * <p>
   * O seu valor  {@value #COLUMN_ELEMENT_IS_EDITABLE_DEFAULT_VALUE}.
   * </p>
   */
  protected static final boolean COLUMN_ELEMENT_IS_EDITABLE_DEFAULT_VALUE =
    true;

  /**
   * O atributo {@value #COLUMN_ELEMENT_IS_OPTIONAL_ATTRIBUTE} dos elementos do
   * tipo coluna: indica se a coluna  editvel,  opcional, o valor-padro 
   * {@link #COLUMN_ELEMENT_IS_OPTIONAL_DEFAULT_VALUE} e  do tipo booleano.
   */
  protected static final String COLUMN_ELEMENT_IS_OPTIONAL_ATTRIBUTE =
    "opcional";

  /**
   * <p>
   * O valor-padro do atributo {@link #COLUMN_ELEMENT_IS_OPTIONAL_ATTRIBUTE}
   * dos elementos do tipo coluna.
   * </p>
   * <p>
   * O seu valor  {@value #COLUMN_ELEMENT_IS_OPTIONAL_DEFAULT_VALUE}.
   * </p>
   */
  protected static final boolean COLUMN_ELEMENT_IS_OPTIONAL_DEFAULT_VALUE =
    false;

  /**
   * O atributo {@value #COLUMN_ELEMENT_LABEL_ATTRIBUTE} dos elementos do tipo
   * coluna: define um rtulo para a coluna,  obrigatrio e  do tipo string.
   */
  protected static final String COLUMN_ELEMENT_LABEL_ATTRIBUTE = "rotulo";

  /**
   * <p>
   * O elemento {@value #BOOLEAN_COLUMN_ELEMENT}: descreve uma
   * {@link BooleanColumn coluna do tipo booleano} de um {@link TableParameter
   * parmetro do tipo tabela}.
   * </p>
   * <p>
   *  filho do elemento {@link #TABLE_PARAMETER_ELEMENT}.
   */
  protected static final String BOOLEAN_COLUMN_ELEMENT = "coluna_de_booleanos";

  /**
   * <p>
   * O elemento {@value #DOUBLE_COLUMN_ELEMENT}: descreve uma
   * {@link DoubleColumn coluna do tipo real} de um {@link TableParameter
   * parmetro do tipo tabela}.
   * </p>
   * 
   * <p>
   *  filho do elemento {@link #TABLE_PARAMETER_ELEMENT}.
   * </p>
   */
  protected static final String DOUBLE_COLUMN_ELEMENT = "coluna_de_reais";

  /**
   * <p>
   * O elemento {@value #ENUMERATION_COLUMN_ELEMENT}: indica que o
   * {@link TableParameter parmetro do tipo tabela} que que est descrita no
   * elemento-pai deste elemento ter uma {@link EnumerationColumn coluna do
   * tipo enumerao}.
   * </p>
   * <p>
   *  filho do elemento {@link #TABLE_PARAMETER_ELEMENT}.
   * </p>
   */
  protected static final String ENUMERATION_COLUMN_ELEMENT =
    "coluna_de_enumeracao";

  /**
   * <p>
   * O elemento {@value #INTEGER_COLUMN_ELEMENT}: descreve uma
   * {@link IntegerColumn coluna do tipo inteiro} de um {@link TableParameter
   * parmetro do tipo tabela}.
   * </p>
   * <p>
   *  filho do elemento {@link #TABLE_PARAMETER_ELEMENT}.
   * </p>
   */
  protected static final String INTEGER_COLUMN_ELEMENT = "coluna_de_inteiros";

  /**
   * <p>
   * O elemento {@value #TEXT_COLUMN_ELEMENT}: descreve uma {@link TextColumn
   * coluna do tipo texto} de um {@link TableParameter parmetro do tipo tabela}
   * .
   * </p>
   * <p>
   *  filho do elemento {@link #TABLE_PARAMETER_ELEMENT}.
   * </p>
   */
  protected static final String TEXT_COLUMN_ELEMENT = "coluna_de_texto";

  /**
   * {@inheritDoc}
   */
  @Override
  public TableParameter createSimpleParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, ParameterGroup group) throws ParseException {
    boolean delimitTable =
      parser.extractAttributeValueAsBoolean(
        TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_ATTRIBUTE,
        TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_DEFAULT_VALUE);
    boolean delimitRows =
      parser.extractAttributeValueAsBoolean(
        TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_ATTRIBUTE,
        TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_DEFAULT_VALUE);
    Integer rowCount =
      parser.extractAttributeValueAsInteger(
        TABLE_PARAMETER_ELEMENT_ROW_COUNT_ATTRIBUTE, null, null,
        TABLE_PARAMETER_ELEMENT_ROW_COUNT_MINIMUM_VALUE);
    Integer visibleRowCount =
      parser.extractAttributeValueAsInteger(
        TABLE_PARAMETER_ELEMENT_VISIBLE_ROW_COUNT_ATTRIBUTE, null, null,
        TABLE_PARAMETER_ELEMENT_ROW_COUNT_MINIMUM_VALUE);

    Integer maxRowCount =
      parser.extractAttributeValueAsInteger(
        TABLE_PARAMETER_ELEMENT_MAX_ROW_COUNT_ATTRIBUTE, null, null, null);

    Integer minRowCount =
      parser.extractAttributeValueAsInteger(
        TABLE_PARAMETER_ELEMENT_MIN_ROW_COUNT_ATTRIBUTE, null, null, null);

    List<Column<?>> columns = loadColumns(parser);
    loadCellValues(parser, columns);
    TableParameter parameter =
      new TableParameter(name, label, description, isOptional, isVisible,
        commandLinePattern, columns, delimitTable, delimitRows, rowCount,
        visibleRowCount, minRowCount, maxRowCount);
    return parameter;
  }

  /**
   * <p>
   * Carregas os valores das clulas da tabela.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @param columns As colunas da tabela.
   * 
   * @throws ParseException Se houver um erro no XML.
   */
  protected void loadCellValues(XmlParser parser, List<Column<?>> columns)
    throws ParseException {
    if (parser.goToFirstChild(CELL_VALUE_ELEMENT)) {
      do {
        Integer rowIndex =
          parser.extractAttributeValueAsInteger(
            CELL_VALUE_ELEMENT_ROW_INDEX_ATTRIBUTE, null, 0);
        String columnId =
          parser.extractAttributeValue(CELL_VALUE_ELEMENT_COLUMN_ID_ATTRIBUTE);
        Column<?> theColumn = null;
        for (Column<?> aColumn : columns) {
          if (aColumn.getId().equals(columnId)) {
            theColumn = aColumn;
            break;
          }
        }
        if (theColumn == null) {
          throw new ParseException("A coluna {0} no encontrada.", columnId);
        }
        if (theColumn instanceof TextColumn) {
          TextColumn textColumn = (TextColumn) theColumn;
          String cellValue =
            parser.extractAttributeValue(CELL_VALUE_ELEMENT_VALUE_ATTRIBUTE,
              null);
          textColumn.addDefaultValue(rowIndex, cellValue);
        }
        else if (theColumn instanceof DoubleColumn) {
          DoubleColumn doubleColumn = (DoubleColumn) theColumn;
          Double cellValue =
            parser.extractAttributeValueAsDouble(
              CELL_VALUE_ELEMENT_VALUE_ATTRIBUTE, null,
              doubleColumn.getMaximum(), doubleColumn.getMinimum());
          doubleColumn.addDefaultValue(rowIndex, cellValue);
        }
        else if (theColumn instanceof IntegerColumn) {
          IntegerColumn integerColumn = (IntegerColumn) theColumn;
          Integer cellValue =
            parser.extractAttributeValueAsInteger(
              CELL_VALUE_ELEMENT_VALUE_ATTRIBUTE, null,
              integerColumn.getMaximum(), integerColumn.getMinimum());
          integerColumn.addDefaultValue(rowIndex, cellValue);
        }
        else if (theColumn instanceof BooleanColumn) {
          BooleanColumn booleanColumn = (BooleanColumn) theColumn;
          boolean cellValue =
            parser
              .extractAttributeValueAsBoolean(CELL_VALUE_ELEMENT_VALUE_ATTRIBUTE);
          booleanColumn.addDefaultValue(rowIndex, cellValue);
        }
        else {
          throw new ParseException(
            String
              .format(
                "Foi solicitado que o valor-padro da clula cuja linha  %d "
                  + "e a coluna  %s fosse modificada.\n"
                  + "Porm o tipo desta coluna no suporta este tipo de operao.\n",
                rowIndex, columnId));
        }
      } while (parser.goToNextSibling(CELL_VALUE_ELEMENT));
      parser.goToParent();
    }
  }

  /**
   * <p>
   * Carregas as colunas da tabela.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @return As colunas.
   * 
   * @throws ParseException Se houver um erro no XML, inclusive se no houver
   *         colunas.
   */
  protected List<Column<?>> loadColumns(XmlParser parser) throws ParseException {
    List<Column<?>> columns = new LinkedList<Column<?>>();
    if (!parser.goToFirstChild()) {
      throw new ParseException("A lista de colunas da tabela est vazia.");
    }
    do {
      Column<?> column = null;
      String elementName = parser.getElementName();
      if (elementName.equals(BOOLEAN_COLUMN_ELEMENT)) {
        column = loadBooleanColumn(parser);
      }
      else if (elementName.equals(DOUBLE_COLUMN_ELEMENT)) {
        column = loadDoubleColumn(parser);
      }
      else if (elementName.equals(INTEGER_COLUMN_ELEMENT)) {
        column = loadIntegerColumn(parser);
      }
      else if (elementName.equals(TEXT_COLUMN_ELEMENT)) {
        column = loadTextColumn(parser);
      }
      else if (elementName.equals(ENUMERATION_COLUMN_ELEMENT)) {
        column = loadEnumerationColumn(parser);
      }
      if (column != null) {
        if (columns.contains(column)) {
          throw new ParseException("J existe uma coluna {0}.", column);
        }
        columns.add(column);
      }
    } while (parser.goToNextSibling());
    parser.goToParent();
    return columns;
  }

  /**
   * <p>
   * Carrega uma {@link BooleanColumn}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #BOOLEAN_COLUMN_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @return O parmetro.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  protected BooleanColumn loadBooleanColumn(XmlParser parser)
    throws ParseException {
    String label = parser.extractAttributeValue(COLUMN_ELEMENT_LABEL_ATTRIBUTE);
    String id =
      parser.extractAttributeValue(COLUMN_ELEMENT_ID_ATTRIBUTE, label);
    String falseCommand =
      parser.extractAttributeValue(
        BooleanParameterFactory.BOOLEAN_ELEMENT_FALSE_ATTRIBUTE, null);
    String trueCommand =
      parser.extractAttributeValue(
        BooleanParameterFactory.BOOLEAN_ELEMENT_TRUE_ATTRIBUTE, null);
    boolean defaultValue =
      parser.extractAttributeValueAsBoolean(
        COLUMN_ELEMENT_DEFAULT_VALUE_ATTRIBUTE, false);
    boolean isEditable =
      parser.extractAttributeValueAsBoolean(
        COLUMN_ELEMENT_IS_EDITABLE_ATTRIBUTE,
        COLUMN_ELEMENT_IS_EDITABLE_DEFAULT_VALUE);
    parser.checkAttributes();
    parser.checkChildElements();
    return new BooleanColumn(id, label, defaultValue, isEditable, falseCommand,
      trueCommand);
  }

  /**
   * <p>
   * Carrega uma {@link DoubleColumn}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #DOUBLE_COLUMN_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @return O parmetro.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private DoubleColumn loadDoubleColumn(XmlParser parser) throws ParseException {
    String label = parser.extractAttributeValue(COLUMN_ELEMENT_LABEL_ATTRIBUTE);
    String id =
      parser.extractAttributeValue(COLUMN_ELEMENT_ID_ATTRIBUTE, label);
    Double defaultValue =
      parser.extractAttributeValueAsDouble(
        COLUMN_ELEMENT_DEFAULT_VALUE_ATTRIBUTE, null);
    boolean isOptional =
      parser.extractAttributeValueAsBoolean(
        COLUMN_ELEMENT_IS_OPTIONAL_ATTRIBUTE,
        COLUMN_ELEMENT_IS_OPTIONAL_DEFAULT_VALUE);
    boolean isEditable =
      parser.extractAttributeValueAsBoolean(
        COLUMN_ELEMENT_IS_EDITABLE_ATTRIBUTE,
        COLUMN_ELEMENT_IS_EDITABLE_DEFAULT_VALUE);
    Double maximum =
      parser.extractAttributeValueAsDouble(
        AbstractDoubleParameterParser.DOUBLE_ELEMENT_MAXIMUM_ATTRIBUTE, null,
        null, null);
    boolean isMaximumIncluded =
      parser
        .extractAttributeValueAsBoolean(
          AbstractDoubleParameterParser.DOUBLE_ELEMENT_INCLUDE_MAXIMUM_ATTRIBUTE,
          AbstractDoubleParameterParser.DOUBLE_ELEMENT_INCLUDE_MAXIMUM_DEFAULT_VALUE);
    Double minimum =
      parser.extractAttributeValueAsDouble(
        AbstractDoubleParameterParser.DOUBLE_ELEMENT_MINIMUM_ATTRIBUTE, null,
        maximum, null);
    boolean isMinimumIncluded =
      parser
        .extractAttributeValueAsBoolean(
          AbstractDoubleParameterParser.DOUBLE_ELEMENT_INCLUDE_MINIMUM_ATTRIBUTE,
          AbstractDoubleParameterParser.DOUBLE_ELEMENT_INCLUDE_MINIMUM_DEFAULT_VALUE);
    parser.checkAttributes();
    parser.checkChildElements();
    return new DoubleColumn(id, label, defaultValue, isOptional, isEditable,
      maximum, isMaximumIncluded, minimum, isMinimumIncluded);
  }

  /**
   * <p>
   * Carrega uma {@link EnumerationColumn}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #ENUMERATION_COLUMN_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @return O parmetro.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private EnumerationColumn loadEnumerationColumn(XmlParser parser)
    throws ParseException {
    String label = parser.extractAttributeValue(COLUMN_ELEMENT_LABEL_ATTRIBUTE);
    String id =
      parser.extractAttributeValue(COLUMN_ELEMENT_ID_ATTRIBUTE, label);
    String defaultItemId =
      parser.extractAttributeValue(COLUMN_ELEMENT_DEFAULT_VALUE_ATTRIBUTE);
    List<EnumerationItem> items = new LinkedList<EnumerationItem>();
    if (!parser
      .goToFirstChild(AbstractEnumerationParameterParser.ENUMERATION_ITEM_ELEMENT)) {
      throw new ParseException(
        "No foram informados os itens da coluna de enumerao {0}.",
        new Object[] { label });
    }
    EnumerationParameterParser enumerationParser =
      new EnumerationParameterParser();
    do {
      EnumerationItem enumerationItem =
        enumerationParser.loadEnumerationItem(parser);
      if (items.contains(enumerationItem)) {
        throw new ParseException(
          "O item {0} est duplicado na coluna de enumerao {1}.",
          enumerationItem.getId(), label);
      }
      for (EnumerationItem item : items) {
        if (enumerationItem.getLabel().equals(item.getLabel())) {
          throw new ParseException(
            "O item {0} est duplicado na coluna de enumerao {1}.",
            enumerationItem.getLabel(), label);
        }
      }
      items.add(enumerationItem);
    } while (parser
      .goToNextSibling(AbstractEnumerationParameterParser.ENUMERATION_ITEM_ELEMENT));
    parser.goToParent();
    boolean mustSort =
      parser
        .extractAttributeValueAsBoolean(
          AbstractEnumerationParameterParser.ENUMERATION_ELEMENT_MUST_SORT_ATTRIBUTE,
          AbstractEnumerationParameterParser.ENUMERATION_ELEMENT_MUST_SORT_DEFAULT_VALUE);
    if (mustSort) {
      Collections.sort(items);
    }
    parser.checkAttributes();
    EnumerationItem defaultItem =
      enumerationParser.getEnumerationItem(defaultItemId, items, defaultItemId);
    return new EnumerationColumn(id, label, defaultItem, items);
  }

  /**
   * <p>
   * Carrega uma {@link IntegerColumn}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #INTEGER_COLUMN_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @return O parmetro.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private IntegerColumn loadIntegerColumn(XmlParser parser)
    throws ParseException {
    String label = parser.extractAttributeValue(COLUMN_ELEMENT_LABEL_ATTRIBUTE);
    String id =
      parser.extractAttributeValue(COLUMN_ELEMENT_ID_ATTRIBUTE, label);
    boolean isOptional =
      parser.extractAttributeValueAsBoolean(
        COLUMN_ELEMENT_IS_OPTIONAL_ATTRIBUTE,
        COLUMN_ELEMENT_IS_OPTIONAL_DEFAULT_VALUE);
    boolean isEditable =
      parser.extractAttributeValueAsBoolean(
        COLUMN_ELEMENT_IS_EDITABLE_ATTRIBUTE,
        COLUMN_ELEMENT_IS_EDITABLE_DEFAULT_VALUE);
    Integer maximum =
      parser.extractAttributeValueAsInteger(
        AbstractIntegerParameterParser.INTEGER_ELEMENT_MAXIMUM_ATTRIBUTE, null,
        null, null);
    Integer minimum =
      parser.extractAttributeValueAsInteger(
        AbstractIntegerParameterParser.INTEGER_ELEMENT_MINIMUM_ATTRIBUTE, null,
        maximum, null);
    Integer defaultValue =
      parser.extractAttributeValueAsInteger(
        COLUMN_ELEMENT_DEFAULT_VALUE_ATTRIBUTE, null, maximum, minimum);
    parser.checkAttributes();
    parser.checkChildElements();
    return new IntegerColumn(id, label, defaultValue, isOptional, isEditable,
      maximum, minimum);
  }

  /**
   * <p>
   * Carrega uma {@link TextColumn}.
   * </p>
   * 
   * <p>
   * O elemento corrente do {@link XmlParser analisador de XML} precisa ser um
   * elemento {@link #TEXT_COLUMN_ELEMENT}.
   * </p>
   * 
   * @param parser O analisador (No aceita {@code null}).
   * 
   * @return O parmetro.
   * 
   * @throws ParseException Se houver algum erro no XML.
   */
  private TextColumn loadTextColumn(XmlParser parser) throws ParseException {
    String label = parser.extractAttributeValue(COLUMN_ELEMENT_LABEL_ATTRIBUTE);
    String id =
      parser.extractAttributeValue(COLUMN_ELEMENT_ID_ATTRIBUTE, label);
    String defaultValue =
      parser
        .extractAttributeValue(COLUMN_ELEMENT_DEFAULT_VALUE_ATTRIBUTE, null);
    boolean isOptional =
      parser.extractAttributeValueAsBoolean(
        COLUMN_ELEMENT_IS_OPTIONAL_ATTRIBUTE,
        COLUMN_ELEMENT_IS_OPTIONAL_DEFAULT_VALUE);
    boolean isEditable =
      parser.extractAttributeValueAsBoolean(
        COLUMN_ELEMENT_IS_EDITABLE_ATTRIBUTE,
        COLUMN_ELEMENT_IS_EDITABLE_DEFAULT_VALUE);
    Integer maxCharacters =
      parser
        .extractAttributeValueAsInteger(
          AbstractTextParameterParser.TEXT_ELEMENT_MAXIMUM_CHARACTERS_ATTRIBUTE,
          null,
          null,
          AbstractTextParameterParser.TEXT_ELEMENT_MAXIMUM_CHARACTERS_MINIMUM_VALUE);
    parser.checkAttributes();
    parser.checkChildElements();
    return new TextColumn(id, label, defaultValue, isOptional, isEditable,
      maxCharacters);
  }

}
