package csbase.logic.algorithms.xml.category;

import java.io.IOException;
import java.io.Writer;
import java.rmi.RemoteException;
import java.util.List;
import java.util.Set;
import java.util.SortedSet;

import tecgraf.javautils.xml.ImprovedXMLListIterator;
import tecgraf.javautils.xml.XMLBasicElement;
import tecgraf.javautils.xml.XMLElementInterface;
import tecgraf.javautils.xml.XMLEmptyElement;
import csbase.exception.ServiceFailureException;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.Category;
import csbase.remote.ClientRemoteLocator;

/**
 * Esta classe implementa o elemento XML que representa uma categoria de
 * algoritmos.
 * 
 */
public class XmlCategoryElement extends XMLBasicElement {

  /** Mantm a categoria corrente que est sendo escrita */
  private Category currentCategory;

  /**
   * Constri o elemento xml que descreve uma categoria.
   */
  public XmlCategoryElement() {
    super();
    setTag(XmlCategoriesUtils.CATEGORY_TAG);
  }

  /**
   * Construtor utilizado no processo de escrita do arquivo xml de categorias. A
   * categoria da aplicao especificada deve ser usada no contexto da
   * aplicao.
   * 
   * @param category categoria de algoritmos a ser escrita
   */
  public XmlCategoryElement(Category category) {
    this();
    currentCategory = category;
  }

  /**
   * Executado quando a tag de fim do elemento XML  encontrada. Deve-se criar o
   * objeto correspondente na camada da aplicao.
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#endTag(java.util.List)
   */
  @Override
  public void endTag(List<XMLElementInterface> childrenList) {
    currentCategory = createCategory();
    // System.out.println("Categoria criada: " + currentCategory);

    ImprovedXMLListIterator iterator =
      new ImprovedXMLListIterator(childrenList);
    XMLElementInterface nextElem = iterator.next(ImprovedXMLListIterator.ANY);
    while (nextElem != null) {
      if (isAlgorithmElement(nextElem)) {
        addAlgorithmToCategory(nextElem);
      }
      else if (isCategoryElement(nextElem)) {
        // Adiciona as subCategorias
        addSubCategory(nextElem);
      }
      nextElem = iterator.next(ImprovedXMLListIterator.ANY);
    }
  }

  /**
   * Adiciona a sub-categoria na categoria corrente.
   * 
   * @param xmlElem elemento xml que representa a sub-categoria
   */
  private void addSubCategory(XMLElementInterface xmlElem) {
    Category subCategory = ((XmlCategoryElement) xmlElem).getCategory();
    if (subCategory != null) {
      subCategory.setParentCategory(currentCategory);
      currentCategory.addCategory(subCategory);
    }
  }

  /**
   * Cria e adiciona as informaes do algoritmo na categoria corrente.
   * 
   * @param xmlElem elemento xml que representa o algoritmo
   */
  private void addAlgorithmToCategory(XMLElementInterface xmlElem) {
    Object algorithmId = getAlgorithmId(xmlElem);
    AlgorithmInfo algorithmInfo = null;
    try {
      algorithmInfo = ClientRemoteLocator.algorithmService.getInfo(algorithmId);
      if (algorithmInfo != null) {
        currentCategory.addAlgorithm(algorithmInfo);
      }
      else {
        System.err.println("Algoritmo com id [" + algorithmId
          + "] no encontrado.");
      }
    }
    catch (RemoteException re) {
      throw new ServiceFailureException(
        "Ocorreu um problema ao obter o algoritmo com id [" + algorithmId
          + "].", re);
    }
  }

  /**
   * Verifica se o elemento especificado corresponde a uma representao de um
   * algoritmo.
   * 
   * @param xmlElem elemento xml
   * @return retorna true, se o elemento especificado for a representao de um
   *         algoritmo, caso contrrio, retorna false
   */
  private boolean isAlgorithmElement(XMLElementInterface xmlElem) {
    return xmlElem.getTag().equals(XmlCategoriesUtils.ALGORITHM_TAG);
  }

  /**
   * Verifica se o elemento especificado corresponde a uma representao de uma
   * categoria.
   * 
   * @param xmlElem elemento xml
   * @return retorna true, se o elemento especificado for a representao de uma
   *         categoria, caso contrrio, retorna false
   */
  private boolean isCategoryElement(XMLElementInterface xmlElem) {
    return xmlElem.getTag().equals(XmlCategoriesUtils.CATEGORY_TAG);
  }

  /**
   * Obtm o objeto da aplicao categoria que foi criado a partir desse
   * elemento xml.
   * 
   * @return a categoria correspondente ao elemento xml
   */
  public Category getCategory() {
    return currentCategory;
  }

  /**
   * Cria o objeto da aplicao que representa essa categoria lida do xml.
   * 
   * @return a categoria
   */
  private Category createCategory() {
    String id = getAttributeStrValue(XmlCategoriesUtils.CATEGORY_ID_ATTR);
    String name = getAttributeStrValue(XmlCategoriesUtils.CATEGORY_NAME_ATTR);
    // O pai da categoria  estabelecido no momento de adicionar como
    // sub-categoria
    return new Category(id, name);
  }

  /**
   * Obtm o identificador do algoritmo a partir do elemento xml especificado.
   * 
   * @param xmlElem elemento xml que especifica o algoritmo
   * @return o id do algoritmo
   */
  private Object getAlgorithmId(XMLElementInterface xmlElem) {
    if (isAlgorithmElement(xmlElem)) {
      return xmlElem.getAttributeStrValue(XmlCategoriesUtils.ALGORITHM_ID_ATTR);
    }
    return null;
  }

  /**
   * Escreve todos os elementos que descrevem os algoritmos pertencentes 
   * categoria.
   * 
   * @param writer stream de sada
   * @param ident identao corrente
   * @throws IOException errro de I/o
   */
  private void writeAllAlgorithmElement(final Writer writer, final String ident)
    throws IOException {
    String newIdent = XMLBasicElement.getNextIdentation(ident);

    Set<AlgorithmInfo> algorithms = currentCategory.getAlgorithms();
    if (!algorithms.isEmpty()) {
      for (AlgorithmInfo algorithmInfo : algorithms) {
        writeAlgorithmElement(writer, newIdent, algorithmInfo);
      }
    }
  }

  /**
   * Escreve todos os elementos que descrevem as sub-categorias na categoria do
   * contexto da aplicao.
   * 
   * @param writer stream de sada
   * @param ident identao corrente
   * @throws IOException erro de I/O
   */
  private void writeAllSubCategoryElement(Writer writer, String ident)
    throws IOException {
    String newIdent = XMLBasicElement.getNextIdentation(ident);

    SortedSet<Category> subCategories = currentCategory.getCategories();
    if (!subCategories.isEmpty()) {
      for (Category category : subCategories) {
        writeSubCategoryElement(writer, newIdent, category);
      }
    }
  }

  /**
   * Escreve o elemento que descreve uma sub-categoria na categoria do contexto
   * da aplicao.
   * 
   * @param writer stream de sada
   * @param ident identao corrente
   * @param subCategory sub-categoria a ser escrita na categoria
   * @throws IOException erro de I/O
   */
  private void writeSubCategoryElement(final Writer writer, final String ident,
    Category subCategory) throws IOException {
    XmlCategoryElement xmlcategory = new XmlCategoryElement(subCategory);
    xmlcategory.write(writer, ident);
  }

  /**
   * Escreve o elemento que descreve um algoritmo na categoria.
   * 
   * @param writer stream de sada
   * @param ident identao corrente
   * @param algorithm algoritmo que pertence  categoria
   * @throws IOException erro de I/O
   */
  private void writeAlgorithmElement(Writer writer, String ident,
    AlgorithmInfo algorithm) throws IOException {
    XMLEmptyElement xmlAlgoElement = new XMLEmptyElement();
    xmlAlgoElement.setTag(XmlCategoriesUtils.ALGORITHM_TAG);

    String algoId = algorithm.getId();
    xmlAlgoElement.newAttribute(XmlCategoriesUtils.ALGORITHM_ID_ATTR, algoId);
    xmlAlgoElement.write(writer, ident);
  }

  @Override
  public void write(Writer writer, String ident) throws IOException {
    String categoryId = currentCategory.getId();
    String categoryName = currentCategory.getName();
    newAttribute(XmlCategoriesUtils.CATEGORY_ID_ATTR, categoryId);
    newAttribute(XmlCategoriesUtils.CATEGORY_NAME_ATTR, categoryName);
    writeStartTagln(writer, ident);
    writeAllSubCategoryElement(writer, ident);
    writeAllAlgorithmElement(writer, ident);
    writeEndTag(writer, ident);
  }

}
