package csbase.console;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import csbase.client.applicationmanager.ApplicationManager;
import csbase.logic.User;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Aplicao de console que ajuda a identificar chaves de bundle que no foram
 * definidas nos arquivos de idioma ou chaves nos arquivos de idioma que no
 * esto sendo usadas no cdigo.
 * 
 * Essa aplicao somente pode ser executada em ambiente de desenvolvimento com
 * o cdigo fonte das classes.
 * 
 * Assume que os mtodos usados para fazer internacionalizao se chamam
 * "getString". No se aplica no caso de chaves construdas dinamicamente no
 * cdigo.
 * 
 * @author Tecgraf PUC-Rio
 */
public class CheckBundlesApp extends AbstractConsoleApp {

  /**
   * Indica erro de autenticao no servidor. Este valor (3)  esperado no
   * script runserver.base para indicar falha de autenticao do servidor
   */
  static final int ADMIN_AUTHENTICATION_ERROR_CODE = 3;

  /**
   * Nome do mtodo default usado na busca.
   */
  private final String DEFAULT_METHOD_TO_FIND = "getString";

  /**
   * Guarda as validaes feitas em uma aplicao:
   * <ul>
   * <li>chaves encontradas em arquivos fonte e no definidas nos bundles;
   * <li>chaves nos bundles no encontradas nos arquivos fontes das aplicaes.
   * </ul>
   */
  private static class BundleValidation {
    /**
     * Chaves encontradas em arquivos fonte e no definidas nos bundles.
     */
    private Map<String, Set<String>> keysNotDefinedInBundles;
    /**
     * Chaves dos bundles usadas nos arquivos fontes das aplicaes.
     */
    private Set<String> keysDefinedInBundles;

    /**
     * Construtor.
     */
    BundleValidation() {
      this.keysNotDefinedInBundles = new HashMap<String, Set<String>>();
      this.keysDefinedInBundles = new TreeSet<String>();
    }

    /**
     * Adiciona uma chave encontrada em um arquivo fonte e que no est definida
     * nos bundles da aplicao.
     * 
     * @param fileName nome do arquivo fonte
     * @param key chave
     */
    public void addKeyNotDefined(String fileName, String key) {
      if (keysNotDefinedInBundles.get(fileName) == null) {
        keysNotDefinedInBundles.put(fileName, new TreeSet<String>());
      }
      keysNotDefinedInBundles.get(fileName).add(key);
    }

    /**
     * Adiciona uma chave encontrada em um arquivo fonte e definida nos bundles
     * da aplicao.
     * 
     * @param key chave
     */
    public void addKeyDefined(String key) {
      keysDefinedInBundles.add(key);
    }
  }

  /**
   * O mapa com as validaes por aplicao.
   */
  private Map<ApplicationRegistry, BundleValidation> validation;

  /**
   * Construtor.
   * 
   * @param args argumentos passados na linha de comando, pela console.
   */
  CheckBundlesApp(String[] args) {
    super(args);
    this.validation = new HashMap<ApplicationRegistry, BundleValidation>();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected BasicParams createParams() {
    return new CheckBundlesParams();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getLogin() {
    CheckBundlesParams params = (CheckBundlesParams) getParams();
    String login =
      (String) (params.userLogin == null ? User.getAdminId() : params.userLogin);
    return login;
  }

  /**
   * Executa o cliente. Caso haja falha na autenticao do administrador,
   * retorna um cdigo (ADMIN_AUTHENTICATION_ERROR).
   * 
   * @param args os argumentos passados pela linha de comando
   * @throws IOException em caso de erro.
   */
  public static void main(String[] args) throws IOException {
    CheckBundlesApp client = new CheckBundlesApp(args);
    if (client.login()) {
      client.execute();
      client.logout();
    }
    else {
      System.exit(ADMIN_AUTHENTICATION_ERROR_CODE);
    }
  }

  /**
   * Interpreta os parmetros e faz as respectivas requisies ao servidor.
   */
  private void execute() {
    CheckBundlesParams params = (CheckBundlesParams) getParams();
    try {
      if (params.english) {
        ApplicationManager.setInstance(new Locale("en", "US"));
      }
      else {
        ApplicationManager.setInstance(new Locale("pt", "BR"));
      }
      if (params.appToCheck != null) {
        checkApplicationsBundle(params.appToCheck);
      }
      else if (params.checkAllApps) {
        checkAllApplicationsBundles();
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Verifica se as chaves de bundle usadas em todas as aplicaes esto
   * devidamente configuradas nos arquivos de bundle carregados.
   * 
   * @throws Exception
   */
  private void checkAllApplicationsBundles() throws Exception {
    List<ApplicationRegistry> regs =
      ApplicationManager.getInstance().getAllApplicationRegistries();
    for (ApplicationRegistry reg : regs) {
      checkApplicationsBundle(reg);
    }
  }

  /**
   * Verifica se as chaves de bundle usadas em uma aplicao esto devidamente
   * configuradas nos arquivos de bundle carregados.
   * 
   * @param id identificador de registro da aplicao
   * @throws Exception erro na verificao
   */
  private void checkApplicationsBundle(String id) throws Exception {
    ApplicationRegistry reg =
      ApplicationManager.getInstance().getApplicationRegistry(id);
    checkApplicationsBundle(reg);
  }

  /**
   * Verifica se as chaves de bundle usadas em uma aplicao esto devidamente
   * configuradas nos arquivos de bundle carregados.
   * 
   * @param reg registro da aplicao
   * @throws Exception erro na verificao
   */
  private void checkApplicationsBundle(ApplicationRegistry reg)
    throws Exception {
    this.validation.put(reg, new BundleValidation());
    Class<?> appClass = Class.forName(reg.getClassName());
    String classSourceName = appClass.getSimpleName() + ".java";
    URL baseUrl = appClass.getResource(".");
    URL classURL = null;
    if (baseUrl != null) {
      classURL = new URL(baseUrl, classSourceName);
    }
    else {
      classURL = appClass.getResource(classSourceName);
    }
    if (classURL == null) {
      System.out.println("Arquivo " + classSourceName + " no encontrado.");
      return;
    }
    if (!"file".equalsIgnoreCase(classURL.getProtocol())) {
      throw new IllegalStateException("Classe no est em um arquivo.");
    }
    File path = new File(classURL.getPath());
    scanFilesForApp(reg, path.getParentFile());
  }

  /**
   * Verifica se as chaves de bundle usadas em uma aplicao esto devidamente
   * configuradas nos arquivos de bundle carregados.
   * 
   * @param reg registro da aplicao
   * @param root path da raiz de diretrio para a busca dos arquivos fontes
   * @throws Exception erro na verificao
   */
  private void scanFilesForApp(ApplicationRegistry reg, File root)
    throws Exception {
    scanFiles(reg, root);
    printResult(reg);
  }

  /**
   * Exibe o resultado na console com as chaves no encontradas nos bundles das
   * aplicaes e as chaves de bundles no usadas nas aplicaes.
   * 
   * @param reg
   */
  private void printResult(ApplicationRegistry reg) {
    System.out.println("------- CHAVES NO ENCONTRADAS NOS BUNDLES DA APP "
      + reg.getId() + " -------");
    BundleValidation appValidation = this.validation.get(reg);
    for (String fileName : appValidation.keysNotDefinedInBundles.keySet()) {
      System.out.println(">>> Arquivo: " + fileName);
      Set<String> keysNotDefined =
        appValidation.keysNotDefinedInBundles.get(fileName);
      for (String key : keysNotDefined) {
        System.out.println(key);
      }
    }
    CheckBundlesParams params = (CheckBundlesParams) getParams();
    if (params.showUnreferencedKeys) {
      Set<String> appKeys = reg.getResourceBundle().allKeys();
      appKeys.removeAll(validation.get(reg).keysDefinedInBundles);
      if (appKeys != null && !appKeys.isEmpty()) {
        System.out.println("------- CHAVES NO USADAS NA APP " + reg.getId()
          + " -------");
        for (String key : appKeys) {
          System.out.println(key);
        }
      }
    }
  }

  /**
   * Verifica se as chaves de bundle usadas em uma aplicao esto devidamente
   * configuradas nos arquivos de bundle carregados. Na sada, o conjunto
   * appDefinedKeys retorna preenchido com as chaves encontradas nos arquivos
   * fontes.
   * 
   * @param reg registro da aplicao
   * @param root path da raiz de diretrio para a busca dos arquivos fontes
   * @throws Exception erro na verificao
   */
  private void scanFiles(ApplicationRegistry reg, File root) throws Exception {
    File[] list = root.listFiles();
    if (list == null) {
      return;
    }
    for (File f : list) {
      if (f.isDirectory()) {
        scanFiles(reg, new File(f.getAbsolutePath()));
      }
      else {
        searchPattern(reg, f);
      }
    }
  }

  /**
   * Procura pelas ocorrncias de uso do nome de mtodo que faz a traduo no
   * cdigo fonte. O mtodo default  o DEFAULT_METHOD_TO_FIND.
   * 
   * @param reg o registro da aplicao cujo cdigo fonte ser inspecionado
   * @param file o nome do arquivo
   * @throws Exception erro na verificao
   */

  private void searchPattern(ApplicationRegistry reg, File file)
    throws Exception {
    if (file.getName().endsWith(".svn-base")) {
      return;
    }
    BufferedReader r = null;
    try {
      // A chave  o primeiro (ou nico) parmetro de getString
      // getString + 0 ou mais espaos + ( + 0 ou mais espaos + " + KEY + " + 0 ou mais espaos + , ou )
      Pattern pattStaticKey =
        Pattern.compile(DEFAULT_METHOD_TO_FIND
          + "\\s*\\(\\s*\"(.*?)\"\\s*[,|\\)]");
      r = new BufferedReader(new FileReader(file));
      String line;
      while ((line = r.readLine()) != null) {
        Matcher m = pattStaticKey.matcher(line);
        while (m.find()) {
          int count = m.groupCount();
          for (int i = 1; i <= count; i++) {
            String key = m.group(i);
            if (reg.hasString(key)) {
              validation.get(reg).addKeyDefined(key);
            }
            else {
              validation.get(reg).addKeyNotDefined(file.getName(), key);
            }
          }
        }
      }
    }
    finally {
      if (r != null) {
        r.close();
      }
    }
  }
}
