package csbase.console;

import java.io.File;
import java.io.InputStream;

import csbase.client.ClientLocalFile;
import csbase.logic.ClientFile;
import csbase.logic.SyncRemoteFileChannel;
import csbase.logic.User;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.remote.ClientRemoteLocator;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Cliente sem interface grfica, para realizar a importao de uma verso de
 * Algoritmo. Esse cliente deve ser executado diretamente da linha de comando.
 *
 * TODO: Avaliar se vale a pena substituir ClientLocalFile por RandomAccessFile,
 * para remover dependncia de csbase-client.
 *
 * @author Tecgraf
 */
public class ImportAlgoVersion extends AbstractConsoleApp {

  /**
   * Construtor do cliente para importar uma verso de algoritmo. Obtm a senha
   * interativamente.
   *
   * @param args parmetros fornecidos na linha de comando
   */
  ImportAlgoVersion(String[] args) {
    super(args);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected BasicParams createParams() {
    return new AlgorithmsVersionParams();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void postLoginInit() {
  }

  /**
   * Efetua a transferncia do arquivo local atravs do canal remoto.
   *
   * @param file referncia para o arquivo local (aplicao standalone)
   * @param info informaes do canal remoto de transferncia (FTC)
   * @return true se a transferncia ocorreu com sucesso, caso contrrio,
   *         retorna false
   * @throws Exception Em caso de falha na operao.
   */
  private static boolean transfer(ClientFile file, RemoteFileChannelInfo info)
    throws Exception {
    SyncRemoteFileChannel channel =
      new SyncRemoteFileChannel(info.getIdentifier(), info.isWritable(),
        info.getHost(), info.getPort(), info.getKey());
    channel.open(!info.isWritable());
    try (InputStream inputStream = file.getInputStream()) {
      long totalBytes = channel.syncTransferFrom(inputStream, 0, file.size());
      return (totalBytes == file.size());
    }
    finally {
      channel.close();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean preLogout() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getLogin() {
    AlgorithmsVersionParams params = (AlgorithmsVersionParams) getParams();
    String login = (String) (params.userLogin == null ? User.getAdminId()
      : params.userLogin);
    return login;
  }

  /**
   * Obtm o arquivo local que ser copiado para o diretrio remoto.
   *
   * @param localSrcPath path para o arquivo local
   * @return o arquivo local, ou <code>null</code> caso este no exista ou seja
   *         um diretrio
   */
  private File getLocalSrcFile(String localSrcPath) {
    File localFile;
    System.out.println(localSrcPath);
    if (localSrcPath.charAt(0) != '/') {
      localFile = new File(getCurrentDir(), localSrcPath);
    }
    else {
      localFile = new File(localSrcPath);
    }
    if (!localFile.exists()) {
      printError(localSrcPath + " no existe");
      return null;
    }
    if (localFile.isDirectory()) {
      printError(localSrcPath + "  um diretrio");
      return null;
    }
    return localFile;
  }

  /**
   * Interpreta os parmetros e faz as respectivas requisies ao servidor.
   *
   * @throws Exception
   */
  private void execute() throws Exception {
    AlgorithmsVersionParams params = (AlgorithmsVersionParams) getParams();
    startAlgorithmImport(params.algorithmId, params.versionZipPath);
    return;
  }

  /**
   * Inicia o processo de manipulao do PA para importao. Esse comando
   * realiza a transferncia do PA para o servidor e retorna um token para o
   * dado do PA que requisitou essa iniciao. Esse token dever ser passado
   * para todas as demais operaes sobre esse PA.
   *
   * @param algorithmId Id do algoritmo.
   * @param zipPath caminho do arquivo zipado da verso do algoritmo.
   *
   * @throws Exception
   */
  private void startAlgorithmImport(String algorithmId, String zipPath)
    throws Exception {
    /*
     * TODO: Cast pra Object necessrio pois alm do getInfo(Object) desejado,
     * existe um mtodo getInfo(String) que espera no o id mas o nome do
     * algoritmo. Talvez fosse o caso destes mtodos terem nomes diferentes.
     */
    AlgorithmInfo info =
      ClientRemoteLocator.algorithmService.getInfo((Object) algorithmId);
    if (info == null) {
      System.err.println("Id de algoritmo invlido.");
      return;
    }
    String filename = FileUtils.getFileName(zipPath);
    RemoteFileChannelInfo channelInfo = ClientRemoteLocator.algorithmService
      .prepareUploadVersionPack(algorithmId, filename);

    File localFile = getLocalSrcFile(zipPath);
    if (localFile == null) {
      System.err.println("Path inexistente para verso de algoritmo.");
      return;
    }
    ClientFile clientFile = new ClientLocalFile(localFile);
    if (!transfer(clientFile, channelInfo)) {
      logout();
    }
  }

  /**
   * Executa o cliente de importao de uma verso de algoritmo.
   *
   * @param args parmetros da linha de comando
   * @throws Exception em caso de erro na parametrizao.
   */
  public static void main(String[] args) throws Exception {
    ImportAlgoVersion importVersionClient = new ImportAlgoVersion(args);
    if (importVersionClient.login()) {
      System.out
        .println("Processo de importao de uma Verso de Algoritmo...");
      importVersionClient.execute();
      importVersionClient.logout();
    }
  }

}
