package csbase.server.plugin.service.sgaservice;

import java.util.Map;

/**
 * Interface pblica do mecanismo de tranasferncia de dados entre o servidor
 * CSBase e o SGA.
 *
 * @author Tecgraf/PUC-Rio
 */
public interface ISGADataTransfer {

  /**
   * Nome da propriedade que o SGA define para o mecanismo de transferncia
   * adotado.
   */
  public static final String TRANSFER_NAME_PROPERTY = "csbase_transfer_name";

  /**
   * Nome da propriedade que possui a raiz a ser usada no target da cpia. O
   * valor definido na IDL do SGA (SGA_CSFS_ROOT_DIR);
   */
  public static final String ROOT = "csbase_csfs_root_dir";

  /**
   * Atribui as proprieades do SGA sobre o qual a instncia do mecanismo atua.
   *
   * @param sgaProperties
   */
  public abstract void setSGAProperties(java.util.Properties sgaProperties);

  /**
   * Copia um arquivo da rea de armazenamento principal para uma mquina de
   * execuo. O caminho final na mquina de execuo  dependente da
   * implementaa do mecanismo de transferncia.
   *
   * @param source o caminho absoluto do arquivo de origem
   * @param target o caminho do arquivo relativo ao diretrio definido em
   *        {@link ISGADataTransfer#ROOT ROOT} e que ser criado no destino
   * @throws SGADataTransferException em caso de erro.
   */
  public abstract void copyTo(String[] source, String[] target)
    throws SGADataTransferException;

  /**
   * Copia um arquivo de uma mquina de execuo para a rea de armazenamento
   * principal.
   *
   * @param source o caminho do arquivo, relativo ao diretrio definido em
   *        {@link ISGADataTransfer#ROOT ROOT}, na mquina de execucao
   * @param target o caminho absoluto do arquivo a ser criado na rea de
   *        armazenamento principal.
   * @throws SGADataTransferException em caso de erro.
   */
  public abstract void copyFrom(String[] source, String[] target)
    throws SGADataTransferException;

  /**
   * Cria um diretrio em uma mquina.
   *
   * @param path o caminho absoluto do diretrio
   * @throws SGADataTransferException em caso de erro.
   */
  public abstract void createDirectory(String[] path)
    throws SGADataTransferException;

  /**
   * Remove uma rvore remota recursivamente no servidor especificado.
   *
   * @param path o caminho absoluto do arquivo
   * @throws SGADataTransferException em caso de erro.
   */
  public abstract void remove(String[] path) throws SGADataTransferException;

  /**
   *
   * Verifica a existncia de um caminho em um determinado servidor.
   *
   * @param path o caminho absoluto do arquivo
   * @return a indicao de existncia
   * @throws SGADataTransferException em caso de erro.
   */
  public abstract boolean checkExistence(String[] path)
    throws SGADataTransferException;

  /**
   * Obtm o caminho para o mount-point do repositrio de projetos.
   *
   * @return o caminho para o repositrio de projetos.
   * @throws SGADataTransferException em caso de erro.
   */
  public abstract String[] getProjectsRootPath()
    throws SGADataTransferException;

  /**
   * Obtm o caminho para o mount-point do repositrio de algoritmos.
   *
   * @return o caminho para o repositrio de algoritmos.
   * @throws SGADataTransferException em caso de erro.
   */
  public abstract String[] getAlgorithmsRootPath()
    throws SGADataTransferException;

  /**
   * Obtm um mapeamento entre um arquivo e sua data de modificao para todos
   * os arquivos de uma arvore de diretrios, dentro da mquina de execuo.
   *
   * @param path caminho relativo ao diretorio {@link ISGADataTransfer#ROOT
   *        ROOT}
   *
   * @return mapeamento entre um arquivo e sua data de modificao para todos os
   *         arquivos de uma rvore de diretrios, dentro da mquina de
   *         execuo.
   *
   * @throws SGADataTransferException em caso de erro.
   */
  public abstract Map<String[], Long> getRemoteTimestamps(String[] path)
    throws SGADataTransferException;

  /**
   * Obtm um mapeamento entre um arquivo e sua data de modificao para todos
   * os arquivos de uma arvore de diretrios, dentro do servidor CSBase.
   *
   * @param path caminho absoluto ao diretorio.
   *
   * @return mapeamento entre um arquivo e sua data de modificao para todos os
   *         arquivos de uma rvore de diretrios, dentro do servidor CSBase.
   *
   * @throws SGADataTransferException em caso de erro.
   */
  public abstract Map<String[], Long> getLocalTimestamps(String[] path)
    throws SGADataTransferException;
}
