/*
 * $Id: PermissionIO.java 115229 2011-02-03 19:57:31Z cassino $
 */
package csbase.server.services.administrationservice;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Vector;

import csbase.logic.AttributesPermission;
import csbase.logic.ChoicePermission;
import csbase.logic.IdFactory;
import csbase.logic.Permission;
import csbase.logic.UserPasswordPermission;
import csbase.server.Server;

/**
 * A classe <code>PermissionIO</code>  responsvel pela persistncia dos
 * registros das Permissions. Todos os objetos so mantidos em um arquivo texto
 * que  atualizado sempre que os mtodos de atualizao so invocados. 
 * responsvel tambm pela gerao do identificador de uma Permission. O
 * identificador  um inteiro sequencial. O algoritmo reutiliza os valores de
 * identificadores liberados pela remoo de Permissions.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class PermissionIO {
  /**
   * O gerador de identificadores de UNs
   */
  private IdFactory idFactory;

  /**
   * Chave do nome da Permission na Hashtable persistida
   */
  private static final String NAME = "name";

  /**
   * Chave da descrio da Permission na Hashtable persistida
   */
  private static final String DESCRIPTION = "description";

  /**
   * Chave da class da Permission na Hashtable persistida
   */
  private static final String CLASS_NAME = "class";

  /**
   * Chave do usurio da Permission na Hashtable persistida
   */
  private static final String USER = "user";

  /**
   * Chave da senha da Permission na Hashtable persistida
   */
  private static final String PASSWORD = "password";

  /**
   * Chave dos atributos da Permission na Hashtable persistida
   */
  private static final String ATTRIBUTES = "attributes";

  /**
   * Nome da propriedade que possui o valor do nome do arquivo de Permissions
   */
  private static String FILENAME_PROPERTY = "PermissionIO.filename";

  /**
   * Nome do arquivo default de Permissions, usado no caso da propriedade acima
   * no existir
   */
  private static String FILENAME = "permission.dat";

  /**
   * Nome do arquivo de Permissions no existir.
   */
  private static String fileName = FILENAME;

  /**
   * Constri um objeto responsvel pela persistncia de Permissions. O nome do
   * arquivo de dados  obtido da propriedade <b>PermissionIO.filename</b>.
   */
  public PermissionIO() {
    final Server server = Server.getInstance();
    AdministrationService admSrv = AdministrationService.getInstance();
    fileName = server.getPersistencyRootDirectoryName() + File.separator
      + admSrv.getStringProperty(FILENAME_PROPERTY);
    Server.logInfoMessage("Arquivo de permisses: " + fileName);
  }

  /**
   * Cria uma nova Permission e grava no arquivo.
   * 
   * @param permission os dados a serem usados na criao da Permission
   * @return uma nova permissionidade de negcio
   * @throws Exception
   */
  public synchronized Permission writeNew(Permission permission)
    throws Exception {
    Vector<Permission> permissions = new Vector<Permission>(Permission
      .getAllPermissions());
    if (idFactory == null) {
      idFactory = new IdFactory(permissions);
    }
    Object id = idFactory.next();
    permission.setId(id);
    if (permission instanceof UserPasswordPermission) {
      ajustPassword((UserPasswordPermission) permission);
    }
    permissions.add(permission);
    writeAll(permissions);
    return permission;
  }

  /**
   * Grava uma Permission no arquivo. A Permission substitui outra com o mesmo
   * identificador.
   * 
   * @param id o identificador da Permission a ser gravada
   * @param permission os dados da Permission a ser gravada
   * @return a permissionidade de negcio gravada
   * @throws Exception se essa Permission no existir no arquivo ou se ocorrer
   *         algum erro de gravao
   */
  public synchronized Permission write(Object id, Permission permission)
    throws Exception {
    try {
      Permission oldPermission = Permission.getPermission(id);
      Vector<Permission> permissions = new Vector<Permission>(Permission
        .getAllPermissions());
      if (!permissions.remove(oldPermission)) {
        throw new Exception("Permission no existe");
      }
      permission.setId(id);

      // Se a senha no foi passada na alterao, permanece a mesma.
      if (permission instanceof UserPasswordPermission) {
        String password = ((UserPasswordPermission) permission).getPassword();
        if ((password == null) || (password.equals(""))) {
          ((UserPasswordPermission) permission)
            .setLocalPassword(((UserPasswordPermission) oldPermission)
              .getLocalPassword());
        }
        else {
          ajustPassword((UserPasswordPermission) permission);
        }
      }
      permissions.add(permission);
      writeAll(permissions);
      return permission;
    }
    catch (Exception e) {
      throw new Exception(e.getMessage());
    }
  }

  /**
   * Grava uma nova permisso no arquivo.
   * 
   * @param permission a permisso a ser gravado.
   * @return a permisso
   * @throws Exception erro durante a criao da permisso
   */
  public synchronized Permission write(Permission permission) throws Exception {
    Vector<Permission> permissions = new Vector<Permission>(Permission
      .getAllPermissions());
    Object id = permission.getId();
    if (Permission.getPermission(id) != null) {
      throw new Exception("Permisso j existe");
    }
    permissions.add(permission);
    writeAll(permissions);
    return permission;
  }

  /**
   * Coloca a senha como atributo transient da UserPasswordPermission.
   * 
   * @param permission
   */
  private void ajustPassword(UserPasswordPermission permission) {
    permission.setLocalPassword(permission.getPassword());
    permission.setPassword(null);
  }

  /**
   * Remove uma Permission do arquivo.
   * 
   * @param id o identificador da permission a ser removida
   * @throws Exception se essa Permission no existe no arquivo ou se ocorrer
   *         algum erro de gravao
   */
  public synchronized void delete(Object id) throws Exception {
    Vector<Permission> permissions;
    Permission permission;
    try {
      permission = Permission.getPermission(id);
      permissions = new Vector<Permission>(Permission.getAllPermissions());
      if (idFactory == null) {
        idFactory = new IdFactory(permissions);
      }
    }
    catch (Exception e) {
      throw new Exception();
    }
    if (!permissions.remove(permission)) {
      throw new Exception("Permission no existe");
    }
    writeAll(permissions);
    idFactory.free(permission.getId());
  }

  /**
   * Salva todas as Permissions no arquivo de dados.
   * 
   * @param permissions as Permissions a serem gravadas no arquivo
   * @throws Exception a operao no pode ser efetuada
   */
  private synchronized void writeAll(List<Permission> permissions)
    throws Exception {
    Hashtable<Object, Hashtable<String, Object>> permissionsHash = new Hashtable<Object, Hashtable<String, Object>>();
    for (Permission permission : permissions) {
      Hashtable<String, Object> permissionHash = new Hashtable<String, Object>();
      permissionHash.put(NAME, permission.getName());
      permissionHash.put(DESCRIPTION, permission.getDescription());
      permissionHash.put(CLASS_NAME, permission.getClass().getName());
      if (permission instanceof AttributesPermission) {
        permissionHash.put(ATTRIBUTES, ((AttributesPermission) permission)
          .getAttributes());
      }
      else if (permission instanceof UserPasswordPermission) {
        permissionHash.put(USER, ((UserPasswordPermission) permission)
          .getUser());
        String key = Server.getInstance().getAdminPassword();
        byte[] crypted = FileCrypto.encrypt(key,
          ((UserPasswordPermission) permission).getLocalPassword());
        permissionHash.put(PASSWORD, crypted);
      }
      else if (permission instanceof ChoicePermission) {
        permissionHash.put(ATTRIBUTES, ((ChoicePermission) permission)
          .getAttributes());
      }
      permissionsHash.put(permission.getId(), permissionHash);
    }
    ObjectOutputStream out = new ObjectOutputStream(new DataOutputStream(
      new BufferedOutputStream(new FileOutputStream(fileName))));
    try {
      out.writeObject(permissionsHash);
    }
    finally {
      out.close();
    }
  }

  /**
   * Devolve uma Permission que possua um determinado identificador.
   * 
   * @param id o identificador da Permission procurada
   * @return a Permission procurada ou null caso essa Permission no exista
   * @throws Exception
   */
  public synchronized Permission read(Object id) throws Exception {
    for (Permission permission : Permission.getAllPermissions()) {
      if (permission.getId().equals(id)) {
        return permission;
      }
    }
    return null;
  }

  /**
   * L e retorna todas as Permissions existentes no arquivo de dados. Caso
   * falhe na leitura de alguma, grava no arquivo apenas as validas.
   * 
   * @return um vetor com todas as Permissions existentes no sistema
   */
  public synchronized List<Permission> readAll() {
    AdministrationService srv = AdministrationService.getInstance();
    List<Permission> permissions = new Vector<Permission>();
    File permissionsFile = new File(fileName);
    if (!permissionsFile.exists()) {
      /*
       * arquivo de permisses ainda no existe, registramos e retornamos uma
       * lista vazia
       */
      Server.logInfoMessage("Criando novo arquivo de permisses " + fileName);
      return permissions;
    }
    ObjectInputStream in = null;
    try {
      in = new ObjectInputStream(new DataInputStream(new BufferedInputStream(
        new FileInputStream(permissionsFile))));
      Hashtable<Object, Hashtable<String, Object>> permissionsHash = (Hashtable<Object, Hashtable<String, Object>>) in
        .readObject();
      boolean allRead = true;
      for (Entry<Object, Hashtable<String, Object>> permEntry : permissionsHash
        .entrySet()) {
        Object id = permEntry.getKey();
        Hashtable<String, Object> permissionHash = permEntry.getValue();
        Permission permission = makePermission(id, permissionHash);
        if (permission != null) {
          permissions.add(permission);
        }
        else {
          allRead = false;
        }
      }
      if (!allRead) {
        try {
          Server.logInfoMessage("Persistindo apenas as permisses vlidas.");
          writeAll(permissions);
        }
        catch (Exception e) {
          Server.logSevereMessage("Falha na persistencia de apenas as "
            + "permisses vlidas.", e);
        }
      }
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro durante a leitura e descriptografia das "
        + "permisso de usurios", e);
    }
    finally {
      if (in != null) {
        try {
          in.close();
        }
        catch (IOException e) {
          Server.logSevereMessage("Erro fechando arquivo de permisses");
        }
      }
    }
    return permissions;
  }

  /**
   * Cria uma permisso a partir dos dados lidos.
   * 
   * @param id Identificador da permisso.
   * @param permissionHash Dados da permisso numa Hashtable.
   * @return Permisso lida a partir da Hash.
   */
  private Permission makePermission(Object id,
    Hashtable<String, Object> permissionHash) {
    try {
      String name = (String) permissionHash.get(NAME);
      String desc = (String) permissionHash.get(DESCRIPTION);
      String className = (String) permissionHash.get(CLASS_NAME);
      Class<?> permissionClass = Class.forName(className);
      Permission permission = (Permission) permissionClass.newInstance();
      permission.setId(id);
      permission.setName(name);
      permission.setDescription(desc);
      if (permission instanceof AttributesPermission) {
        String[] attributes = (String[]) permissionHash.get(ATTRIBUTES);
        ((AttributesPermission) permission).setAttributes(attributes);
      }
      else if (permission instanceof UserPasswordPermission) {
        String user = (String) permissionHash.get(USER);
        byte[] pass = (byte[]) permissionHash.get(PASSWORD);
        String key = Server.getInstance().getAdminPassword();
        String decrypted = FileCrypto.decrypt(key, pass);
        ((UserPasswordPermission) permission).setUser(user);
        ((UserPasswordPermission) permission).setLocalPassword(decrypted);
      }
      else if (permission instanceof ChoicePermission) {
        ((ChoicePermission) permission)
          .setAttributes((Map<?, ?>) permissionHash.get(ATTRIBUTES));
      }
      return permission;
    }
    catch (Exception e) {
      AdministrationService srv = AdministrationService.getInstance();
      if (srv != null) {
        Server.logSevereMessage("Falha na leitura da Permission: " + id
          + " Message: " + e.getMessage());
      }
      return null;
    }
  }
}
