/*
 * $Id$
 */
package csbase.server.services.loginservice;

import java.util.Hashtable;
import java.util.Map;

import csbase.exception.CSBaseException;
import csbase.logic.ServerMonitor;
import csbase.logic.ServerURI;
import csbase.remote.ServerEntryPoint;
import csbase.server.Server;

/**
 * 
 * Cache de servidores contactados para login por referncia.  mantido uma
 * monitorao usando {@link ServerMonitor} para cada servidor no cache afim de
 * minimizar o tempo gasto contactando o servidor de referncia.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ReferedServerCache {

  /**
   * Monitor dos servidores.
   * 
   * @author Tecgraf/PUC-Rio
   * 
   */
  private final class ReferedServerMonitor extends ServerMonitor {

    /**
     * {@inheritDoc}
     */
    @Override
    protected void logError(final String msg, final Throwable t) {
      Server.logSevereMessage(msg, t);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void logInfo(final String msg) {
      Server.logInfoMessage(msg);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void logWarning(final String msg) {
      Server.logWarningMessage(msg);
    }

    /**
     * Construtor
     * 
     * @param serverURI URI
     */
    public ReferedServerMonitor(final ServerURI serverURI) {
      super(serverURI, maxRetries);
    }

  }

  /**
   * A instncia nica do cache
   */
  private static ReferedServerCache instance = null;

  /**
   * Nmero mximo de tentativas para contactar o servidor
   */
  private final int maxRetries = 3;

  /**
   * Mapa dos servidores referenciados
   */
  private final Map<ServerURI, ServerMonitor> referedServers;

  /**
   * Finaliza a monitorao dos servidores disponveis.
   */
  public final void clear() {
    synchronized (this.referedServers) {
      for (final ServerMonitor monitor : this.referedServers.values()) {
        try {
          monitor.stopMonitoring();
        }
        catch (final Throwable t) {
          final ServerURI uri = monitor.getURI();
          Server.logSevereMessage("Erro finalizando monitorao de "
            + uri, t);
        }
      }
      this.referedServers.clear();
    }
  }

  /**
   * Retorna o ponto de entrada de um servidor. Se o servidor no estiver
   * alcanvel  feito um lookup para tentar obter a referncia para o mesmo e
   * caso o lookup tenha sucesso  iniciada a monitorao
   * 
   * @param serverURI A URI do servidor
   * @return O ponto de entrada do servidor ou null se o servidor estiver fora
   *         do ar.
   * @throws CSBaseException Se o servidor no est sendo monitorado.
   */
  public final ServerEntryPoint getServer(final ServerURI serverURI)
    throws CSBaseException {

    ServerMonitor monitor = null;

    synchronized (this.referedServers) {
      monitor = this.referedServers.get(serverURI);
      if (monitor == null) {
        monitor = new ReferedServerMonitor(serverURI);
        this.referedServers.put(serverURI, monitor);
      }
    }

    if (!monitor.isReachable()) {
      if (monitor.lookup()) {
        monitor.startMonitoring();
      }
    }
    return monitor.getServer();
  }

  /**
   * Busca do singleton
   * 
   * @return instncia
   */
  public static ReferedServerCache getInstance() {
    if (ReferedServerCache.instance == null) {
      ReferedServerCache.instance = new ReferedServerCache();
    }
    return ReferedServerCache.instance;
  }

  /**
   * Construtor
   */
  private ReferedServerCache() {
    this.referedServers = new Hashtable<ServerURI, ServerMonitor>();
  }
}
