/**
 * $Id$
 */

package csbase.server.services.messageservice;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.List;
import java.util.Vector;

import tecgraf.javautils.core.filter.IFilter;
import csbase.logic.User;
import csbase.logic.UserOutline;
import csbase.remote.IRemoteMessageListener;
import csbase.remote.MessageServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.loginservice.LoginService;
import csbase.server.services.loginservice.LoginServiceListener;
import csbase.server.services.loginservice.ServerSession;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;
import csbase.util.messages.MessageBroker;

/**
 * Implementa o servio de mensagens.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class MessageService extends Service implements MessageServiceInterface,
  LoginServiceListener {

  /**
   * Chave para a propriedade da sesso do usurio que mantm referncia para o
   * identificador do usurio como consumidor de mensagens do
   * {@link MessageBroker} do {@link Server}.
   * 
   * @see Server#getMessageBroker()
   */
  private static final String CONSUMER_ID_SESSION_PROPERTY = SERVICE_NAME
    + ".consumer_id";

  /**
   * Referncia ao MessageBroker
   */
  private MessageBroker broker;

  /**
   * Nome do tpico utilizado para troca de mensagens entre servios no
   * servidor.
   */
  private String serverTopicName = "SERVER_TOPIC";;

  /**
   * Obtm a instncia nica do servio.
   * 
   * @return A instncia nica do servio.
   */
  public static MessageService getInstance() {
    return (MessageService) getInstance(SERVICE_NAME);
  }

  /**
   * Cria a instncia nica do servio.
   * 
   * @throws ServerException Se houver erro na inicializao.
   */
  public static void createService() throws ServerException {
    new MessageService();
  }

  /**
   * Construtor do servio.
   * 
   * @throws ServerException em caso de falha na construo do servio.
   */
  public MessageService() throws ServerException {
    super(SERVICE_NAME);
  }

  /*
   * Mtodos publicos da interface MessageServiceInterface
   */

  /**
   * {@inheritDoc}
   */
  @Override
  public void broadcast(Message message, long timeToLive)
    throws RemoteException {

    if (!isActive()) {
      return;
    }

    Object senderName = getUser().getLogin();
    try {
      List<User> users = User.getAllUsers();
      String[] logins = new String[users.size() - 1];
      for (int inx = 0; inx < users.size(); inx++) {
        String aLogin = users.get(inx).getLogin();
        if (!aLogin.equals(senderName)) {
          logins[inx] = aLogin;
        }
      }
      broker.send(message, timeToLive, logins);
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha de notificao all-users.", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void send(Message message, long timeToLive, String... usersLogins)
    throws RemoteException {

    if (!isActive()) {
      return;
    }

    broker.send(message, timeToLive, usersLogins);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Message[] receive(IFilter<Message> filter) throws RemoteException {

    if (!isActive()) {
      return new Message[0];
    }

    String topicName = getUser().getLogin();
    Serializable consumerId = getConsumerId();
    return broker.receive(topicName, consumerId, filter);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void clearMessageListener() throws RemoteException {
    if (!isActive()) {
      return;
    }

    String topicName = getUser().getLogin();
    Serializable consumerId = getConsumerId();

    broker.removeMessageListener(topicName, consumerId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setMessageListener(final IRemoteMessageListener listener,
    final IFilter<Message> filter) throws RemoteException {

    if (!isActive()) {
      return;
    }

    String topicName = getUser().getLogin();
    Serializable consumerId = getConsumerId();

    broker.setMessageListener(topicName, consumerId, listener, filter);
  }

  /*
   * Mtodos da interface LoginServiceListener
   */

  /**
   * {@inheritDoc}
   */
  @Override
  public void systemNameSet(String login, Object sessionKey, String systemName,
    long time) {
    // Nada a ser feito!
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void sessionCreated(String login, Object sessionKey, long time) {
    // Nada a ser feito!       
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void connectionLost(String login, Object sessionKey,
    String systemName, long time) {
    // No precisa tratar, pois o userLoggingOut  chamado aps o conncetionLost!
  }

  /**
   * <p>
   * Trata o fim da sesso do usurio, seja por logout, ou connection lost,
   * removendo ouvinte de mensagens daquela sesso.
   * </p>
   * 
   * @param login Login do usurio.
   * @param sessionKey Sesso do usurio.
   */
  @Override
  public void userLoggingOut(String login, Object sessionKey,
    String systemName, long time) {
    LoginService loginService = LoginService.getInstance();
    Serializable consumerId =
      loginService.getSessionProperty(sessionKey, CONSUMER_ID_SESSION_PROPERTY);
    if (consumerId != null) {
      broker.removeMessageListener(login, consumerId);
    }
  }

  /*
   * Mtodos da classe abstrata Service
   */

  /**
   * {@inheritDoc}
   */
  @Override
  protected void initService() throws ServerException {
    LoginService.getInstance().addListener(this);
    broker = Server.getInstance().getMessageBroker();
    broker.start();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void shutdownService() throws ServerException {
    LoginService.getInstance().removeListener(this);
    broker.stop();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return true;
  }

  /*
   * Mtodos para a troca de mensagens entre objetos localizados no servidor.
   * Esses mtodos no so visveis na interface exportada para os clientes.
   */

  /**
   * Envia uma mensagem a vrios destinatrios.
   * 
   * @param message Mensagem a ser enviada.
   * @param usersLogins Login dos usurios de destino.
   * 
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   */
  public void send(Message message, String... usersLogins)
    throws RemoteException {
    send(message, 0, usersLogins);
  }

  /**
   * Envia uma mensagem para todos os usurios ativos no momento.
   * 
   * @param message Mensagem a ser enviada.
   */
  public void sendToAll(Message message) {
    try {
      UserOutline[] users = LoginService.getInstance().getLoggedUsers();
      String logins[] = new String[users.length];
      for (int i = 0; i < users.length; i++) {
        logins[i] = users[i].getLogin();
      }

      send(message, logins);
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha de notificao sendToAll.", e);
    }
  }

  /**
   * Envia uma mensagem para o administrator.
   * 
   * @param message Mensagem a ser enviada.
   * @param timeToLive Tempo, em milisegundos, que essa mensagem deve persistir
   *        at que seja consumida.
   * 
   * @throws RemoteException
   */
  public void sendToAdmin(Message message, long timeToLive)
    throws RemoteException {
    Vector<Object> ids = new Vector<Object>();

    try {
      ids.addAll(User.getAdminIds());
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Erro ao obter a lista de usurios adminstradores.", e);
    }

    send(message, timeToLive, ids.toArray(new String[0]));
  }

  /**
   * Envia uma mensagem para o administrator.
   * 
   * @param message Mensagem a ser enviada.
   * 
   * @throws RemoteException
   */
  public void sendToAdmin(Message message) throws RemoteException {
    sendToAdmin(message, 0);
  }

  /**
   * Envia uma mensagem para o prprio servidor.
   * 
   * @param message Mensagem a ser enviada.
   */
  public void sendToServer(Message message) {
    try {
      send(message, 0, serverTopicName);
    }
    catch (RemoteException e) {
      e.printStackTrace();
    }
  }

  /**
   * Remove do servio de mensagens o ouvinte de mensagens para o identificador
   * especificado. Utilizado para objetos no servidor.
   * 
   * @param consumerId identificador do consumidor
   */
  public void clearServerMessageListener(Serializable consumerId) {
    if (consumerId == null) {
      return;
    }
    if (!isActive()) {
      return;
    }

    broker.removeMessageListener(serverTopicName, consumerId);
  }

  /**
   * Atribui o ouvinte de mensagens para objetos no servidor ao servio de
   * mensagens.
   * 
   * @param listener Ouvinte de mensagens.
   * @param filter Filtro que determina as mensagens que sero repassadas ao
   *        ouvinte.
   * 
   * @return o identificador do consumidor no servio de mensagens
   */
  public Serializable setServerMessageListener(final IMessageListener listener,
    final IFilter<Message> filter) {

    if (!isActive()) {
      return null;
    }

    Serializable consumerId = broker.createConsumerId();
    broker.setMessageListener(serverTopicName, consumerId, listener, filter);

    return consumerId;
  }

  /**
   * Remove do servio de mensagens, o ouvinte de mensagens de uma sesso
   * especfica.
   * 
   * @param session a sesso
   */
  public void clearMessageListener(ServerSession session) {
    if (!isActive()) {
      return;
    }

    String topicName = session.getUser().getLogin();
    Serializable consumerId = getConsumerId(session);
    if (consumerId != null) {
      broker.removeMessageListener(topicName, consumerId);
    }
  }

  /*
   * Mtodos privados
   */

  /**
   * Obtm o identificador do consumidor para a sesso que fez a chamada.
   * 
   * @return o identificador do consumidor para a sesso que fez a chamada.
   */
  private synchronized Serializable getConsumerId() {
    LoginService loginService = LoginService.getInstance();

    Serializable consumerId =
      loginService.getSessionProperty(getKey(), CONSUMER_ID_SESSION_PROPERTY);
    if (consumerId == null) {
      consumerId = broker.createConsumerId();
      loginService.setSessionProperty(getKey(), CONSUMER_ID_SESSION_PROPERTY,
        consumerId);
    }

    return consumerId;
  }

  /**
   * Obtm o identificador do consumidor de uma sesso especfica.
   * 
   * @param session a sesso
   * 
   * @return o identificador do consumidor de uma sesso especfica.
   */
  private Serializable getConsumerId(ServerSession session) {
    return (Serializable) session.getProperty(CONSUMER_ID_SESSION_PROPERTY);
  }
}
