package csbase.server.services.opendreamsservice.opendreams.v1_8;

import java.lang.reflect.Field;

import org.omg.CORBA.UserException;
import org.omg.PortableServer.POA;

import tecgraf.openbus.DRMAA.v1_8.AuthorizationException;
import tecgraf.openbus.DRMAA.v1_8.InternalException;
import tecgraf.openbus.DRMAA.v1_8.Session;
import tecgraf.openbus.DRMAA.v1_8.SessionHelper;
import tecgraf.openbus.opendreams.v1_8.ProjectNotFoundException;
import csbase.exception.PermissionException;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.administrationservice.AdministrationService;
import csbase.server.services.openbusservice.OpenBusService;
import csbase.server.services.projectservice.ProjectService;

/**
 * Implementa a idl <code>IOpenDreams</code>.
 * 
 * Recupera a sesso ativa associada a um determinado usurio e a um projeto
 * csbase. Se a sesso ainda no existir, cria a sesso.
 * 
 * @author Tecgraf PUC-Rio
 */
public class OpenDreams extends tecgraf.openbus.opendreams.v1_8.IOpenDreamsPOA {
  /**
   * Referncia para a sesso do opendreams
   */
  private Session session;
  /**
   * Referncia para o gerenciador das sesses
   */
  private SessionManager sessionManagement;

  /**
   * No definido.
   */
  public static final String UNDEFINED = "indefinido";

  /**
   * Constri o objeto que implementa a interface <code>IOpenDremas</code>.
   * 
   */
  public OpenDreams() {
    sessionManagement = new SessionManager();
  }

  /**
   * Obtm a sesso passando apenas o nome do projeto. O owner do projeto  o
   * usurio da credencial. {@inheritDoc}
   */
  @Override
  public Session getSession(String projectName) throws AuthorizationException,
    InternalException, ProjectNotFoundException {
    org.omg.CORBA.Object obj;
    String userId = UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreams: getSession (" + projectName + ")");
      checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      checkProject(projectName, userId); // Verifica se o projeto existe
      POA poa = OpenBusService.getInstance().getRootPOA();
      obj =
        poa.servant_to_reference(sessionManagement.getSession(userId,
          projectName));
      session = SessionHelper.narrow(obj);
    }
    catch (AuthorizationException e) {
      String msg =
        "Falha na tentativa de obter uma sesso do usurio " + userId
          + " para o projeto " + projectName + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (ProjectNotFoundException e) {
      String msg =
        "Falha na tentativa de obter uma sesso do usurio " + userId
          + " para o projeto " + projectName + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg =
        "Erro ao obter a sesso do usurio " + userId + " para o projeto "
          + projectName;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
    return session;
  }

  /**
   * Obtm a sesso passando o nome do projeto e o usurio owner do projeto.
   * {@inheritDoc}
   */
  @Override
  public Session getSessionByProjUserId(String projectName, String userName)
    throws AuthorizationException, InternalException, ProjectNotFoundException {
    org.omg.CORBA.Object obj;
    String userId = UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreams: getSessionByProjUserId ("
        + projectName + ", " + userName + ")");
      checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      checkProject(projectName, userName); // Verifica se o projeto existe
      checkProjectPermission(projectName, userName); // Verifica se o usurio da credencial tem acesso ao projeto
      POA poa = OpenBusService.getInstance().getRootPOA();
      obj =
        poa.servant_to_reference(sessionManagement.getSession(userName,
          projectName));
      session = SessionHelper.narrow(obj);
    }
    catch (AuthorizationException e) {
      String msg =
        "Falha na tentativa de obter uma sesso do usurio " + userId
          + " para o projeto " + projectName + " do usurio " + userName + ": "
          + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (ProjectNotFoundException e) {
      String msg =
        "Falha na tentativa de obter uma sesso do usurio " + userId
          + " para o projeto " + projectName + " do usurio " + userName + ": "
          + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg =
        "Erro ao obter a sesso do usurio " + userId + " para o projeto "
          + projectName + " do usurio " + userName;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
    return session;
  }

  /**
   * Cria uma mensagem formatada com o texto de uma mensagem junto com
   * informaes mais detalhadas sobre a prpria exceo.
   * 
   * @param e a exceo que ocorreu
   * @param msg uma mensagem descritiva do contexto da exceo
   * @return um texto formatado
   */
  public static String formatMessage(Throwable e, String msg) {
    StringBuffer message = new StringBuffer(msg);
    message.append(": " + e.getClass().getName() + ": " + e.getMessage());
    message.append("\nCaused by:\n");
    StackTraceElement[] stackTrace = e.getStackTrace();
    for (int i = 0; i < stackTrace.length; i++) {
      message.append(stackTrace[i].toString() + "\n");
    }
    return message.toString();
  }

  /**
   * Verifica se o usurio  cadastrado no CSBase.
   * 
   * @param userId identificador do usurio
   * @throws AuthorizationException caso no exista um usurio com o
   *         identificador
   */
  public static void checkUser(String userId) throws AuthorizationException {
    if (AdministrationService.getInstance().getUser(userId) == null) {
      throw new AuthorizationException("O usurio " + userId + " no existe.");
    }
  }

  /**
   * Verifica se o usurio possui um projeto cujo nome  passado como parmetro.
   * 
   * @param projectName nome do projeto
   * @param userId identificador do usurio
   * @throws ProjectNotFoundException caso no exista um projeto com o
   *         identificador passado como parmetro
   */
  public static void checkProject(String projectName, String userId)
    throws ProjectNotFoundException {
    if (AdministrationService.getInstance().getUser(userId) == null) {
      throw new ProjectNotFoundException(
        "No existe um usurio com identificador " + userId);
    }
    ProjectService projectService = ProjectService.getInstance();
    Object projectId = projectService.getProjectId(userId, projectName);
    if (!projectService.existsProject(projectId)) {
      throw new ProjectNotFoundException("O usurio " + userId
        + " no possui um projeto com nome " + projectName);
    }
  }

  /**
   * Verifica se o usurio da credencial possui acesso um projeto de um outro
   * usurio.
   * 
   * @param projectName nome do projeto
   * @param userId identificador do usurio dono do projeto
   * @throws AuthorizationException caso no exista um usurio com o
   *         identificador
   */
  public static void checkProjectPermission(String projectName, String userId)
    throws AuthorizationException {
    ProjectService projectService = ProjectService.getInstance();
    Object projectId = projectService.getProjectId(userId, projectName);
    try {
      projectService.checkWritePermission(projectId);
    }
    catch (PermissionException e) {
      throw new AuthorizationException("O usurio " + Service.getUser().getId()
        + " no possui acesso de escrita no projeto " + projectName
        + " do usurio " + userId);
    }
  }

  /**
   * Registra no log uma mensagem de warning referente a uma exceo prevista na
   * API da IDL do OpenDreams.
   * 
   * @param msg a mensagem
   * @param exception a exceo
   */
  public static void logWarningException(String msg, UserException exception) {
    try {
      Field messageField = exception.getClass().getField("message");
      Server.logWarningMessage(msg + ": " + messageField.get(exception));
    }
    catch (Exception e) {
      Server.logSevereMessage(msg, e);
    }
  }
}
