package csbase.server.services.wioservice;

import csbase.remote.EventLogServiceInterface;
import csbase.server.Server;
import csbase.server.services.eventlogservice.EventLogService;

/**
 * Classe interna do WIO-Service para auditagem do uso de arquivos e projetos
 * abertos no WIO.
 */
class WIOAudit {

  /**
   * Thread que faz a auditagem
   */
  private Thread thread = null;

  /**
   * Flag de sada da thread
   */
  private boolean exitFlag = false;

  /**
   * Servio WIO
   */
  final private WIOService service;

  /**
   * Intervalo de auditagem
   */
  final private long interval;

  /**
   * Criao de thread de monitorao de projetos abertos.
   */
  private void createThread() {
    thread = new Thread(new Runnable() {
      @Override
      public void run() {
        while (!exitFlag) {
          doFileAudit();
          doProjectAudit();
          try {
            // Server.logInfoMessage("Thread iniciada: WIOAudit.createThread");
            Thread.sleep(interval);
            // Server.logInfoMessage("Thread terminada:
            // WIOAudit.createThread");
          }
          catch (final InterruptedException ie) {
            Server.logWarningMessage("Monitorao interrompida!");
          }
        }
        Server.logWarningMessage("Finalizando monitorao.");
      }
    });
    exitFlag = false;
    thread.start();
    Server.logFineMessage("Thread de auditagem instanciada para execuo.");
  }

  /**
   * Mtodo de auditagem de projetos.
   */
  private void doFileAudit() {
    try {
      final int num = service.getNumOpenedFiles();
      final int max = service.getMaxOpenedFiles();
      logEvent("file", num, max);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha na auditagem: " + e.getMessage());
    }
  }

  /**
   * Mtodo de auditagem de projetos.
   */
  private void doProjectAudit() {
    try {
      final int num = service.getNumOpenedProjects();
      final int max = service.getMaxOpenedProjects();
      logEvent("project", num, max);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha na auditagem: " + e.getMessage());
    }
  }

  /**
   * Ajuste de fim da thread.
   */
  public void finish() {
    exitFlag = true;
  }

  /**
   * Mtodo de registro de um evento para o EventLog (servidor).
   * 
   * @param queue um identificador da fila no EventLog.
   * @param num nmero com a informao a ser registrada
   * @param max nmero mximo da informao registrada
   * @see EventLogServiceInterface
   */
  private void logEvent(final String queue, final int num, final int max) {
    final EventLogService eventService = EventLogService.getInstance();
    final String[] reQueue = new String[] { "wio", queue };
    final String[] info =
      new String[] { Integer.toString(num), Integer.toString(max), };
    try {
      eventService.addServerInformation(reQueue, info);
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha de log de eventos " + e.getMessage());
    }
  }

  /**
   * Construtor.
   * 
   * @param srv o servio de WIO.
   * @param itv o intervalo entre linhas.
   * @see WIOService
   */
  public WIOAudit(final WIOService srv, final long itv) {
    this.service = srv;
    this.interval = itv;
    createThread();
  }
}
