/*
 * $Id$
 */
package csbase.server;

import java.rmi.RemoteException;
import java.util.Locale;

import csbase.exception.CSBaseException;
import csbase.logic.MonitoredServer;
import csbase.logic.ServerURI;
import csbase.logic.Session;
import csbase.logic.SingleServerManager;
import csbase.server.keystore.CSKeyStore;

/**
 * Representa uma classe que ser usada pelo servidor local para monitorar o
 * estado da conexo com um servidor central.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class LocalServerRemoteMonitor extends SingleServerManager {
  /**
   * O locale que o monitor utilizar para efeturar login no servidor central.
   */
  private static final Locale LOCALE = new Locale("pt", "br");

  /**
   * A instncia nica do monitorador de servidor central.
   */
  private static LocalServerRemoteMonitor instance;

  /**
   * Obtm a instncia nica do monitorador de servidor central.
   * 
   * @return A instncia nica.
   */
  public static final LocalServerRemoteMonitor getInstance() {
    return instance;
  }

  /**
   * Tratamento de erro.
   * 
   * @param t a exceo.
   */
  protected final void error(final Throwable t) {
    t.printStackTrace();
    Server.logSevereMessage(t.getMessage());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String lang(final String key) {
    return Server.getInstance().getString(key);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected final Session login() throws CSBaseException {
    String monitoredVersion;
    try {
      monitoredVersion = this.getServer().getVersionName();
    }
    catch (RemoteException e1) {
      Server.logSevereMessage("Erro ao buscar verso do servidor monitorado.");
      return null;
    }
    String serverVersion = Server.getInstance().getVersion();
    if (!monitoredVersion.equals(serverVersion)) {
      Server.logSevereMessage(
        "Verso do monitor  diferente da verso do servidor monitorado.");
      return null;
    }
    final CSKeyStore keyStore = CSKeyStore.getInstance();
    if (keyStore == null) {
      Server.logSevereMessage("O repositrio de chaves no foi criado.");
      return null;
    }
    final String localServerName = Server.getInstance().getSystemName();
    if (localServerName == null) {
      Server.logSevereMessage("O nome do servidor no foi informado.");
      return null;
    }
    final String privateKeyPassword =
      Server.getInstance().getPrivateKeyPassword();
    if (privateKeyPassword == null) {
      Server
        .logSevereMessage("A senha da chave privada do servidor no foi informada.");
      return null;
    }
    byte[] signedLocalServerName;
    try {
      signedLocalServerName =
        keyStore.sign(localServerName, privateKeyPassword, localServerName);
      if (signedLocalServerName == null) {
        Server
          .logSevereMessage("O servidor no possui chave gerada no repositrio de chaves/certificados.");
        return null;
      }
    }
    catch (final CSBaseException e) {
      Server.logSevereMessage("Erro ao assinar o nome do servidor local.", e);
      throw e;
    }
    try {
      final Session session =
        this.getServer().login(localServerName, signedLocalServerName, LOCALE);
      if (session == null) {
        Server.logSevereMessage("O login no servidor central falhou.");
      }
      return session;
    }
    catch (final RemoteException e) {
      throw new CSBaseException(e);
    }
  }

  /**
   * Inicia o monitor.
   */
  public final void start() {
    this.startMonitoring();
  }

  /**
   * Tratamanto de aviso (warning)
   * 
   * @param message a mensagem.
   */
  protected final void warning(final String message) {
    Server.logWarningMessage(message);
  }

  /**
   * Cria um monitorador remoto de servidor central que ser usado pelo servidor
   * local.
   * 
   * @param serverURI A URI do servidor a ser monitorado.
   * @param locator A classe com o localizador de servios. Ser a classe
   *        csbase.remote.ServerRemoteLocator ou um de seus descendentes.
   */
  public LocalServerRemoteMonitor(final ServerURI serverURI,
    final Class<?> locator) {
    this(serverURI, locator, MonitoredServer.RELOAD_TIME);
    instance = this;
  }

  /**
   * Cria um monitorador remoto de servidor central que ser usado pelo servidor
   * local.
   * 
   * @param serverURI A URI do servidor a ser monitorado.
   * @param locator A classe com o localizador de servios. Ser a classe
   *        csbase.remote.ServerRemoteLocator ou um de seus descendentes.
   * @param windowSize Valor mximo da janela de backoff.
   */
  public LocalServerRemoteMonitor(final ServerURI serverURI,
    final Class<?> locator, int windowSize) {
    super(serverURI, locator, windowSize);
    instance = this;
  }
}
