/*
 * ServiceLogFormatter.java
 *
 * $Author$ $Revision$ - $Date: 2011-02-03 17:57:31 -0200
 * (Thu, 03 Feb 2011) $
 */
package csbase.server;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.logging.Formatter;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import csbase.server.services.loginservice.LoginService;

/**
 * Formato o registro de log no formato padro do CSBase.
 */
public class ServerLogFormatter extends Formatter {

  /** Separador de linha */
  private final String LINE_SEPARATOR = "\n";

  /**
   * Formata o <code>LogRecord</code> dado.
   *
   * @param record registro de log a ser formatado.
   *
   * @return registro de log formatado.
   */
  @Override
  public synchronized String format(LogRecord record) {
    StringBuffer sb = new StringBuffer();
    SimpleDateFormat dateHourFormat = new SimpleDateFormat(
      "dd/MM/yyyy HH:mm:ss");
    dateHourFormat.setLenient(false);
    String date = dateHourFormat.format(record.getMillis());
    sb.append(record.getLevel().getLocalizedName());
    sb.append(": ");
    sb.append(date);
    sb.append(" ");
    if (record instanceof ServerLogRecord) {
      ServerLogRecord serverRecord = (ServerLogRecord) record;
      if (record.getLevel().equals(Level.SEVERE)) {
        sb.append(formatStackTraceElement(serverRecord.getMethod()));
      }
      else {
        sb.append(record.getLoggerName());
      }
      sb.append(" --- ");
      if (serverRecord.getUser() != null) {
        sb.append("(");
        sb.append(serverRecord.getUser().getId() + LoginService.getInstance()
          .getRealUserForLog(Service.getKey()));
        sb.append(") ");
      }
    }
    else {
      sb.append(record.getLoggerName());
      sb.append(" --- ");
    }
    String message = formatMessage(record);
    sb.append(message);
    sb.append(LINE_SEPARATOR);
    if (record.getThrown() != null) {
      sb.append(formatThrowable(record.getThrown()));
    }
    return sb.toString();
  }

  /**
   * Formata uma linha de execuo de pilha.
   *
   * @param m A indentificao do elemento de pilha.
   * @return A representao textual do elemento de pilha.
   */
  private Object formatStackTraceElement(StackTraceElement m) {
    return m.getClassName() + "." + m.getMethodName() + "(" + m.getFileName()
      + ":" + m.getLineNumber() + ")";
  }

  /**
   * Formata uma exceo.
   *
   * @param t a exceo
   * @return o texto que descreve a exceo
   */
  private String formatThrowable(Throwable t) {
    StringWriter sw = new StringWriter();
    PrintWriter pw = new PrintWriter(sw);
    pw.format("Exceo: %s - %s%n", t.getClass().getName(), t.getMessage());
    printStack(t, pw);
    Throwable cause = t.getCause();
    while (cause != null) {
      printStack(cause, pw);
      pw.format("Causada por: %s - %s%n", cause.getClass().getName(), cause
        .getMessage());
      printStack(cause, pw);
      cause = cause.getCause();
    }
    pw.close();
    return sw.toString();
  }

  /**
   * Imprime a pilha correspondente.
   *
   * @param t Exceo que determina a pilha.
   * @param pw PrintWriter onde a pilha ser impressa.
   */
  private void printStack(Throwable t, PrintWriter pw) {
    StackTraceElement[] stack = t.getStackTrace();
    for (int i = 0; i < stack.length; i++) {
      pw.print("  em ");
      pw.println(formatStackTraceElement(stack[i]));
    }
  }
}