/**
 * $Id$
 */
package csbase.server.services.projectservice;

import csbase.logic.SecureKey;

/**
 * Interface com as operaes comuns aos locks de arquivos.
 *
 * @see ProjectService#acquireExclusiveLock(Object, String[])
 * @see ProjectService#releaseLock(Object, String[], Object)
 */
interface FileLockInterface {

  /**
   * Estado do lock.
   *
   * @see FileLockInterface#checkLockStatus(SecureKey)
   * @see FileLockInterface#checkLockStatus(String)
   */
  enum LockStatus {
    /**
     * Bloqueado pelo prprio usurio.
     */
    LOCKED_BY_USER,
    /**
     * Bloqueado por outro(s) usurio(s).
     */
    LOCKED_BY_OTHERS,
    /**
     * Desbloqueado.
     */
    UNLOCKED,
    /**
     * Expirado.
     */
    EXPIRED,
  }

  /**
   * Adiciona um lock associado a uma sesso.
   *
   * @param sessionKey sesso ao qual o lock ser associado. Permite
   *        <code>null</null> quando no h sesso associada ao lock.
   * @param lockId identificador do lock
   * @return flag indicando se o lock foi criado
   */
  boolean newLocker(SecureKey sessionKey, Object lockId);

  /**
   * Remove um lock associado a um identificador.
   *
   * @param lockId identificador do lock
   * @return flag indicando se o lock foi removido
   */
  boolean removeLocker(Object lockId);

  /**
   * Indica se existe um lock associado a uma sesso, e se o lock est associado
   *  sesso especificada.
   *
   * @param sessionKey identificador da sesso do usurio
   * @return situao do lock
   * @see LockStatus
   */
  LockStatus checkLockStatus(SecureKey sessionKey);

  /**
   * Indica se existe um lock associado a uma sesso identificada por um hash.
   *
   * @param sessionKey identificador da sesso do usurio
   * @return situao do lock
   * @see LockStatus
   */
  LockStatus checkLockStatus(String sessionKey);

  /**
   * Retorna a contagem total de referncias para o lock.
   *
   * @return contagem total de referncias para o lock
   */
  int getLockRefCount();

  /**
   * Retorna a contagem de referncias ao lock atribudas a uma determinada
   * sesso.
   *
   * @param sessionKey sesso ao qual o lock ser associado. Permite
   *        <code>null</null> quando no h sesso associada ao lock.
   * @return contagem de referncias ao lock
   */
  int getLockRefCount(SecureKey sessionKey);

  /**
   * Indica se todas as referncias ao lock expiraram.
   *
   * @return flag indicando se todas as referncias ao lock expiraram
   */
  boolean hasExpired();

  /**
   * Indica se o lock  compartilhado.
   *
   * @return flag indicando se o lock  compartilhado
   */
  boolean isShared();

}
