/*
 * $Id$
 */
package csbase.server.services.projectservice;

import java.io.Serializable;

/**
 * Representa as informaes sobre um arquivo atualizvel.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class UpdatableFileInfo implements Serializable {
  /**
   * O nome da classe do servio responsvel pela atualizao do arquivo.
   */
  private String serviceClassName;
  /**
   * Dados necessrios  atualizao do arquivo.
   * Estes dados sero, tipicamente, a representao de um identificador do
   * arquivo de origem e devem ser utilizados pelo servio que ir atualizar o
   * arquivo.
   */
  private Serializable userData;

  /**
   * Dados extras e opcionais para realizar a atualizao do arquivo.
   */
  private Serializable extraUserData;

  /**
   * Cria um conjunto de informaes de um arquivo atualizvel.
   * 
   * @param updaterClass A classe do responsvel pela atualizao do arquivo.
   * @param userData Dados necessrios  atualizao do arquivo.
   */
  public UpdatableFileInfo(Class<? extends ProjectFileUpdater> updaterClass,
    Serializable userData) {
    this.serviceClassName = updaterClass.getName();
    this.userData = userData;
  }

  /**
   * Obtm a classe do responsvel pela atualizao do arquivo.
   * 
   * @return A classe do responsvel pela atualizao do arquivo.
   * 
   * @throws ClassNotFoundException Caso a classe no seja encontrada.
   */
  public Class<? extends ProjectFileUpdater> getServiceClass()
    throws ClassNotFoundException {
    return (Class<? extends ProjectFileUpdater>) Class
      .forName(this.serviceClassName);
  }

  /**
   * Obtm os dados necessrios  atualizao do arquivo.
   * 
   * @return Dados necessrios  atualizao do arquivo.
   */
  public Serializable getUserData() {
    return this.userData;
  }

  /**
   * Obtm os dados extras para realizar a atualizao do arquivo.
   * 
   * @return Os dados extras.
   */
  public Serializable getExtraUserData() {
    return this.extraUserData;
  }

  /**
   * Define os dados extras para realizar a atualizao do arquivo.
   * 
   * @param extraUserData Os dados extras.
   */
  public void setExtraUserData(Serializable extraUserData) {
    this.extraUserData = extraUserData;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!this.getClass().equals(obj.getClass())) {
      return false;
    }
    UpdatableFileInfo updatableFileInfo = (UpdatableFileInfo) obj;
    return ((this.serviceClassName.equals(updatableFileInfo.serviceClassName)) && (this.userData
      .equals(updatableFileInfo.userData)));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return this.serviceClassName.hashCode() * this.userData.hashCode();
  }
}
