/**
 * $Id$
 */

package csbase.server.services.serverservice;

import java.io.File;
import java.io.IOException;
import java.rmi.RemoteException;
import java.security.cert.Certificate;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import csbase.exception.CSBaseException;
import csbase.exception.OperationFailureException;
import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.logic.ServerURI;
import csbase.logic.User;
import csbase.logic.diagnosticservice.DeploymentInfo;
import csbase.logic.server.ServerInfo;
import csbase.logic.server.ServerInfoAddEvent;
import csbase.logic.server.ServerInfoData;
import csbase.logic.server.ServerInfoEvent;
import csbase.logic.server.ServerInfoModifyEvent;
import csbase.logic.server.ServerInfoRemoveEvent;
import csbase.remote.ServerServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.keystore.CSKeyStore;
import csbase.server.services.loginservice.LoginService;
import csbase.server.services.messageservice.MessageService;
import csbase.util.messages.Message;

/**
 * <p>
 * Representa o servio de servidores.
 * </p>
 * <p>
 * Este servio mantm um repositrio com informaes dos servidores.
 * </p>
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ServerService extends Service implements
  ServerServiceInterface {
  /**
   * Nome da propriedade que contm o nome do arquivo do repositrio de
   * informaes dos servidores.
   */
  private static final String SERVER_REPOSITORY_NAME_PROPERTY =
    "serverRepositoryName";

  /** O caminho do arquivo do repositrio de servidores confivies. */
  private String serverRepository;

  /** O conjunto ordenado (por nome) de informaes dos servidores. */
  private SortedSet<ServerInfo> serverSortedSet;

  /** Um mapa que liga um id ao seu respectivo servidor */
  private SortedMap<Integer, ServerInfo> serverIdSortedMap;

  /** Um mapa que liga um nome ao seu respectivo servidor */
  private Map<String, ServerInfo> serverNameMap;

  /** A instncia nica do servio de servidores. */
  private static ServerService instance;

  /**
   * Nome dos elementos e atributos presentes no XML que server como repositrio
   * das informaes dos servidores.
   */

  private static final String SERVER_ROOT_TAG = "servidores";
  private static final String SERVER_TAG = "servidor";
  private static final String SERVER_ID_TAG = "id";
  private static final String SERVER_NAME_TAG = "nome";
  private static final String SERVER_SUSPENDED_TAG = "suspenso";
  private static final String SERVER_LOCAL_TAG = "local";
  private static final String SERVER_URI_TAG = "uri";

  /**
   * <p>
   * Cria o servio de servidores.
   * </p>
   * 
   * @throws ServerException Caso ocorra algum erro durante a criao do
   *         servio.
   */
  private ServerService() throws ServerException {
    super(SERVICE_NAME);
    this.createCache();
    this.readProperties();
    if (!this.createServerInfoRepository()) {
      this.reloadCache(loadServerInfoRepository(this.serverRepository));
      try {
        this.updateServersInfos();
      }
      catch (OperationFailureException e) {
        Server.logSevereMessage(
          "Erro ao carregar o repositrio de chaves/certificados.", e);
      }
    }
    this.calculateNextId();
  }

  /**
   * Cria as estruturas de dados que sero utilizados como cache das informaes
   * dos servidores.
   */
  private void createCache() {
    this.serverSortedSet = new TreeSet<ServerInfo>();
    this.serverIdSortedMap = new TreeMap<Integer, ServerInfo>();
    this.serverNameMap = new HashMap<String, ServerInfo>();
  }

  /**
   * L as propriedades do servio.
   */
  private void readProperties() {
    final Server server = Server.getInstance();
    final String dirPath = server.getPersistencyRootDirectoryName();
    this.serverRepository =
      dirPath + File.separator
        + this.getStringProperty(SERVER_REPOSITORY_NAME_PROPERTY);
  }

  /**
   * Gera o prximo identificador vlido para um servidor. Esse identificador 
   * utilizado na criao de um novo servidor.
   * 
   * @return o identificador do servidor
   */
  private int calculateNextId() {
    if (this.serverIdSortedMap.isEmpty()) {
      return 1;
    }
    return (this.serverIdSortedMap.lastKey() + 1);
  }

  /**
   * <p>
   * Cria o repositrio de informaes de servidores, caso ainda no exista.
   * </p>
   * <p>
   * A coleo de informaes de servidores  gravada no arquivo e, por isso,
   * deve ser criada antes da chamada deste mtodo.
   * </p>
   * 
   * @return .
   * 
   * @throws ServerException Caso ocorra algum problema ao criar o repositrio.
   * @throws IllegalStateException Caso a coleo de informaes de servidores
   *         ainda no tenha sido criada.
   */
  private boolean createServerInfoRepository() throws ServerException {
    if (this.serverSortedSet == null) {
      throw new IllegalStateException(
        "Tentativa de criar o arquivo do repositrio de informaes de servidores antes de criar a coleo de informaes servidores.");
    }
    File localServersFile = new File(this.serverRepository);
    if (localServersFile.exists()) {
      return false;
    }
    try {
      File dir = localServersFile.getParentFile();
      if (dir != null && !dir.exists()) {
        dir.mkdirs();
      }
      localServersFile.createNewFile();
    }
    catch (IOException e) {
      throw new ServerException(
        "Erro ao criar o arquivo com o repositrio de informaes dos servidores. Caminho usado: {0}.",
        new Object[] { localServersFile.getPath() }, e);
    }
    try {
      writeServerRepository(this.serverSortedSet, this.serverRepository);
    }
    catch (CSBaseException e) {
      throw new ServerException(
        "Erro ao gravar as informaes dos servidores no repositrio.", e);
    }
    return true;
  }

  /**
   * Obtm a instncia nica do servio de servidores.
   * 
   * @return A instncia nica.
   */
  public static ServerService getInstance() {
    return instance;
  }

  /**
   * Cria a instncia do servio de servidores.
   * 
   * @throws ServerException Caso ocorra alguma falha na criao do servio.
   */
  public static void createService() throws ServerException {
    instance = new ServerService();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initService() {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void shutdownService() {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return true;
  }

  /**
   * Atualiza o cache de informaes dos servidores. Alteraes no repositrio
   * de chaves/certificados sero refletidas nos {@link ServerInfo}.
   * 
   * @throws OperationFailureException
   */
  private void updateServersInfos() throws OperationFailureException {
    CSKeyStore keyStore = CSKeyStore.getInstance();
    if (keyStore == null) {
      return;
    }
    List<ServerInfo> localServerList = new LinkedList<ServerInfo>();
    List<ServerInfo> localServerOldStateList = new LinkedList<ServerInfo>();
    List<ServerInfo> localServerNewStateList = new LinkedList<ServerInfo>();
    Iterator<ServerInfo> localServerIterator = this.serverSortedSet.iterator();
    while (localServerIterator.hasNext()) {
      ServerInfo localServer = localServerIterator.next();
      Certificate certificate;
      try {
        certificate = this.getCertificate(localServer.getName());
      }
      catch (CSBaseException e) {
        continue;
      }
      ServerInfo newLocalServer;
      if (certificate == null) {
        newLocalServer = new ServerInfo(localServer);
      }
      else {
        newLocalServer = new ServerInfo(localServer, certificate);
      }
      if (localServer.isCompleted() != newLocalServer.isCompleted()) {
        localServerOldStateList.add(localServer);
        localServerNewStateList.add(newLocalServer);
      }
      localServerList.add(newLocalServer);
    }
    writeServerRepository(localServerList, this.serverRepository);
    this.reloadCache(localServerList);
    this.fireWasModifiedServerInfo(localServerOldStateList,
      localServerNewStateList);
  }

  /**
   * Recarrega o cache, limpando-o e adicionando os elementos da coleo
   * recebida.
   * 
   * @param serverInfoCollection A coleo com os elementos que faro parte do
   *        cache.
   */
  private void reloadCache(Collection<ServerInfo> serverInfoCollection) {
    this.serverSortedSet.clear();
    this.serverIdSortedMap.clear();
    this.serverNameMap.clear();
    Iterator<ServerInfo> localServerIterator = serverInfoCollection.iterator();
    while (localServerIterator.hasNext()) {
      ServerInfo localServer = localServerIterator.next();
      this.serverSortedSet.add(localServer);
      this.serverIdSortedMap.put((Integer) localServer.getId(), localServer);
      this.serverNameMap.put(localServer.getName(), localServer);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SortedSet<ServerInfo> getServersInfos() {
    try {
      this.updateServersInfos();
    }
    catch (CSBaseException e) {
      throw new ServiceFailureException(ServerService.getInstance().getString(
        "SERVER_SERVICE_LOAD_REPOSITORIES_ERROR"), e);
    }
    return Collections.unmodifiableSortedSet(this.serverSortedSet);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean addServerInfo(ServerInfoData serverInfoData) {
    if (this.hasServerInfo(serverInfoData.getName())) {
      return false;
    }
    ServerInfo newLocalServer;
    try {
      Certificate certificate = this.getCertificate(serverInfoData.getName());
      int localServerId = this.calculateNextId();
      if (this.serverIdSortedMap.get(localServerId) != null) {
        throw new IllegalStateException(
          MessageFormat
            .format(
              "Foi gerado um identificador repetido para as informaes do servidor. Identificador: {0}.",
              localServerId));
      }
      if (certificate == null) {
        newLocalServer = new ServerInfo(localServerId, serverInfoData);
      }
      else {
        newLocalServer =
          new ServerInfo(localServerId, serverInfoData, certificate);
      }
      SortedSet<ServerInfo> localServerSortedSetTemp =
        new TreeSet<ServerInfo>(this.serverSortedSet);
      localServerSortedSetTemp.add(newLocalServer);
      writeServerRepository(localServerSortedSetTemp, this.serverRepository);
    }
    catch (CSBaseException e) {
      throw new ServiceFailureException(ServerService.getInstance().getString(
        "SERVER_SERVICE_ADD_SERVER_ERROR"), e);
    }
    this.addToCache(newLocalServer);
    this.fireWasAddedServerInfo(newLocalServer);
    return true;
  }

  /**
   * Obtm o certificado de um servidor.
   * 
   * @param serverName Nome do servidor do qual se deseja o certificado.
   * 
   * @return O certificado ou null, caso no exista certificado ou se o
   *         repositrio de chaves/certificados ainda no existir.
   * 
   * @throws OperationFailureException Caso ocorra alguma falha na operao.
   */
  private Certificate getCertificate(String serverName)
    throws OperationFailureException {
    CSKeyStore keyStore = CSKeyStore.getInstance();
    if (keyStore == null) {
      return null;
    }
    return keyStore.getCertificate(serverName);
  }

  /**
   * Verifica se existem informaes de algum servidor no repositrio com o
   * mesmo nome do servidor recebido.
   * 
   * @param serverInfo As informaes do servidor a ser procurado.
   * 
   * @return true, caso existam informaes de algum servidor no repositrio com
   *         o mesmo nome, ou false, caso contrrio.
   */
  private boolean hasServerInfo(ServerInfo serverInfo) {
    ServerInfo matchedLocalServer =
      this.serverNameMap.get(serverInfo.getName());
    if (matchedLocalServer == null) {
      return false;
    }
    if (matchedLocalServer.getId().equals(serverInfo.getId())) {
      return false;
    }
    return true;
  }

  /**
   * Verifica se existem, no repositrio, informaes de algum servidor com o
   * nome dado.
   * 
   * @param serverName O nome do servidor
   * 
   * @return true, caso j existam informaes de um servidor com o nome dado,
   *         ou false.
   */
  private boolean hasServerInfo(String serverName) {
    ServerInfo matchedLocalServer = this.serverNameMap.get(serverName);
    if (matchedLocalServer == null) {
      return false;
    }
    return true;
  }

  /**
   * Adiciona informaes do servidor ao cache.
   * 
   * @param serverInfo As informaes a serem adicionado.
   */
  private void addToCache(ServerInfo serverInfo) {
    this.serverSortedSet.add(serverInfo);
    this.serverIdSortedMap.put((Integer) serverInfo.getId(), serverInfo);
    this.serverNameMap.put(serverInfo.getName(), serverInfo);
  }

  /**
   * Envia aos observadores uma notificao de que foram adicionadas informaes
   * de um servidor.
   * 
   * @param serverInfo Informaes adicionadas.
   */
  private void fireWasAddedServerInfo(ServerInfo serverInfo) {
    sendEvent(new ServerInfoAddEvent(serverInfo));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeServerInfo(ServerInfo serverInfo) {
    if (this.serverIdSortedMap.get(serverInfo.getId()) == null) {
      throw new IllegalStateException(
        MessageFormat
          .format(
            "No  possvel remover as informaes de um servidor que no existe. Identificador {0}.",
            new Object[] { serverInfo.getId() }));
    }

    CSKeyStore keyStore = CSKeyStore.getInstance();
    if (keyStore != null) {
      keyStore.removeCertificate(serverInfo.getName());
    }

    SortedSet<ServerInfo> localServerSortedSetTemp =
      new TreeSet<ServerInfo>(this.serverSortedSet);
    localServerSortedSetTemp.remove(serverInfo);
    try {
      writeServerRepository(localServerSortedSetTemp, this.serverRepository);
    }
    catch (CSBaseException e) {
      throw new ServiceFailureException(ServerService.getInstance().getString(
        "SERVER_SERVICE_REMOVE_SERVER_ERROR"), e);
    }
    this.removeFromCache(serverInfo);
    this.fireWasRemovedServerInfo(serverInfo);
  }

  /**
   * Remove as informaes do servidor da cache.
   * 
   * @param serverInfo Informaes a serem removidas.
   */
  private void removeFromCache(ServerInfo serverInfo) {
    this.serverSortedSet.remove(serverInfo);
    this.serverIdSortedMap.remove(serverInfo.getId());
    this.serverNameMap.remove(serverInfo.getName());
  }

  /**
   * Envia aos observadores uma notificao de que foram removidas as
   * informaes de um servidor.
   * 
   * @param serverInfo Informaes removidas.
   */
  private void fireWasRemovedServerInfo(ServerInfo serverInfo) {
    sendEvent(new ServerInfoRemoveEvent(this, serverInfo));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean modifyServerInfo(ServerInfo serverInfo,
    ServerInfoData serverInfoData) {
    if (this.serverIdSortedMap.get(serverInfo.getId()) == null) {
      throw new IllegalStateException(
        MessageFormat
          .format(
            "No  possvel modificar as informaes de um servidor que no existe. Identificador {0}.",
            new Object[] { serverInfo.getId() }));
    }
    ServerInfo newLocalServer;
    try {
      Certificate certificate = this.getCertificate(serverInfoData.getName());
      if (certificate == null) {
        newLocalServer =
          new ServerInfo((Integer) serverInfo.getId(), serverInfoData);
      }
      else {
        newLocalServer =
          new ServerInfo((Integer) serverInfo.getId(), serverInfoData,
            certificate);
      }
      if (this.hasServerInfo(newLocalServer)) {
        return false;
      }
      SortedSet<ServerInfo> localServerSortedSetTemp =
        new TreeSet<ServerInfo>(this.serverSortedSet);
      localServerSortedSetTemp.remove(serverInfo);
      localServerSortedSetTemp.add(newLocalServer);
      writeServerRepository(localServerSortedSetTemp, this.serverRepository);
    }
    catch (OperationFailureException e) {
      throw new ServiceFailureException(ServerService.getInstance().getString(
        "SERVER_SERVICE_MODIFY_SERVER_ERROR"), e);
    }
    this.updateCache(serverInfo, newLocalServer);
    this.fireWasModifiedServerInfo(serverInfo, newLocalServer);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean addCert(String serverName, Certificate c) {
    CSKeyStore keyStore = CSKeyStore.getInstance();
    if (keyStore == null) {
      throw new ServiceFailureException(ServerService.getInstance().getString(
        "SERVER_SERVICE_KEYSTORE_NOT_FOUND"));
    }
    return keyStore.addCertificate(serverName, c);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeCert(String serverName) {
    CSKeyStore keyStore = CSKeyStore.getInstance();
    if (keyStore == null) {
      throw new ServiceFailureException(ServerService.getInstance().getString(
        "SERVER_SERVICE_KEYSTORE_NOT_FOUND"));
    }
    keyStore.removeCertificate(serverName);
  }

  /**
   * Atualiza as informaes de um servidor no cache.
   * 
   * @param oldServerInfo Informaes do servidor em seu estado antigo.
   * @param currentServerInfo Informaes do servidor em seu estado atual.
   */
  private void updateCache(ServerInfo oldServerInfo,
    ServerInfo currentServerInfo) {
    this.serverSortedSet.remove(oldServerInfo);
    this.serverSortedSet.add(currentServerInfo);
    this.serverIdSortedMap.put((Integer) currentServerInfo.getId(),
      currentServerInfo);
    this.serverNameMap.put(currentServerInfo.getName(), currentServerInfo);
  }

  /**
   * Envia aos observadores uma notificao de que um servidor foi alterado.
   * 
   * @param oldServerInfo Informaes do servidor em seu estado antigo.
   * @param currentServerInfo Informaes do servidor em seu estado atual.
   */
  private void fireWasModifiedServerInfo(ServerInfo oldServerInfo,
    ServerInfo currentServerInfo) {
    sendEvent(new ServerInfoModifyEvent(this, oldServerInfo, currentServerInfo));
  }

  /**
   * <p>
   * Envia aos observadores uma notificao de que as informaes de um ou mais
   * servidores foram alteradas.
   * </p>
   * <p>
   * OBS: a posio das informaes dos dos servidores nas listas tem que ser a
   * mesma. As informaes de um servidor que est na posio 1 da lista de
   * estados antigos, deve estar na posio 1 da lista de estados novos.
   * </p>
   * 
   * @param serverInfoOldStateList Uma lista com as informaes dos servidores
   *        locais que foram alterados, em seu estado antigo.
   * @param serverInfoNewStateList Uma lista com as informaes dos servidores
   *        locais que foram alterados, em seu estado atual.
   * 
   * @throws IllegalArgumentException Caso o tamanho das duas colees seja
   *         diferent.
   */
  private void fireWasModifiedServerInfo(
    List<ServerInfo> serverInfoOldStateList,
    List<ServerInfo> serverInfoNewStateList) {
    if (serverInfoOldStateList.size() != serverInfoNewStateList.size()) {
      throw new IllegalArgumentException(
        "A quantidade de servidor locais deve ser igual nas duas listas.");
    }
    Iterator<ServerInfo> localServerOldStateIterator =
      serverInfoOldStateList.iterator();
    Iterator<ServerInfo> localServerNewStateIterator =
      serverInfoNewStateList.iterator();
    while (localServerOldStateIterator.hasNext()) {
      ServerInfo oldLocalServer = localServerOldStateIterator.next();
      ServerInfo newLocalServer = localServerNewStateIterator.next();
      sendEvent(new ServerInfoModifyEvent(this, oldLocalServer, newLocalServer));
    }
  }

  /**
   * Carrega as informaes dos servidores, do repositrio.
   * 
   * @param filePath O caminho para o arquivo do repositrio de servidores.
   * 
   * @return Uma coleo com as informaes carregadas.
   * 
   * @throws ServerException Caso ocorra algum erro ao carregar as informaes.
   */
  private static Collection<ServerInfo> loadServerInfoRepository(String filePath)
    throws ServerException {
    DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    factory.setIgnoringElementContentWhitespace(true);
    DocumentBuilder builder;
    try {
      builder = factory.newDocumentBuilder();
    }
    catch (ParserConfigurationException e) {
      throw new ServerException(
        "Erro ao criar leitor de documentos para a leitura do repositrio de servidores.",
        e);
    }
    Document document;
    try {
      document = builder.parse(new File(filePath));
    }
    catch (SAXException e) {
      throw new ServerException(
        "O arquivo do repositrio de servidores est invlido. Caminho {0}",
        new Object[] { filePath }, e);
    }
    catch (IOException e) {
      throw new ServerException(
        "O arquivo do repositrio de servidores est invlido. Caminho {0}",
        new Object[] { filePath }, e);
    }
    Element rootElement = document.getDocumentElement();
    NodeList childNodeList = rootElement.getChildNodes();
    List<ServerInfo> localServerList = new LinkedList<ServerInfo>();
    for (int i = 0; i < childNodeList.getLength(); i++) {
      Node node = childNodeList.item(i);
      if (node.getNodeType() == Node.ELEMENT_NODE) {
        Element localServerElement = (Element) node;
        Integer localServerId =
          new Integer(localServerElement.getAttribute(SERVER_ID_TAG));
        String localServerName =
          localServerElement.getAttribute(SERVER_NAME_TAG);
        Boolean isComplete =
          new Boolean(localServerElement.getAttribute(SERVER_SUSPENDED_TAG));
        Boolean isLocal =
          new Boolean(localServerElement.getAttribute(SERVER_LOCAL_TAG));
        String uriStr = localServerElement.getAttribute(SERVER_URI_TAG);
        localServerList.add(new ServerInfo(localServerId, localServerName,
          isComplete.booleanValue(), isLocal.booleanValue(), ServerURI
            .parse(uriStr)));
      }
    }
    return localServerList;
  }

  /**
   * Grava o repositrio de servidores.
   * 
   * @param serverInfoCollection Uma coleo com as informaes dos servidores
   *        que compem o repositrio.
   * @param filePath O caminho para o arquivo onde ser gravado o repositrio.
   * 
   * @throws OperationFailureException Caso ocorra algum problema ao gravar o
   *         repositrio.
   */
  private static void writeServerRepository(
    Collection<ServerInfo> serverInfoCollection, String filePath)
    throws OperationFailureException {
    final String INDENT_OUTPUT_PROPERTY = "indent";
    final String YES_VALUE = "yes";
    DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    DocumentBuilder builder;
    try {
      builder = factory.newDocumentBuilder();
    }
    catch (ParserConfigurationException e) {
      throw new OperationFailureException(
        "Erro ao criar gerador de documentos para a gravao do repositrio de servidores locais.",
        e);
    }
    Document document = builder.newDocument();
    Element rootElement = document.createElement(SERVER_ROOT_TAG);
    Iterator<ServerInfo> serversInfosIterator = serverInfoCollection.iterator();
    while (serversInfosIterator.hasNext()) {
      ServerInfo serverInfo = serversInfosIterator.next();
      Element serverElement = document.createElement(SERVER_TAG);
      Attr idAttribute = document.createAttribute(SERVER_ID_TAG);
      idAttribute.setValue(serverInfo.getId().toString());
      serverElement.setAttributeNode(idAttribute);
      Attr nameAttribute = document.createAttribute(SERVER_NAME_TAG);
      nameAttribute.setValue(serverInfo.getName());
      serverElement.setAttributeNode(nameAttribute);
      Attr stateAttribute = document.createAttribute(SERVER_SUSPENDED_TAG);
      stateAttribute.setValue(String.valueOf(serverInfo.isSuspended()));
      serverElement.setAttributeNode(stateAttribute);
      Attr localAttribute = document.createAttribute(SERVER_LOCAL_TAG);
      localAttribute.setValue(String.valueOf(serverInfo.isLocal()));
      serverElement.setAttributeNode(localAttribute);
      Attr uriAttribute = document.createAttribute(SERVER_URI_TAG);
      uriAttribute.setValue(serverInfo.getURI().toString());
      serverElement.setAttributeNode(uriAttribute);
      rootElement.insertBefore(serverElement, null);
    }
    document.appendChild(rootElement);
    try {
      TransformerFactory transformerFactory = TransformerFactory.newInstance();
      Transformer transformer = transformerFactory.newTransformer();
      Source input = new DOMSource(document);
      Result output = new StreamResult(new File(filePath));
      transformer.setOutputProperty(INDENT_OUTPUT_PROPERTY, YES_VALUE);
      transformer.transform(input, output);
    }
    catch (TransformerException e) {
      throw new OperationFailureException(
        "Erro ao gerar o repositrio de servidores.", e);
    }
  }

  /**
   * Obtm as informaes de um servidor a partir do seu nome.
   * 
   * @param serverName O nome do servidor.
   * 
   * @return O servidor, ou null, caso no exista.
   * 
   * @throws OperationFailureException Caso a operao falhe.
   */
  public ServerInfo getServerInfo(String serverName)
    throws OperationFailureException {
    this.updateServersInfos();
    return this.serverNameMap.get(serverName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String listRuntimeProperties() {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
    return Server.getInstance().listRuntimeProperties();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, String> getRuntimeProperties() throws RemoteException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
    return Server.getInstance().getRuntimeProperties();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DeploymentInfo getDeploymentInfo() throws RemoteException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
    return DeploymentInfo.getInstance();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getNumRegisteredUsers() throws RemoteException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
    return User.getNumRegisteredUsers();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getStartupTime() {
    return Server.getInstance().getStartupTime();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Integer> getLoginStats(boolean succeeded)
    throws RemoteException {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
    return LoginService.getInstance().getLoginStats(succeeded);
  }

  private void sendEvent(ServerInfoEvent event) {
    MessageService.getInstance().sendToAll(new Message(event));
  }
}
