/*
 * $Id$
 */
package csbase.server.services.wioservice;

import java.io.File;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;

import org.omg.PortableServer.POA;

import csbase.logic.User;
import csbase.logic.UserProjectInfo;
import csbase.server.Server;
import csbase.server.services.loginservice.LoginService;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.wioservice.idl.AlgorithmInfo;
import csbase.server.services.wioservice.idl.UserProject;
import csbase.server.services.wioservice.idl.WIOFileSystemPOA;
import csbase.server.services.wioservice.idl.WIOProject;
import csbase.server.services.wioservice.idl.WIOProjectHelper;
import csbase.server.services.wioservice.idl.WIOServiceException;

/**
 * A classe <code>WIOServerFileSystem</code> implementa o <i>servant</i>
 * associado  interface idl <code>WIOFileSystem</code>. As requisies
 * recebidas dos cliente so repassadas para  implementao do servio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class WIOServerFileSystem extends WIOFileSystemPOA {
  /**
   * Separador de arquivos no servidor
   */
  protected static String FILE_SEPARATOR = "|";

  /**
   * Referncia para a implementao do servio SGAService
   */
  private WIOService service = null;

  /**
   * Nmero de projetos abertos neste file system.
   */
  private int numProjects = 0;

  /**
   * Nmero total de arquivos abertos neste file system.
   */
  private int numTotalFiles = 0;

  /**
   * Comparador de {@link UserProject}.
   */
  private static Comparator<UserProject> COMPARATOR =
    new Comparator<UserProject>() {
      @Override
      public int compare(final UserProject o1, final UserProject o2) {
        final String n1 = o1.projectId;
        final String n2 = o2.projectId;
        return n1.compareToIgnoreCase(n2);
      }
    };

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean checkPassword(final String userId, final String passwd) {
    return isCorrectPassword(userId, passwd);
  }

  /**
   * Check de instrumentao da existncia do servio.
   * 
   * @throws Exception em caso de erro de acesso.
   */
  private void checkService() throws Exception {
    if (service == null) {
      final String err = "Servico de WIO com referencia nula!";
      throw new Exception(err);
    }
  }

  /**
   * Incrementa o nmero de projetos abertos de WIOServerFile.
   * 
   * @throws Exception em caso de erro de acesso.
   */
  protected void decNumFiles() throws Exception {
    checkService();
    numTotalFiles--;
  }

  /**
   * Decrementa o nmero de projetos abertos
   * 
   * @throws Exception se houver underflow de limite permitido.
   */
  protected void decNumProjects() throws Exception {
    checkService();
    final int MAX_NUM_PROJS = service.getMaxOpenedProjects();
    if (numProjects <= 0) {
      final String err = "Underflow em numero de projetos.";
      throw new Exception(err);
    }
    numProjects--;
    Server.logInfoMessage("Decremento do numero de projetos abertos: "
      + numProjects + "/" + MAX_NUM_PROJS);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getMaxOpenedProjects() throws WIOServiceException {
    try {
      checkService();
      return service.getMaxOpenedProjects();
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha na consulta de max-opened", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getMaxReadKb() throws WIOServiceException {
    try {
      checkService();
      return service.getMaxReadKb();
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha na consulta de max-read", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getMaxWriteKb() throws WIOServiceException {
    try {
      checkService();
      return service.getMaxWriteKb();
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha na consulta de max-write", e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * Consulta ao nmero de projetos abertos no servidor.
   * 
   * @return o nmero de projetos abertos.
   */
  int getNumOpenedFiles() {
    return numTotalFiles;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getNumOpenedProjects() {
    return numProjects;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getProjects(final String userLogin, final String password)
    throws WIOServiceException {
    if (!isCorrectPassword(userLogin, password)) {
      String msg = "Falha de autenticacao na busca por projetos. ";
      msg = msg + "Usuario: " + userLogin;
      Server.logSevereMessage(msg);
      throw new WIOServiceException(msg);
    }

    final ProjectService prjService;
    final String[] names;
    try {
      prjService = ProjectService.getInstance();
      ProjectService.setUserId(User.getAdminId());

      // FIXME estamos assumindo que userId == userLogin, o que pode vir a mudar no futuro!
      final List<UserProjectInfo> projects =
        prjService.getProjectsFromUser(userLogin);
      final int prjsSize = projects.size();
      names = new String[prjsSize];
      for (int i = 0; i < prjsSize; i++) {
        names[i] = projects.get(i).getProjectName();
      }
    }
    catch (final Exception e) {
      final String err = "Falha na busca de projetos proprios";
      Server.logSevereMessage(err, e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      ProjectService.setUserId(null);
    }

    Arrays.sort(names, new Comparator<String>() {
      @Override
      public int compare(final String n1, final String n2) {
        return n1.compareToIgnoreCase(n2);
      }
    });
    return names;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public UserProject[] getProjectsFromOthers(final String userLogin,
    final String password) throws WIOServiceException {
    if (!isCorrectPassword(userLogin, password)) {
      String err = "Falha de autenticacao na busca por projetos compartilhados";
      err = err + "\nUsuario: " + userLogin;
      Server.logSevereMessage(err);
      throw new WIOServiceException(err);
    }
    try {
      final ProjectService prjService = ProjectService.getInstance();
      ProjectService.setUserId(User.getAdminId());

      // FIXME estamos assumindo que userId == userLogin, o que pode vir a mudar no futuro!
      final List<UserProjectInfo> userProjects =
        prjService.getProjectsSharedWithUser(userLogin);
      final int np = userProjects.size();
      final UserProject[] ups = new UserProject[np];
      for (int i = 0; i < np; i++) {
        final UserProjectInfo uInfo = userProjects.get(i);
        ups[i] = new UserProject();
        ups[i].ownerId = (String) uInfo.getOwnerId();
        ups[i].ownerName = User.getName(uInfo.getOwnerId());
        ups[i].projectId = uInfo.getProjectName();
      }
      Arrays.sort(ups, COMPARATOR);
      return ups;
    }
    catch (final Exception e) {
      final String err = "Falha na busca de projetos compartilhados";
      Server.logSevereMessage(err);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
    finally {
      ProjectService.setUserId(null);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getSeparatorChar() {
    return FILE_SEPARATOR;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getServerVersion() {
    return VERSION;
  }

  /**
   * Incrementa o nmero de projetos abertos de WIOServerFile.
   * 
   * @throws Exception em caso de erro de acesso.
   */
  protected void incNumFiles() throws Exception {
    checkService();
    numTotalFiles++;
  }

  /**
   * Incrementa o nmero de projetos abertos de WIOServerFile.
   * 
   * @throws Exception se houver estouro de limite permitido.
   */
  protected void incNumProjects() throws Exception {
    checkService();
    final int MAX_NUM_PROJS = service.getMaxOpenedProjects();
    if (numProjects >= MAX_NUM_PROJS) {
      final String err = "Overflow em numero de projetos.";
      throw new Exception(err);
    }
    numProjects++;
    Server.logInfoMessage("Incremento do numero de projetos abertos: "
      + numProjects + "/" + MAX_NUM_PROJS);
  }

  /**
   * Checagem de senha do usurio
   * 
   * @param userLogin identificador do usurio.
   * @param password senha.
   * 
   * @return um flag indicativo da corretude da senha.
   */
  private boolean isCorrectPassword(final String userLogin,
    final String password) {
    try {
      final LoginService loginService = LoginService.getInstance();
      final User user = loginService.checkLogin(userLogin, password);
      if (user == null) {
        return false;
      }
      return true;
    }
    catch (final Exception e) {
      Server.logSevereMessage("Falha na autenticao do WIOService!", e);
      return false;
    }
  }

  /**
   * Abertura real do projeto.
   * 
   * @param userLogin identificador do usuario que quer
   * @param ownerLogin o identificador do dono do arquivo.
   * @param projectName o identificador do projeto.
   * 
   * @return a referncia ao projeto.
   * 
   * @throws Exception em caso de erro de acesso.
   */
  private WIOProject internalOpen(final String userLogin,
    final String ownerLogin, final String projectName) throws Exception {
    final WIOServerProject prj;
    try {
      prj = new WIOServerProject(this, userLogin, ownerLogin, projectName);
    }
    catch (final Exception e) {
      final String err = "Excecao detectada na abertura de projeto";
      Server.logSevereMessage(err, e);
      throw new Exception(err, e);
    }

    Server.logInfoMessage("Instanciado projeto [" + projectName + "]...");
    final POA poa = prj.activateCorbaProject();
    if (poa == null) {
      final String err = "Falha detectada na ativ. de projeto";
      throw new Exception(err);
    }
    final WIOProject cprj =
      WIOProjectHelper.narrow(poa.servant_to_reference(prj));
    if (cprj == null) {
      final String err = "Falha de narrow ao criar CORBA do projeto";
      Server.logSevereMessage(err + ":" + projectName);
      throw new Exception(err);
    }
    if (cprj._non_existent()) {
      final String err = "Falha de non-existent CORBA em projeto";
      Server.logSevereMessage(err + ":" + projectName);
      throw new Exception(err);
    }
    return cprj;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public WIOProject openProject(final String userLogin, final String password,
    final String projectName) throws WIOServiceException {
    Server.logInfoMessage("Pedido de abertura de [" + projectName + "].");
    if (!isCorrectPassword(userLogin, password)) {
      String msg = "Falha de autenticacao na abertura do projeto ";
      msg = msg + "\nProjeto: " + userLogin;
      msg = msg + "\nUsuario: " + userLogin;
      Server.logSevereMessage(msg);
      throw new WIOServiceException(msg);
    }
    try {
      final WIOProject prj = internalOpen(userLogin, userLogin, projectName);
      final String msg = "Sucesso na abertura de [" + projectName + "]";
      Server.logInfoMessage(msg);
      return prj;
    }
    catch (final Exception e) {
      final String err = "Falha na abertura de [" + projectName + "]";
      Server.logSevereMessage(err, e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public WIOProject openProjectFromOthers(final String userLogin,
    final String password, final String ownerLogin, final String projectName)
    throws WIOServiceException {
    Server.logInfoMessage("Pedido de abertura " + "do projeto compartilhado ["
      + projectName + "]...");
    if (!isCorrectPassword(userLogin, password)) {
      String msg = "Falha de autenticacao na abertura do projeto ";
      msg = msg + "\nProjeto: " + userLogin;
      msg = msg + "\nUsurio: " + userLogin;
      msg = msg + "\nDono: " + ownerLogin;
      Server.logSevereMessage(msg);
      throw new WIOServiceException(msg);
    }
    try {
      final WIOProject prj = internalOpen(userLogin, ownerLogin, projectName);
      final String fmt = "Sucesso na abertura de [%s] (shared)";
      final String msg = String.format(fmt, projectName);
      Server.logInfoMessage(msg);
      return prj;
    }
    catch (final Exception e) {
      final String fmt = "Falha na abertura de [%s] (shared)";
      final String err = String.format(fmt, projectName);
      Server.logSevereMessage(err, e);
      throw new WIOServiceException(WIOService.getExceptionString(e));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String preLogin2Web(final String login, final String password,
    final String projectName, final AlgorithmInfo algorithmInfo) {
    return this.service.preLogin2Web(login, password, projectName,
      algorithmInfo);
  }

  /**
   * Constri um servant.
   * 
   * @param srv referncia para a implementao do servio.
   */
  protected WIOServerFileSystem(final WIOService srv) {
    FILE_SEPARATOR = File.separator;
    if (FILE_SEPARATOR.equals("\\")) {
      FILE_SEPARATOR = "\\\\";
    }
    service = srv;
  }
}
