/**
 * $Id: JobExecutor.java 163602 2015-04-14 18:56:26Z fpina $
 */
package csbase.sga.executor;

import java.util.Map;

import sgaidl.ActionNotSupportedException;
import sgaidl.InvalidActionException;
import sgaidl.JobControlAction;

/**
 * O Executor  o responsvel por iniciar a execuo de jobs em um ambiente
 * especfico e por monitorar o estado do job.
 *
 * Atravs de um {@link JobObserver observer} o executor notifica o
 * {@link JobObserver#onCommandCompleted trmino} e a
 * {@link JobObserver#onCommandLost perda} do comando.
 *
 * @author Tecgraf/PUC-Rio
 */
public interface JobExecutor {

  /**
   * Executa um job no ambiente de execuo. Um job representa um comando de
   * execuo, que  um programa ou script junto com os seus argumento, no
   * ambiente de execuo.
   *
   * @param jobCommand comando do job que deve ser executado.
   * @param extraParams parametros extras usados pelo ambiente de execuo para
   *        executar o job
   * @param observer observador do job
   *
   * @return os dados do job
   */
  public JobData executeJob(String jobCommand, Map<String, String> extraParams,
    JobObserver observer);

  /**
   * Recupera um job no ambiente de execuo.
   *
   * Quando o servidor  reiniciado  necessrio obtr o estado dos jobs que
   * estavam em execuo antes da parada no ambiente de execuo.
   *
   * @param data dados do job
   * @param observer observador do job
   *
   * @return true se o job foi recuperado e o observador cadastrado e false se
   *         no foi possvel recuperar o job no ambiente de execuo
   */
  public boolean retrieveJob(JobData data, JobObserver observer);

  /**
   * Exerce uma ao sobre um job ou sobre um processo filho do job.
   *
   * @param data os dados do job
   * @param child o processo filho do job
   * @param action a ao a ser exercida
   *
   * @throws InvalidActionException se a ao  invlida
   * @throws ActionNotSupportedException se a o no  suportada pelo ambiente
   *         de execuo
   */
  public void controlJob(JobData data, String child, JobControlAction action)
    throws InvalidActionException, ActionNotSupportedException;

  /**
   * Obtm as informaes sobre a execuo do job no ambiente de execuo.
   *
   * @param data os dados do job
   *
   * @return as informaes sobre o job
   */
  public JobInfo getJobInfo(JobData data);
}
