/**
 * $Id$
 */

package csbase.tools;

import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.nio.channels.ServerSocketChannel;
import java.rmi.Naming;
import java.rmi.Remote;
import java.rmi.ServerException;
import java.rmi.UnmarshalException;

import csbase.remote.ServerEntryPoint;

/**
 * Cdigos de trmino da aplicao (podem ser testados pelo shell que a
 * executar).
 */
enum ExitCodes {
  /**
   * Erro de I/O no-especfico.
   */
  IO_ERROR(-5),

  /**
   * Porta especificada no arquivo de propriedades no  um nmero inteiro.
   */
  PORT_NOT_INT(-4),

  /**
   * Arquivo de propriedades no define a propriedade do RMI (ver
   * {@link DetectServer#RMI_PROPERTY_NAME})
   */
  PORT_SPEC_MISSING(-3, "Erro: o arquivo de propriedades nao especifica "
    + DetectServer.RMI_PROPERTY_NAME),
  /**
   * Erro ao processar o arquivo de propriedades.
   */
  PROPS_FILE_ERROR(-2),

  /**
   * Arquivo de propriedades no foi informado como parmetro na linha de
   * comando.
   */
  PROPS_FILE_MISSING(-1, "Erro: Arquivo de propriedades nao fornecido."),

  /**
   * A porta est disponvel.
   */
  PORT_AVAILABLE(0),

  /**
   * H um processo no identificado (i.e. no  um servidor CSBase) utilizando
   * a porta.
   */
  UNKNOWN_SERVER(1, "Erro: porta ocupada por aplicacao desconhecida"),

  /**
   * A porta est sendo utilizada por um servidor CSBase.
   */
  CSBASE_SERVER(2);

  /**
   * Cdigo de trmino (usado pelo shell).
   */
  final int exitCode;
  /**
   * Mensagem de erro (opcional).
   */
  String msg;

  /**
   * Construtor.
   * 
   * @param exitCode cdigo de trmino. Define uma mensagem vazia.
   */
  private ExitCodes(int exitCode) {
    this.exitCode = exitCode;
    this.msg = "";
  }

  /**
   * Construtor.
   * 
   * @param exitCode cdigo de trmino
   * @param msg mensagem de erro
   */
  private ExitCodes(int exitCode, String msg) {
    this(exitCode);
    this.msg = msg;
  }
}

/**
 * Este programa tenta detectar se h um servidor CSBase executando. A porta de
 * conexo  obtida do arquivo de propriedades fornecido como nico parmetro. A
 * propriedade que a especifica  a conforme definio em:
 * {@link DetectServer#RMI_PROPERTY_NAME}. O valor retorno  definido pelas
 * constantes da enumerao {@link ExitCodes}.
 * <p>
 * No caso do retorno ser {@link ExitCodes#CSBASE_SERVER},  impresso em stdout
 * a verso do servidor que foi encontrado. Nos demais casos, as eventuais
 * mensagens de erro so enviadas para stderr.
 */
public class DetectServer {

  /**
   * Porta lida do arquivo <code>Server.properties</code>.
   */
  private static int port = 0;

  /**
   * Nome da propriedade a ser verificada para teste com RMI.
   */
  final static String RMI_PROPERTY_NAME = "Server.registryPort";

  /**
   * Mtodo principal. Caso a porta esteja disponvel, apenas retorna
   * {@link ExitCodes#PORT_AVAILABLE}. Seno, retorna o cdigo de erro
   * especfico e (possivelmente) alguma informao adicional em
   * <code>stderr</code>.
   * <p>
   * Exemplo de uso:
   * 
   * <pre>
   * java -cp . csbase.tools.DetectServer marlim/properties/Server.properties
   * </pre>
   * 
   * @param args parmetros da linha de comando. Deve ser fornecido um nico
   *        parmetro: o path (relativo ou absoluto) para o arquivo
   *        <code>Server.properties</code>.
   */
  public static void main(String[] args) {
    loadProperties(args);
    /*
     * as propriedades foram lidas com sucesso, vamos verificar se existe um
     * servidor CSBase usando a porta.
     */
    tryCSBaseServerConnection();
    /*
     * no existia um servidor CSBase, vamos tentar um bind TCP para verificar
     * se ela est realmente disponvel.
     */
    tryTCPConnection();
    /*
     * nem CSBase, nem TCP: a porta est disponvel
     */
    exit(ExitCodes.PORT_AVAILABLE);
  }

  /**
   * Carrega o arquivo <code>Server.properties</code>.
   * 
   * @param args parmetros fornecidos na linha de comando
   */
  private static void loadProperties(String[] args) {
    if (args.length < 1) {
      terminate(ExitCodes.PROPS_FILE_MISSING);
    }
    try {
      Reader r = new FileReader(args[0]);
      System.getProperties().load(r);
      r.close();
      String sPort = System.getProperty(RMI_PROPERTY_NAME);
      if (sPort == null) {
        terminate(ExitCodes.PORT_SPEC_MISSING);
      }
      Integer iPort = Integer.getInteger(RMI_PROPERTY_NAME);
      if (iPort == null) {
        terminate(ExitCodes.PORT_NOT_INT);
      }
      port = iPort;
    }
    catch (Exception e) {
      terminate(ExitCodes.PROPS_FILE_ERROR,
        "Erro no acesso ao arquivo de propriedades: " + e.getMessage());
    }
  }

  /**
   * Termina a aplicao com um cdigo especfico e a respectiva mensagem.
   * 
   * @param exitCode cdigo de trmino
   */
  private static void terminate(ExitCodes exitCode) {
    terminate(exitCode, exitCode.msg);
  }

  /**
   * Termina a aplicao com um cdigo e uma mensagem especficos.
   * 
   * @param exitCode cdigo de trmino
   * @param msg mensagem de erro
   */
  private static void terminate(ExitCodes exitCode, String msg) {
    System.err.println(msg);
    exit(exitCode);
  }

  /**
   * Termina a aplicao com um cdigo especfico.
   * 
   * @param code cdigo de trmino
   */
  private static void exit(ExitCodes code) {
    System.exit(code.exitCode);
  }

  /**
   * Tenta uma conexo com um servidor CSBase na porta especificada. Aborta a
   * execuo caso a porta no esteja disponvel.
   */
  private static void tryCSBaseServerConnection() {
    String url = "rmi://localhost:" + port;
    String serverURL = url + "/" + ServerEntryPoint.LOOKUP;
    try {
      Naming.list(url);
    }
    catch (Exception e) {
      return;
    }
    Remote server = null;
    try {
      server = Naming.lookup(serverURL);
    }
    catch (Throwable e) {
      terminate(ExitCodes.UNKNOWN_SERVER, "Erro no lookup RMI do servidor: "
        + e.getClass().getName() + '\n' + e.getMessage());
    }
    /*
     * se chegamos at aqui temos um stub para um servidor rodando na porta
     * especificada
     */
    try {
      final ServerEntryPoint serverEntryPoint = (ServerEntryPoint) server;
      String versionName = serverEntryPoint.getVersionName();
      String msg =
        String.format("Servidor %s rodando na porta %d",
          versionName.isEmpty() ? "[???]" : versionName, port);
      terminate(ExitCodes.CSBASE_SERVER, msg);
    }
    catch (ServerException e) {
      Throwable cause = e.getCause();
      if (cause instanceof UnmarshalException) {
        terminate(ExitCodes.UNKNOWN_SERVER, "Porta " + port
          + " ocupada (provavelmente por uma versao diferente do servidor)");
      }
      else {
        terminate(ExitCodes.UNKNOWN_SERVER, "Erro no acesso ao servidor:\n"
          + cause.getMessage());
      }
    }
    catch (Exception e) {
      terminate(ExitCodes.UNKNOWN_SERVER,
        "Erro no acesso ao servidor:\n" + e.getMessage());
    }
  }

  /**
   * Tenta um bind TCP para "localhost" na porta especificada pelo
   * <code>Server.properties</code>. Aborta a execuo caso a porta no esteja
   * disponvel.
   * 
   * @see #tryTCPConnection(String, int, boolean)
   */
  private static void tryTCPConnection() {
    ExitCodes result = tryTCPConnection("localhost", port, true);
    if (result != ExitCodes.PORT_AVAILABLE) {
      terminate(result);
    }
  }

  /**
   * Tenta um bind TCP.
   * 
   * @param serverName nome do servidor
   * @param serverPort porta do servidor
   * @param verbose <code>true</code> se mensagens de erro devem ser exibidas em
   *        <code>stderr</code>
   * @return {@link ExitCodes#PORT_AVAILABLE} se a porta est disponvel,
   *         {@link ExitCodes#IO_ERROR} se no foi possvel abrir ou fechar o
   *         <i>socket channel</i> para verificar a porta,
   *         {@link ExitCodes#UNKNOWN_SERVER} se a porta est ocupada
   */
  public static ExitCodes tryTCPConnection(String serverName, int serverPort,
    boolean verbose) {
    InetSocketAddress addr = new InetSocketAddress(serverName, serverPort);
    ServerSocketChannel server = null;
    try {
      server = ServerSocketChannel.open();
    }
    catch (IOException e) {
      if (verbose) {
        System.err.println(e.getMessage());
      }
      return ExitCodes.IO_ERROR;
    }
    ServerSocket socket = server.socket();
    try {
      socket.bind(addr);
    }
    catch (IOException e) {
      return ExitCodes.UNKNOWN_SERVER;
    }
    finally {
      try {
        socket.close();
      }
      catch (IOException e) {
        if (verbose) {
          System.err.println("Erro ao fechar o socket:\n" + e.getMessage());
        }
        return ExitCodes.IO_ERROR;
      }
    }
    return ExitCodes.PORT_AVAILABLE;
  }
}
