package csbase.util.rmi;

import java.rmi.NoSuchObjectException;
import java.rmi.Remote;
import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;


/**
 * <p>
 * Representa uma referncia a uma porta.<br>
 * Manter uma referncia a uma porta impede que ela seja utilizada por
 * terceiros.
 * </p>
 * <p>
 * Esse referncia tambm permite que sejam exportados objetos por aquela porta
 * e tambm verificar se um endpoint consegue acessar aquela porta e
 * consequentemente chamar os mtodos dos objetos exportados por ela.
 * </p>
 * 
 * @author Tecgraf
 */
public class PortReference {

  /**
   * Utilizado para "<i>logar</i>" informaes no sistema.
   */
  private static final Logger LOGGER =
    Logger.getLogger(PortReference.class.getName());

  /**
   * Fbrica de socket utilizada na exportao dos objetos.<br>
   * Essa fbrica permite obter a porta no qual o pingable foi exportado, mesmo
   * que seja passado o valor 0 para a exportao.
   */
  private SocketFactory socketFactory;
  /**
   * Este objeto tem duas funes:<br>
   * <ol>
   * <li>Manter a reserva sobre uma determinada porta,</li>
   * <li>Testar se um endpoint consegue acessar essa porta.</li>
   * </lo> <br>
   * Essa ltima  importante para saber se o endpoint conseguir chamar os
   * mtodos dos objetos exportados.
   */
  private Pingable pingable;
  /**
   * Porta sendo reservada. Se no h reserva, o valor  -1.
   */
  private int port;

  /**
   * Construtor.
   */
  public PortReference() {
    this.socketFactory = new SocketFactory();
    this.pingable = new Pingable();
    this.port = -1;
  }

  /**
   * Vincula a referncia com uma determinada porta, impedindo que terceiros
   * utilizem aquela porta. Se j ouver vnculo com alguma porta, o vnculo
   * anterior ser desfeito.
   * 
   * @param port Porta na qual a referncia ser vinculada.<br>
   *        Se o valor passado for 0, o sistema ir escolher a porta no qual
   *        essa instncia ir se vincular.
   * 
   * @return <tt>True</tt> se foi possvel se vincular a porta dada, ou
   *         <tt>false</tt> caso ela j estivesse em uso.
   */
  public boolean bind(int port) {
    if (this.port > 0) {
      unbind(); // faz unexport do pingable e faz com que this.port = -1      
    }
    // A partir deste ponto, this.port == -1.

    if (port < 0) {
      return false;
    }

    try {
      UnicastRemoteObject.exportObject(pingable, port, socketFactory,
        socketFactory);
      /*
       * Obtm a partir fbrica de socket a porta por onde o pingable foi
       * exportado. Isso  importante pois pode ter sido passada a porta 0 no
       * construtor, indicando que a porta deve ser escolhida pelo sistema.
       */
      this.port = socketFactory.getLocalPort();
      return true;
    }
    catch (RemoteException e) {
      LogRecord record =
        new LogRecord(Level.WARNING, "Error binding with port " + this.port);
      record.setThrown(e);
      LOGGER.log(record);
    }

    return false;
  }

  /**
   * Caso exista um vinculo com uma porta, desfaz ele.
   */
  public void unbind() {
    if (port > 0) {
      try {
        UnicastRemoteObject.unexportObject(pingable, true);
      }
      catch (RemoteException e) {
        LogRecord record =
          new LogRecord(Level.INFO, "Error unbinding port " + port);
        record.setThrown(e);
        LOGGER.log(record);
      }
      port = -1;
    }
  }

  /**
   * Verifica se esta instncia est vinculada a uma porta.
   * 
   * @return <tt>True</tt> se est instncia est vinculada a uma porta.
   */
  public boolean isBound() {
    return port > 0;
  }

  /**
   * Verifica se um endpoint conseguiria acessar os objetos exportados por essa
   * porta.
   * 
   * @param endpoint Destino dos objetos exportados.
   * 
   * @return <tt>True</tt> se um endpoint conseguiria acessar os objetos
   *         exportados por essa porta.
   */
  public boolean isReachableBy(IPingable endpoint) {
    try {
      return endpoint.canReach(pingable);
    }
    catch (RemoteException e) {
      LogRecord record =
        new LogRecord(Level.WARNING,
          "Error unreachable by endpoint through port " + this.port);
      record.setThrown(e);
      LOGGER.log(record);
    }
    return false;
  }

  /**
   * Exporta um objeto na porta referenciada.
   * 
   * @param obj Objeto a ser exportado.
   * 
   * @return O stub do objeto.
   * 
   * @throws RemoteException Se ocorrer alguma falha durante a exportao.
   * @throws IllegalStateException Se no houver uma porta vinculada a essa
   *         instncia.
   * 
   * @see UnicastRemoteObject#exportObject(Remote, int,
   *      java.rmi.server.RMIClientSocketFactory,
   *      java.rmi.server.RMIServerSocketFactory)
   */
  public Remote exportObject(Remote obj) throws RemoteException {
    if (port < 0) {
      throw new IllegalStateException("isBound() : false");
    }
    return UnicastRemoteObject.exportObject(obj, port, socketFactory,
      socketFactory);
  }

  /**
   * Remove o dado objeto da runtime do RMI.
   * 
   * @param obj Objeto a ser removido da runtime do RMI.
   * @param force Se <tt>true</tt> , prossegue mesmo que haja chamadas RMI
   *        pendentes ou em progresso.
   * @return <tt>True</tt> se foi possvel remover o objeto da runtime do RMI.
   * 
   * @throws NoSuchObjectException Se o objeto no estiver exportado.
   * 
   * @see UnicastRemoteObject#unexportObject(Remote, boolean)
   */
  public boolean unexportObject(Remote obj, boolean force)
    throws NoSuchObjectException {
    return UnicastRemoteObject.unexportObject(obj, force);
  }
}
