package csbase.util.messages;

import java.io.Serializable;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import tecgraf.javautils.core.filter.IFilter;
import csbase.util.data.channel.DataChannel;
import csbase.util.data.dispatcher.IDispatchListener;
import csbase.util.data.dispatcher.IDispatcher;

/**
 * Armazena as mensagens at que elas sejam entregues ou expiradas.
 * 
 * @author Tecgraf
 */
public class Topic implements IDispatchListener<IMessageListener, Message> {

  /**
   * Utilizado para "<i>logar</i>" informaes no sistema.
   */
  private static final Logger LOGGER = Logger.getLogger(Topic.class.getName());

  /**
   * Responsvel por entregar as mensagens aos ouvintes.
   */
  private DataChannel<IMessageListener, Message> channel;
  /**
   * Responsvel por armazenar as mensagens e quem as recebeu.
   */
  private MessageStore store;
  /**
   * Estratgia de entrega das mensagens para os ouvintes.
   */
  private IDispatcher<IMessageListener, Message> dispatcher;

  /**
   * Construtor.
   * 
   * @param store Responsvel por armazenar as mensagens e quem as recebeu.
   * @param dispatcher Estratgia utilizada para entregar as mensagens aos
   *        consumidores.
   */
  public Topic(MessageStore store,
    IDispatcher<IMessageListener, Message> dispatcher) {
    this.store = store;
    this.dispatcher = dispatcher;
    this.channel =
      new DataChannel<IMessageListener, Message>(this.dispatcher, this);
  }

  /**
   * Obtm as mensagens a serem entregues a um dado consumidor e as marca como
   * recebidas.
   * 
   * @param consumerId Identificador do consumidor.
   * @param filter filtro que determina as mensagens que sero retornadas.
   * 
   * @return as mensagens a serem entregues a um dado consumidor.
   */
  public Message[] receive(Serializable consumerId, IFilter<Message> filter) {
    return store.receive(consumerId, filter);
  }

  /**
   * Adiciona uma mensagem no tpico e notifica seus ouvintes.
   * 
   * @param message Mensagem a ser includa.
   * @param timeToLive Tempo, em milisegundos, que essa mensagem deve persistir
   *        at que seja consumida.
   */
  public void publish(Message message, long timeToLive) {
    store.publish(message, timeToLive);
    channel.publish(message);
  }

  /**
   * Inclui um ouvinte de mensagens nessa estrutura.
   * 
   * @param consumerId Identificador do consumidor.
   * @param listener Interface utilizada para entregar as mensagens de forma
   *        assncrona.
   * @param selector Filtro que determina as mensagens que sero repassadas ao
   *        ouvinte.
   * 
   * @return o nmero de ouvintes restantes.
   * 
   * @throws IllegalArgumentException Se j existir um ouvinte com o mesmo
   *         identificador.
   */
  public int subscribe(Serializable consumerId, IMessageListener listener,
    IFilter<Message> selector) {

    AsynchronousConsumer consumer =
      new AsynchronousConsumer(consumerId, listener, store);
    int size = channel.subscribe(consumer, selector);

    Message[] messages = store.peek(consumerId, selector);
    if (messages.length > 0) {
      dispatcher.dispatch(this, consumer, messages);
    }

    return size;
  }

  /**
   * <p>
   * Remove um determinado ouvinte desta estrutura.
   * </p>
   * <p>
   * Uma vez removido, o ouvinte no ir mais receber mensagens a menos que seja
   * inserido de novo atravs do mtodo
   * {@link #subscribe(Serializable, IMessageListener, IFilter)}.
   * </p>
   * 
   * @param consumerId Identificador do consumidor.
   * 
   * @return o nmero de ouvintes restantes.
   * 
   * @see #subscribe(Serializable, IMessageListener, IFilter)
   */
  public int unsubscribe(Serializable consumerId) {
    return channel
      .unsubscribe(new AsynchronousConsumer(consumerId, null, null));
  }

  /**
   * Obtm o componente responsvel por armazenar as mensagens do tpico.
   * 
   * @return o componente responsvel por armazenar as mensagens do tpico.
   */
  public MessageStore getMessageStore() {
    return store;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onExceptionThrown(Exception e, IMessageListener consumer,
    Message... messages) {
    AsynchronousConsumer asyncConsumer =
      AsynchronousConsumer.class.cast(consumer);

    LogRecord record =
      new LogRecord(Level.WARNING,
        "Error ao tentar entregar mensagens para o consumidor "
          + asyncConsumer.getId());
    record.setThrown(e);
    LOGGER.log(record);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void onDataDelivered(IMessageListener consumer, Message... messages) {
  }
}
