package csbase.util.proxy;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.reflect.InvocationHandler;

/**
 * Utilizado para informar qual cadeia de {@link InvocationHandler} deve ser
 * utilizada na criao de um proxy para a classe anotada.
 * 
 * @see ProxyUtils#createInvocationHandler(Object)
 * 
 * @author Tecgraf
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.TYPE)
public @interface InvocationHandlers {
  /**
   * Obtm a cadeia de {@link InvocationHandler} que deve ser utilizada na
   * criao do proxy.
   *
   * As implementaes de {@link InvocationHandler} apontadas devem ter um
   * construtor que receba como parmetro um {@link Object} e outro
   * {@link InvocationHandler}, pois a cadeia  montada da seguinte forma:
   * <ol>
   * <li>O ltimo elemento do array  instanciado recebendo o objeto para o qual
   * est se criando o proxy e um {@link Invoker} instanciado
   * recebendo o objeto como parmetro.</li>
   * <li>O elemento anterior a ele  instanciado recebendo o objeto e o ltimo
   * InvocationHandler criado.</li>
   * <li>Refaz a etapa anterior at no sobrar mais elemento no array.</li>
   * </ol>
   * Por exemplo, se uma classe C for anotada com {@code @InvocationHandlers(
   * HandlerA.class, HandlerB.class})}, o cdigo:
   * 
   * {@code 
   * Object c = new C();
   * InvocationHandler handler = ProxyUtils.createInvocationHandler(c);
   * }
   * 
   * equivale a
   * 
   * {@code 
   * Object c = new C();
   * InvocationHandler handler = new HandlerDelegator(c);
   * handler = new HandlerB(c, handler);
   * handler = new HandlerA(c, handler);
   * }
   *
   * @return A implementao de {@link InvocationHandler} que deve ser utilizada
   *         na criao do proxy.
   */
  Class<? extends InvocationHandler>[] value();
}
