package validations;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;

import tecgraf.javautils.core.io.FileUtils;
import validations.util.ValidatorUtils;
import csbase.logic.User;

/**
 * <p>
 * Percorrer os arquivos de persistncia do usurio substituindo a propriedade
 * {@value #OLD_EMAIL_ATTRIBUTE_KEY} que armazena o e-mail do usurio pela
 * propriedade {@link User#EMAILS} que tem como valor um array de e-mails do
 * usurio.<br>
 * Durante esse processo, caso o valor da propriedade
 * {@value #OLD_EMAIL_ATTRIBUTE_KEY} seja diferente de {@code null}, ele ser
 * includo no array de e-mails. <br>
 * Obs. Esse validador ir executar apenas uma vez.
 * </p>
 * <p>
 * Esse validador  uma consequncia da issue [CSBASE-2016][CSBASE] Permitir que
 * usurios possam cadastrar mais de um email no sistema.
 * </p>
 * 
 * @author Tecgraf
 */
public class UserMail2EmailsValidation extends AbstractValidation {

  /**
   * Chave da propriedade que diz o caminho do diretrio de persistncia dos
   * atributos dos usurios.
   */
  private static final String USERS_DIR_PROPERTY = "UsersData.directory";
  /**
   * Subdiretrio do backup onde ser armazenado o backup dos atributos dos
   * usurios.
   */
  private static final String BKP_USER_SUBDIR = File.separator + "persistency"
    + File.separator + "users";

  /** Antiga chave do atributo de e-mail do usurio. */
  private static final String OLD_EMAIL_ATTRIBUTE_KEY = "mail";
  /** Nova chave do atributo de e-mail do usurio. */
  private static final String NEW_EMAIL_ATTRIBUTE_KEY = User.EMAILS;
  /** Chave do atributo de login do usurio. */
  private static final String LOGIN_ATTRIBUTE_KEY = User.LOGIN;

  /** Extenso dos arquivos de atributos dos usurios. */
  private String fileExtension;
  /** Diretrio aonde residem os arquivos de atributos dos usurios. */
  private File usersDir;
  /** Caminho para o diretrio de backup dos arquivos de atributos dos usurios. */
  private String usersBkpPath;

  /** Todos os usurios do servidor. */
  private List<Hashtable<String, Object>> usersAttributes;

  /**
   * Se <tt>true</tt>, indica que a execuo do validador deve ser interrompida
   * sem dar erro e sem marcar o validador como executado. Nesse caso, uma
   * mensagem pode ter sido fornecida atravs do atributo 'warning' explicando o
   * porqu.
   */
  private boolean terminate = false;
  /**
   * Mensagem opcional que explica por que a execuo foi terminada (terminate
   * == true).
   */
  private String warning = null;

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean init() {
    // Obtm o iretrio onde residem os arquivos de atributos dos usurios.
    String usersDirPath = getMandatorySystemProperty(USERS_DIR_PROPERTY);
    if (usersDirPath == null) {
      terminate = true;
      warning =
        String
          .format(
            "Devido a ausncia da propriedade %s, considerou-se que esse  um novo servidor e devido a isso, o diretrio de persistncia dos usurios no existe.",
            USERS_DIR_PROPERTY);
      return true;
    }

    this.usersDir = new File(usersDirPath);
    if (!usersDir.exists()) {
      terminate = true;
      warning =
        String
          .format(
            "Devido a inexistncia do diretrio '%s' de persistncia dos usurios, considerou-se que esse  um novo servidor ou a propriedade %s est apontando para o diretrio errado.",
            usersDir.getAbsolutePath(), USERS_DIR_PROPERTY);
      return true;
    }

    // L os arquivos de atributos dos usurios.
    this.usersAttributes = new ArrayList<Hashtable<String, Object>>();
    this.usersBkpPath = getTempDirPath() + BKP_USER_SUBDIR;

    File[] usersFiles = usersDir.listFiles();
    if (usersFiles == null || usersFiles.length == 0) {
      final String path = usersDir.getAbsolutePath();
      logger.fine("Nada a fazer na persistncia em: " + path);
      return true;
    }

    final String firstUserFileName = usersFiles[0].getName();
    this.fileExtension = firstUserFileName.split("\\.")[1].trim();

    int inx = 0;
    try {
      while (inx < usersFiles.length) {
        File aUserFile = usersFiles[inx++];

        ObjectInputStream in = null;
        try {
          in =
            new ObjectInputStream(new DataInputStream(new BufferedInputStream(
              new FileInputStream(aUserFile))));

          // usando supresso de warning pois o objeto  serializado
          // com um cast direto para o hashtable com generics.
          @SuppressWarnings("unchecked")
          Hashtable<String, Object> attributes =
            (Hashtable<String, Object>) in.readObject();

          usersAttributes.add(attributes);
        }
        finally {
          if (in != null) {
            try {
              in.close();
            }
            catch (IOException e) {
              logger.exception("Erro fechando o arquivo de usurio '"
                + aUserFile.getPath() + "'.", e);
              return false;
            }
          }
        }
      }
      return true;
    }
    catch (FileNotFoundException e) {
      logger.exception("O arquivo de usurio '" + usersFiles[inx].getPath()
        + "' no foi encontrado.", e);
    }
    catch (IOException e) {
      logger.exception("Erro tentando ler o arquivo de usurio '"
        + usersFiles[inx].getPath() + "'.", e);
    }
    catch (ClassNotFoundException e) {
      logger.exception("Erro tentando ler o arquivo de usurio '"
        + usersFiles[inx].getPath() + "'.", e);
    }

    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getStartMessage() {
    return "Transformando o atributo MAIL:String em EMAILS:String[] nos usurios.";
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void getSpecificFailureMessage(Status status, List<String> errors) {
    switch (status) {
      case VALIDATION_FAILED:
        errors
          .add("Existem usurios com o atributo MAIL:String ou sem o atributo EMAILS:String[]");
        break;

      case PATCH_FAILED:
        errors
          .add("Ocorre um erro ao tentar corrigir alterar o atributo MAIL:String de um ou mais usurios.");
        break;

      case INIT_FAILED:
        errors.add("Falha na inicializao");
        break;

      default:
        errors.add("Estado invlido: " + status.toString());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getSuccessMessage(Status status) {
    switch (status) {
      case VALIDATION_OK:
        if (terminate) {
          if (warning != null) {
            return String.format(
              "A execuo do validador %s foi interrompida. %s", this
                .getClass().getSimpleName(), warning);
          }
          else {
            return String
              .format(
                "A execuo do validador %s foi interrompida por razo desconhecida.",
                this.getClass().getSimpleName());
          }
        }
        return "No h usurio com atributo MAIL:String e todos tem o atributo EMAILS:String[]";
      case PATCH_OK:
        return "O atributo MAIL:String foi transformado para EMAILS:String[] com sucesso em todos os usurios.";
      default:
        return "Estado invlido: " + status.toString();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean backupData() {
    File usersBkp = new File(usersBkpPath);
    if (!usersBkp.exists() && !usersBkp.mkdirs()) {
      final String msg = "Erro criando backup em: '" + usersBkpPath + "'";
      logger.severe(msg);
      return false;
    }

    File[] usersFiles = usersDir.listFiles();
    if (usersFiles == null || usersFiles.length == 0) {
      final String path = usersDir.getAbsolutePath();
      logger.fine("Nada a fazer backup em: " + path);
      return true;
    }

    for (File file : usersFiles) {
      File bkpFile = new File(usersBkp, file.getName());
      if (!ValidatorUtils.copyFile(file, bkpFile, logger, true)) {
        final String fmt = "Erro fazendo backup do arquivo '%s'";
        final String path = file.getAbsolutePath();
        final String err = String.format(fmt, path);
        logger.severe(err);
        return false;
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean restoreBackup() {
    File usersBkp = new File(usersBkpPath);
    if (!usersBkp.exists()) {
      final String fmt = "O diretrio de backup '%s' no foi encontrado.";
      final String err = String.format(fmt, usersBkpPath);
      logger.severe(err);
      return false;
    }

    File[] usersBkpFiles = usersBkp.listFiles();
    if (usersBkpFiles == null || usersBkpFiles.length == 0) {
      final String path = usersBkp.getAbsolutePath();
      final String msg = "Nada restaurar em: " + path + " (backup)";
      logger.fine(msg);
      FileUtils.delete(usersBkp);
      return true;
    }

    for (File bkpFile : usersBkpFiles) {
      File file = new File(usersDir, bkpFile.getName());
      if (!ValidatorUtils.copyFile(bkpFile, file, logger, true)) {
        logger.severe(String.format(
          "Erro restaurando arquivo '%s' para o diretrio '%s'",
          bkpFile.getAbsolutePath(), usersDir.getAbsolutePath()));
        return false;
      }
    }

    /*
     * Tendo o arquivo de propriedades do comando restaurado com sucesso,
     * removemos o diretrio de backup daquele comando.
     */
    FileUtils.delete(usersBkp);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean validate() {
    if (terminate) {
      return true;
    }

    for (Hashtable<String, Object> attributes : usersAttributes) {
      if (attributes.containsKey(OLD_EMAIL_ATTRIBUTE_KEY)) {
        return false;
      }
      if (!attributes.containsKey(NEW_EMAIL_ATTRIBUTE_KEY)) {
        return false;
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean applyPatch() {
    for (Hashtable<String, Object> attributes : usersAttributes) {
      if (attributes.containsKey(OLD_EMAIL_ATTRIBUTE_KEY)
        && attributes.containsKey(User.EMAILS)) {
        final String userLogin = attributes.get(User.LOGIN).toString();
        final String oldAttr = "MAIL:String " + OLD_EMAIL_ATTRIBUTE_KEY;
        final String newAttr = "EMAILS:String[] " + NEW_EMAIL_ATTRIBUTE_KEY;
        final String fmt = "Erro: O usurio %s tem atributos (%s) e (%s)!";
        final String err = String.format(fmt, userLogin, oldAttr, newAttr);
        logger.warning(err);
        return false;
      }
      if (attributes.containsKey(NEW_EMAIL_ATTRIBUTE_KEY)) {
        continue;
      }

      if (attributes.containsKey(OLD_EMAIL_ATTRIBUTE_KEY)) {
        String email = attributes.get(OLD_EMAIL_ATTRIBUTE_KEY).toString();
        attributes.put(NEW_EMAIL_ATTRIBUTE_KEY, new String[] { email });
        attributes.remove(OLD_EMAIL_ATTRIBUTE_KEY);
      }
      else {
        attributes.put(NEW_EMAIL_ATTRIBUTE_KEY, new String[0]);
      }

      // Salvando os atributos alterados.
      final String loginString = attributes.get(LOGIN_ATTRIBUTE_KEY).toString();
      File userFile = new File(usersDir, loginString + "." + fileExtension);
      ObjectOutputStream out = null;
      try {
        out =
          new ObjectOutputStream(new DataOutputStream(new BufferedOutputStream(
            new FileOutputStream(userFile))));
        out.writeObject(attributes);
        out.flush();
      }
      catch (FileNotFoundException e) {
        logger.exception("Erro, o arquivo '" + userFile.getPath()
          + "' no foi encontrado.", e);
        return false;
      }
      catch (IOException e) {
        logger.exception("Erro escrevendo no arquivo '" + userFile.getPath()
          + "'.", e);
        return false;
      }
      finally {
        if (out != null) {
          try {
            out.close();
          }
          catch (IOException e) {
            logger.exception("Erro fechando o arquivo '" + userFile.getPath()
              + "'.", e);
            return false;
          }
        }
      }
    }

    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void finish() {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean runsOnlyOnce() {
    return !terminate;
  }
}
