--------------------------------------------------------------------------------
-- API para criao de um novo mdulo SGA.
--------------------------------------------------------------------------------
--
-- Esse arquivo possui o "esqueleto" das funes que so necessrias
-- para implementar um novo SGA. A configurao do n (em sgad-cnf.lua),
-- dever usar o atributo loadlib para indicar o nome do arquivo que
-- contm o cdigo desenvolvido. Esse atributo faz com que o
-- sga-daemon carregue e execute as funes implementadas. Por exemplo,
-- supondo que o arquivo com a nova API possua o nome NovaAPI.lua, a
-- configurao do n seria da seguinte forma:
--
-- Node {
--   name = "nomeDaMaquina",
--   platform_id = "SGANovo",
--   loadlib = "NovaAPI",
-- }
--
-- A escolha de configurar o SGA como Node ou como Grid vai depender de
-- como voc deseja que este SGA aparea na interface do usurio. Se
-- o usurio deve ver o SGA Boinc como uma nica mquina, voc deve
-- configur-lo como Node. Porm, se o Boinc possui vrios ns e voc
-- deseja que o usurio possa monitor-los, voc deve configur-lo
-- como Grid.
--
-- Grid {
--   name = "nomeDaMaquina",
--   loadlib = "NovaAPI",
--   default = {
--     platform_id = "SGANovo",
--   },
-- }
--
-- Voc encontra mais detalhes em:
-- https://jira.tecgraf.puc-rio.br/confluence/pages/viewpage.action?pageId=37814618
--------------------------------------------------------------------------------

-- @TODO Rever nomenclatura do parametro server na funes

--------------------------------------------------------------------------------
--
-- Especificao das funes pblicas
--
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Incializa o mdulo para execuo. Essa funo deve ser chamada antes
-- de qualquer outra funo desta biblioteca.
--
-- Retorna verdadeiro caso o mdulo seja inicializado com sucesso. Em caso
-- de erro, retorna nil e a mensagem de erro.
--
-- ok, err = open()
--------------------------------------------------------------------------------
open = function()
  return true, nil
end -- function open

--------------------------------------------------------------------------------
-- Termina a execuo do mdulo. Os processos disparados que ainda estejam
-- rodando no so afetados por essa chamada. Essa funo no retorna valores.
--
-- close()
--------------------------------------------------------------------------------
close = function()
end -- function close

--------------------------------------------------------------------------------
--
-- Funes de consulta  configurao de servidor.
--
-- Em SGAs que so clusters, o parmetro server indica o n sobre o
-- qual se deseja obter a informao.
--
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Retorna as informaes de configurao correspondentes s chaves e ao
-- servidor especificados.
--
-- @param server o servidor do qual as informaes sero obtidas. Caso o valor
--        seja nil ser assumido localhost.
-- @param keys a lista contendo as chaves das informaes que sero obtidas. Se
--        a lista for nil ou vazia sero retornadas as informaes padres.
--
-- @return o mapa contendo as chaves com o valor correspondente da informao e
--         uma mensagem em caso de erro.
--         As informaes padres que devem ser suportadas so:
--           csbase_num_processors
--           csbase_memory_ram_info_mb
--           csbase_memory_swap_info_mb
--           csbase_job_control_actions
--          @TODO referenciar a IDL
--         Alm destas, podem-se adicionar informaes padres especficas de
--         plataforma.
--
-- map, err = getConfiguration(server, keys)
--------------------------------------------------------------------------------
getConfiguration = function(server, keys)
  -- @TODO verificar como acessar, via Oil, as constantes definidas na interface IDL
  -- const string SGA_NODE_NUM_PROCESSORS = "csbase_num_processors";
  -- const string SGA_NODE_MEMORY_RAM_INFO_MB = "csbase_memory_ram_info_mb";
  -- const string SGA_NODE_MEMORY_SWAP_INFO_MB = "csbase_memory_swap_info_mb";
  -- @TODO Verificar se essas duas informaes so obtidas neste nvel ou no daemon
  -- const string SGA_NODE_CLOCK_SPEED_MHZ = "clock_speed_mhz";
  -- const string SGA_NODE_PLATFORM_ID = "csbase_platform_id";
  if not keys or #keys == 0 then
    return {
      csbase_num_processors = 1,
      csbase_memory_ram_info_mb = 0,
      csbase_memory_swap_info_mb = 0,
      csbase_job_control_actions = {},
    }, nil
  end -- if no keys

  local configMap = {}
  for _, k in ipairs(keys) do
    if k == "csbase_num_processors" then
      configMap[k] = 1
    elseif k == "csbase_memory_ram_info_mb" then
      configMap[k] = 0
    elseif k == "csbase_memory_swap_info_mb" then
      configMap[k] = 0
    elseif k == "csbase_job_control_actions" then
      configMap[k] = {}
    end
  end
  return configMap, nil
end -- function getConfiguration

--------------------------------------------------------------------------------
--
-- Funes de monitorao de servidor.
--
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Retorna as informaes dinmicas correspondentes s chaves e ao servidor
-- especificados.
--
-- @param server o servidor do qual as informaes sero obtidas. Caso o valor
--        seja nil ser assumido localhost.
-- @param keys a lista contendo as chaves das informaes que sero obtidas. Se
--        a lista for nil ou vazia sero retornadas as informaes padres.
-- @param maxold indica o tempo mximo que a informao pode ter. Caso maxold
--        seja nil,  assumido zero e a informao ser buscada.
--
-- @return o mapa contendo as chaves com o valor correspondente da informao e
--         uma mensagem em caso de erro.
--         As informaes padres so:
--           csbase_load_perc
--           csbase_memory_ram_free
--           csbase_memory_swap_free
--           csbase_number_of_jobs
--         Alm destas, podem-se adicionar informaes padres especficas de
--         plataforma.
--
-- Algunes detalhes sobre as informaes padres:
-- csbase_load_perc:
--   a taxa mdia de ocupao de CPU do ltimo minuto. Esse valor considera o
--   nmero de processadores que o servidor possui. Por exemplo, caso a mtrica
--   de ocupao seja o nmero de processos na fila de prontos, este nmero
--   estar dividido pela quantidade de processadores.
-- csbase_memory_ram_free_perc e csbase_memory_swap_free_perc:
--   a mdia de ocupao, em bytes, de memria do ltimo minuto.
-- csbase_number_of_jobs:
--   o nmero de jobs que esto em execuo.
--
-- map, err = getInfo(server, keys, maxold)
--------------------------------------------------------------------------------
getInfo = function(server, keys, maxold)
  if not keys or #keys == 0 then
    return {
      csbase_load_perc = 0,
      csbase_memory_ram_free = 0,
      csbase_memory_swap_free = 0,
      csbase_number_of_jobs = 0,
    }, nil
  end -- if no keys

  local infoMap = {}
  for _, k in ipairs(keys) do
    if k == "csbase_load_perc" then
      infoMap[k] = 0
    elseif k == "csbase_memory_ram_free" then
      infoMap[k] = 0
    elseif k == "csbase_memory_swap_free" then
      infoMap[k] = 0
    elseif k == "csbase_number_of_jobs" then
      infoMap[k] = 0
    end
  end
  return infoMap, nil
end -- function getInfo

--------------------------------------------------------------------------------
--
-- Funes de execuo, monitorao e controle de processos.
--
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Executa um comando no servidor especificado e retorna um handle que  a
-- referncia para o comando no contexto desta biblioteca. O handle pode ter
-- qualquer formato.
-- Os parmetros extras para a execuo podem ser especficos da plataforma ou
-- um dos seguintes j definidos:
--   csbase_command_input_path caminho do direcionamento da entrada padro.
--   csbase_command_output_path caminho do direcionamento da sada padro.
--   csbase_command_error_path caminho do direcionamento da sada de erro.
-- Os valores de csbase_command_output_path e csbase_command_error_path podem
-- apontar para o mesmo caminho.
--
-- @param id o identificador nico para este comando.
-- @param command o comando a ser executado.
-- @param extraParams os parmetros extras para a excuo do comando.
-- @param server o servidor onde o comando ser executado. Caso seja nil ser
--        assumido localhost.
--
-- @return o handle do comando e, em caso de erro, nil e a mensagem de erro.
--
-- handle, err = executeCommand(id, command, extraParams, server)
--------------------------------------------------------------------------------
executeCommand = function(id, command, extraParams, server)
  -- const string COMMAND_EXECUTION_INPUT_PATH = "csbase_command_input_path";
  -- const string COMMAND_EXECUTION_OUTPUT_PATH = "csbase_command_output_path";
  -- const string COMMAND_EXECUTION_ERROR_PATH = "csbase_command_error_path";
  local handle = {
    id = id,
    command = command
  }
  return handle, nil
end -- function executeCommand

--------------------------------------------------------------------------------
-- Excerce uma ao sobre um comando iniciado atravs da funo executecommand.
--
-- @param handle o handle do comando.
-- @param action a ao que ser feito sobre o comando. Os valores esto
--        definidos na enumerao JobControlAction:
--          SUSPEND: suspender a execuo do comando;
--          RESUME: (re)iniciar o comando;
--          HOLD: alterar o estado do comando para on-hold;
--          RELEASE: retirar o comando do estado de on-hold;
--          TERMINATE: terminar o comando.
-- @param childId identificador do job filho no qual ser feita a ao, ou nulo
-- se a ao deve ser feita no comando.
--
-- @return true se a ao foi exercida com sucesso e, se cocorrer algum erro,
--         false ou nil e a mensagem de erro.
-- @TODO pensar em como lanar as excees ActionNotSupportedException, InvalidActionException e InvalidTransitionException
--
-- ok, ex = control(handle, action, childId)
--------------------------------------------------------------------------------
control = function(handle, action, childId)
  return true, nil
end -- function control

--------------------------------------------------------------------------------
-- Retorna uma tabela Lua com os dados do comando que precisam ser persistidos.
-- Estes dados devem ser suficientes para que um comando seja 'recuperado' aps
-- a reinicializao do SGA, atravs da chamada  retrievecommandhandle.
--
-- @param handle o handle do comando.
--
-- @return os dados do comando ou nil e mensagem de erro na ocorrncia de erro.
--
-- pdata, err = getcommandpersistentdata(handle)
--------------------------------------------------------------------------------
getCommandPersistentData = function(handle)
  return handle, nil
end -- function getCommandPersistentData

--------------------------------------------------------------------------------
-- A partir dos dados persistidos, retorna um handle que  a referncia para o
-- comando no contexto desta biblioteca. O handle pode ter qualquer formato.
--
-- @param pdata os dados do comando.
--
-- @return o handle do comando ou nil e mensagem de erro na ocorrncia de erro.
--
-- handle, err = retrievecommandhandle(pdata)
--------------------------------------------------------------------------------
retrieveCommandHandle = function(pdata)
  return pdata, nil
end -- function retrieveCommandHandle

--------------------------------------------------------------------------------
-- Libera todos os recursos relacionados ao processo especificado. Esta funo
-- precisa ser chamada aps o trmino do processo para que eventuais recursos
-- alocados a ele possam ser liberados. Aps a chamada desta funo, o handle
-- do processo no poder mais ser utilizado.

-- @return true se os recursos foram leberados com sucesso e, se cocorrer algum
--         erro, false ou nil e a mensagem de erro.
--
-- ok, err = releaseCommandResources(handle)
--------------------------------------------------------------------------------
releaseCommandResources = function(handle)
   return true, nil
end -- function releaseCommandResources

--------------------------------------------------------------------------------
-- Retorna as informaes correspondentes s chaves e ao comando especificados.
--
-- @param handle o handle do comando.
-- @param keys a lista contendo as chaves das informaes que sero obtidas. Se
--        a lista for nil ou vazia sero retornadas as informaes padres.
-- @param maxold indica o tempo mximo que a informao pode ter. Caso maxold
--        seja nil,  assumido zero e a informao ser buscada.
--
-- @return o mapa contendo as chaves com o valor correspondente da informao e
--         uma mensagem em caso de erro.
--         As informaes padres so:
--           csbase_command_pid
--           csbase_command_string
--           csbase_command_exec_host
--           csbase_command_state
--           csbase_command_memory_ram_size_mb
--           csbase_command_memory_swap_size_mb
--           csbase_command_cpu_perc
--           csbase_command_time_sec
--           csbase_command_wall_time_sec
--           csbase_command_user_time_sec
--           csbase_command_system_time_sec
--         Alm destas, podem-se adicionar informaes padres especficas de
--         plataforma.
--
-- Algunes detalhes sobre as informaes padres:
-- csbase_command_id:
--   identificador do comando recebido na funo executecommand.
-- csbase_command_pid:
--   identificador do processo iniciado atravs da funo executecommand.
-- csbase_command_ppid:
--   identificador do processo pai iniciado atravs da funo executecommand.
-- csbase_command_string:
--   o comando em execuo.
-- csbase_command_exec_host:
--   a mquina que est executando o comando.
-- csbase_command_state:
--   o estado de um processo iniciado atravs da funo executecommand.
-- csbase_command_memory_ram_size_mb:
--   a mdia de ocupao de memria do ltimo minuto (em MB).
-- csbase_command_memory_swap_size_mb:
--   a mdia de ocupao da rea de swap do ltimo minuto (em MB).
-- csbase_command_cpu_perc:
--   a taxa mdia de ocupao de CPU do ltimo minuto pelo comando.
-- csbase_command_cpu_time_sec:
--   o tempo de CPU da execuo do comando (em SEC).
-- csbase_command_wall_time_sec:
--   o tempo de relgio da execuo do comando (em SEC).
-- csbase_command_user_time_sec:
--   o tempo de usurio da execuo do comando (em SEC).
-- csbase_command_system_time_sec:
--   o tempo de systema da execuo do comando (em SEC).
-- csbase_command_virtual_memory_size_mb:
--   quantidade de memria virtual utilizado na execuo do comando (em MB).
-- csbase_command_bytes_in_kb:
--   quantidade de dados lidos na execuo do comando (em KB).
-- csbase_command_bytes_out_kb:
--   quantidade de dados escritos na execuo do comando (em KB).
-- csbase_command_disk_bytes_read_kb:
--   quantidade de dados lidos do disco na execuo do comando (em KB).
-- csbase_command_disk_bytes_write_kb:
--   quantidade de dados escritos no disco na execuo do comando (em KB).
--
-- map, err = getCommandInfo(handle, keys, maxold)
--------------------------------------------------------------------------------
getCommandInfo = function(handle, keys, maxold)
  if not keys or #keys == 0 then
    return {
      csbase_command_id = 0,
      csbase_command_pid = 0,
      csbase_command_ppid = 0,
      csbase_command_string = handle.command,
      csbase_command_exec_host = "unknown",
      csbase_command_state = servermanager.FINISHED,
      csbase_command_memory_ram_size_mb = 0,
      csbase_command_memory_swap_size_mb = 0,
      csbase_command_cpu_perc = 0,
      csbase_command_cpu_time_sec = 0,
      csbase_command_wall_time_sec = 0,
      csbase_command_user_time_sec = 0,
      csbase_command_system_time_sec = 0,
    }, nil
  end -- if no keys

  local cmdInfoMap = {}
  for _, k in ipairs(keys) do
    if k == "csbase_command_id" then
      cmdInfoMap[k] = 0
    elseif k == "csbase_command_pid" then
      cmdInfoMap[k] = 0
    elseif k == "csbase_command_ppid" then
      cmdInfoMap[k] = 0
    elseif k == "csbase_command_string" then
      cmdInfoMap[k] = handle.command
    elseif k == "csbase_command_exec_host" then
      cmdInfoMap[k] = "unknown"
    elseif k == "csbase_command_state" then
      cmdInfoMap[k] = servermanager.FINISHED
    elseif k == "csbase_command_memory_ram_size_mb" then
      cmdInfoMap[k] = 0
    elseif k == "csbase_command_memory_swap_size_mb" then
      cmdInfoMap[k] = 0
    elseif k == "csbase_command_cpu_perc" then
      cmdInfoMap[k] = 0
    elseif k == "csbase_command_cpu_time_sec" then
      cmdInfoMap[k] = 0
    elseif k == "csbase_command_wall_time_sec" then
      cmdInfoMap[k] = 0
    elseif k == "csbase_command_user_time_sec" then
      cmdInfoMap[k] = 0
    elseif k == "csbase_command_system_time_sec" then
      cmdInfoMap[k] = 0
    end
  end
  return cmdInfoMap, nil
end


--------------------------------------------------------------------------------
-- Define as chaves padres de informaes dinmicas que sero usadas na funo
-- getConfiguration, getInfo e getCommandInfo.
-- Caso o usurio no tenha defindo o conjunto de chaves padres a biblioteca
-- deve ter um conjunto mnimo, que dever conter as chaves:
--   csbase_load_perc
--   csbase_memory_ram_free_perc
--   csbase_memory_swap_free_perc
--   csbase_number_of_jobs
--
--   csbase_num_processors
--   csbase_memory_ram_info_mb
--   csbase_memory_swap_info_mb
--   csbase_job_control_actions
--   csbase_command_pid
--   csbase_command_string
--   csbase_command_exec_host
--   csbase_command_state
--   csbase_command_memory_ram_size_mb
--   csbase_command_memory_swap_size_mb
--   csbase_command_cpu_perc
--   csbase_command_time_sec
--   csbase_command_wall_time_sec
--   csbase_command_user_time_sec
--   csbase_command_system_time_sec
-- @param keys o conjunto de chaves de configurao
--------------------------------------------------------------------------------
setDefaultKeys = function(keys)
end -- function setDefaultKeys

--------------------------------------------------------------------------------
-- Retirna todas as chaves disponveis.
--------------------------------------------------------------------------------
getAllKeys = function()
end -- function getAllKeys

--------------------------------------------------------------------------------
--
-- Funes de consulta  clusters.
--
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Obtm as informaes de todos os processos em execuo no SGA.
-- ** Em desenvolvimento **
--------------------------------------------------------------------------------
getJobsInfo = function()
  return nil
end

--------------------------------------------------------------------------------
-- Retorna uma lista com os ns do servidor.
--
-- @return a lista de ns e, em caso de erro, nil e a mensagem de erro.
--
-- nodes, err = getnodes(server)
--------------------------------------------------------------------------------
getNodes = function()
  return { "localhost" }, nil
end

--------------------------------------------------------------------------------
--
-- Funes de log de histrico
--
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Indica se o histrico deve ser ativado ou no para o SGA em questo.
--
-- setHistoric(historic)
--------------------------------------------------------------------------------
setHistoric = function(historic)
  servermanager._enableHistoric = historic
end

--------------------------------------------------------------------------------
-- Retorna true se o histrico deve ser ativado para o sga em questo,
-- ou false caso contrrio.
--
-- enableHistoric = getHistoric()
--------------------------------------------------------------------------------
getHistoric = function()
  return servermanager._enableHistoric
end

--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
servermanager = {
  -------------
  -- private --
  -------------
  _enableHistoric = false,

  -----------------------
  -- Constantes do mdulo
  -----------------------
  RUNNING = 0,
  NOT_RESPONDING = 1,
  WAITING = 2,
  FINISHED = 3,

  ERROR_CODE = -1,

  -------------
  -- public --
  -------------
  now = now, -- Definida em C nos binrios para Unix.
  sleep = sleep, -- Definida em C nos binrios para Unix.

  -- Funes de controle do mdulo:
  open = open,
  close = close,

  -- Funes de consulta  configurao de servidor:
  getConfiguration = getConfiguration,

  -- Funes de monitorao de servidor:
  getInfo = getInfo,

  -- Funes de execuo, monitorao e controle de processos:
  executeCommand = executeCommand,
  control = control,
  getCommandPersistentData = getCommandPersistentData,
  retrieveCommandHandle = retrieveCommandHandle,
  releaseCommandResources = releaseCommandResources,
  getCommandInfo = getCommandInfo,

  -- Funes para clusters:
  getJobsInfo = getJobsInfo,
  getNodes = getNodes, -- OPCIONAL

  -- Funes de log de histrico:
  setHistoric = setHistoric, -- OPCIONAL
  getHistoric = getHistoric, -- OPCIONAL
}
