/*
 * Detalhes da ltima alterao:
 * 
 * $Author$ $Date$ $Revision$
 */
package tecgraf.ftc_1_3.client;

import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;

import tecgraf.ftc_1_3.common.exception.FailureException;
import tecgraf.ftc_1_3.common.exception.FileLockedException;
import tecgraf.ftc_1_3.common.exception.InvalidProtocolVersionException;
import tecgraf.ftc_1_3.common.exception.MaxClientsReachedException;
import tecgraf.ftc_1_3.common.exception.PermissionException;

/**
 * Representa um canal de acesso a um arquivo remoto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public interface RemoteFileChannel extends Serializable {
  /**
   * Solicita a abertura do arquivo.
   * 
   * @param readOnly Indica se o arquivo deve ser aberto somente para leitura ou
   *        no.
   * 
   * @throws PermissionException Caso o usurio no tenha permisso para
   *         realizar o modo de abertura solicitado.
   * @throws FileNotFoundException Caso o arquivo no exista.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   * @throws MaxClientsReachedException Caso o servidor no possa mais receber
   *         clientes.
   * @throws InvalidProtocolVersionException Caso o servidor no suporte a
   *         verso do protocolo utilizada pelo cliente.
   */
  void open(boolean readOnly) throws PermissionException,
    FileNotFoundException, FailureException, MaxClientsReachedException,
    InvalidProtocolVersionException;

  /**
   * Verifica se o arquivo est aberto.
   * 
   * @return {@code true} caso o arquivo esteja aberto, ou {@code false}, caso
   *         contrrio.
   */
  boolean isOpen();

  /**
   * Solicita o fechamento do arquivo.
   * 
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   */
  void close() throws FailureException;

  /**
   * Obtm a posio atual do arquivo.
   * 
   * @return A posio atual do arquivo ou {@literal -1}, caso ocorra algum
   *         erro.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   */
  long getPosition() throws FailureException;

  /**
   * Define a nova posio do arquivo.
   * 
   * @param position A nova posio.
   * 
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   */
  void setPosition(long position) throws FailureException;

  /**
   * Obtm o tamanho do arquivo.
   * 
   * @return O tamanho do arquivo ou {@literal -1}, caso ocorra algum erro.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   */
  long getSize() throws FailureException;

  /**
   * Altera o tamanho do arquivo para um novo tamanho.
   * 
   * @param size O novo tamanho.
   * 
   * @throws PermissionException Caso o arquivo tenha sido aberto somente para
   *         leitura.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   */
  void setSize(long size) throws PermissionException, FailureException;

  /**
   * L uma sequncia de bytes do arquivo.
   * 
   * @param target O buffer.
   * 
   * @return A quantidade de bytes lidos.
   * 
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   */
  int read(byte[] target) throws FailureException;

  /**
   * L uma sequncia de bytes do arquivo.
   * 
   * @param target O buffer.
   * @param position A posio do arquivo a partir da qual a leitura vai ser
   *        iniciada.
   * 
   * @return A quantidade de bytes lidos.
   * 
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   */
  int read(byte[] target, long position) throws FailureException;

  /**
   * L uma sequncia de bytes do arquivo.
   * 
   * @param target O buffer.
   * @param offset O deslocamento no buffer.
   * @param length A quantidade mxima de bytes a serem lidos.
   * 
   * @return A quantidade de bytes lidos.
   * 
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   */
  int read(byte[] target, int offset, int length) throws FailureException;

  /**
   * L uma sequncia de bytes do arquivo.
   * 
   * @param target O buffer.
   * @param offset O deslocamento no buffer.
   * @param length A quantidade mxima de bytes a serem lidos.
   * @param position A posio do arquivo a partir da qual a leitura vai ser
   *        iniciada.
   * 
   * @return A quantidade de bytes lidos.
   * 
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   */
  int read(byte[] target, int offset, int length, long position)
    throws FailureException;

  /**
   * Escreve uma sequncia de bytes no arquivo.
   * 
   * @param source O buffer.
   * 
   * @return A quantidade de bytes escritos.
   * 
   * @throws PermissionException Caso o arquivo tenha sido aberto somente para
   *         leitura.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   * @throws FileLockedException Indica que o arquivo est reservado para outro
   *         usurio.
   */
  int write(byte[] source) throws PermissionException, FailureException,
    FileLockedException;

  /**
   * Escreve uma sequncia de bytes no arquivo.
   * 
   * @param source O buffer.
   * @param position A posio do arquivo a partir da qual a escrita vai ser
   *        iniciada.
   * 
   * @return A quantidade de bytes escritos.
   * 
   * @throws PermissionException Caso o arquivo tenha sido aberto somente para
   *         leitura.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   * @throws FileLockedException Indica que o arquivo est reservado para outro
   *         usurio.
   */
  int write(byte[] source, long position) throws PermissionException,
    FailureException, FileLockedException;

  /**
   * Escreve uma sequncia de bytes no arquivo.
   * 
   * @param source O buffer.
   * @param offset O deslocamento no buffer.
   * @param length A quantidade mxima de bytes a serem escritos.
   * 
   * @return A quantidade de bytes escritos.
   * 
   * @throws PermissionException Caso o arquivo tenha sido aberto somente para
   *         leitura.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   * @throws FileLockedException Indica que o arquivo est reservado para outro
   *         usurio.
   */
  int write(byte[] source, int offset, int length) throws PermissionException,
    FailureException, FileLockedException;

  /**
   * Escreve uma sequncia de bytes no arquivo.
   * 
   * @param source O buffer.
   * @param offset O deslocamento no buffer.
   * @param length A quantidade mxima de bytes a serem escritos.
   * @param position A posio do arquivo a partir da qual a escrita vai ser
   *        iniciada.
   * 
   * @return A quantidade de bytes escritos.
   * 
   * @throws PermissionException Caso o arquivo tenha sido aberto somente para
   *         leitura.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   * @throws FileLockedException Indica que o arquivo est reservado para outro
   *         usurio.
   */
  int write(byte[] source, int offset, int length, long position)
    throws PermissionException, FailureException, FileLockedException;

  /**
   * Transfere os dados do arquivo para um fluxo de sada.
   * 
   * @param position A posio inicial a partir daqual o arquivo ser lido.
   * @param count A quantidade de bytes que sero transferidos.
   * @param target O fluxo de sada que receber os dados do arquivo.
   * 
   * @return A quantidade de bytes efetivamente transferidos.
   * 
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   */
  long transferTo(long position, long count, OutputStream target)
    throws FailureException;

  /**
   * Transfere os dados do arquivo para um outro arquivo remoto.
   * 
   * @param position A posio inicial a partir daqual o arquivo ser lido.
   * @param count A quantidade de bytes que sero transferidos.
   * @param target O canal do arquivo que receber os dados.
   * 
   * @return A quantidade de bytes efetivamente transferidos.
   * 
   * @throws PermissionException Caso o outro arquivo remoto tenha sido aberto
   *         somente para leitura.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   * @throws FileLockedException Caso o outro arquivo remoto esteja reservado
   *         para outro usurio.
   */
  long transferTo(long position, long count, RemoteFileChannel target)
    throws PermissionException, FailureException, FileLockedException;

  /**
   * Transfere os dados de um fluxo de entrada para o arquivo.
   * 
   * @param source O fluxo de entrada.
   * @param position A posio inicial a partir daqual o arquivo ser escrito.
   * @param count A quantidade de bytes que sero transferidos.
   * 
   * @return A quantidade de bytes efetivamente transferidos.
   * 
   * @throws PermissionException Caso o arquivo tenha sido aberto somente para
   *         leitura.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   * @throws FileLockedException Indica que o arquivo est reservado para outro
   *         usurio.
   */
  long transferFrom(InputStream source, long position, long count)
    throws PermissionException, FailureException, FileLockedException;

  /**
   * Transfere os de um outro arquivo remoto para o arquivo.
   * 
   * @param source O canal do arquivo de entrada.
   * @param position A posio inicial a partir daqual o arquivo ser escrito.
   * @param count A quantidade de bytes que sero transferidos.
   * 
   * @return A quantidade de bytes efetivamente transferidos.
   * 
   * @throws PermissionException Caso o arquivo tenha sido aberto somente para
   *         leitura.
   * @throws FailureException Caso ocorra alguma falha no procedimento.
   * @throws FileLockedException Indica que o arquivo est reservado para outro
   *         usurio.
   */
  long transferFrom(RemoteFileChannel source, long position, long count)
    throws PermissionException, FailureException, FileLockedException;

  /**
   * Operao que faz um poll no servidor com o intuito de manter a conexao
   * ativa
   * 
   * @throws FailureException Caso ocorra alguma falha na operao.
   */
  void keepAlive() throws FailureException;

}
