/*
 * $Id: AccessKey.java 88412 2009-02-13 22:13:31Z vfusco $
 */
package tecgraf.ftc_1_3.server;

import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;

import tecgraf.ftc_1_3.common.exception.InvalidArraySize;

/**
 * Representa uma chave de acesso a um canal de dados.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class AccessKey {
  /**
   * O nome do algoritmo de gerao de nmeros pseudo-aleatrios.
   */
  private static final String PRNG_ALGORITHM = "SHA1PRNG";
  /**
   * O tamanho da chave de acesso.
   */
  public static final int KEY_SIZE = 16;
  /**
   * A chave propriamente dita.
   */
  private byte[] key;

  /**
   * Cria uma chave de acesso a um canal de dados.
   */
  public AccessKey() {
    SecureRandom secureRandom = null;
    try {
      secureRandom = SecureRandom.getInstance(PRNG_ALGORITHM);
    }
    catch (NoSuchAlgorithmException e) {
      secureRandom = new SecureRandom();
    }
    this.key = new byte[KEY_SIZE];
    secureRandom.nextBytes(this.key);
  }

  /**
   * Cria uma chave de acesso a um canal de dados a partir de um conjunto de
   * bytes.
   * 
   * @param bytes O conjunto de bytes, que deve possuir tamanho
   *        {@value #KEY_SIZE}.
   * @throws InvalidArraySize
   */
  public AccessKey(byte[] bytes) throws InvalidArraySize {
    if (bytes.length < KEY_SIZE)
      throw new InvalidArraySize();
    this.key = new byte[KEY_SIZE];
    System.arraycopy(bytes, 0, this.key, 0, KEY_SIZE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!this.getClass().equals(obj.getClass())) {
      return false;
    }
    AccessKey accessKey = (AccessKey) obj;
    return Arrays.equals(this.key, accessKey.key);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return Arrays.hashCode(this.key);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return Arrays.toString(this.key);
  }

  /**
   * Obtm os bytes que compem a chave de acesso.
   * 
   * @return Os bytes que compem a chave de acesso.
   */
  byte[] getBytes() {
    byte[] bytes = new byte[KEY_SIZE];
    System.arraycopy(this.key, 0, bytes, 0, KEY_SIZE);
    return bytes;
  }
}
